/*
 * Copyright 2017-2018 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.springframework.cloud.gcp.autoconfigure.spanner;

import com.google.cloud.spanner.DatabaseClient;

import org.springframework.beans.factory.ObjectProvider;
import org.springframework.boot.autoconfigure.AutoConfigureBefore;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnSingleCandidate;
import org.springframework.boot.autoconfigure.transaction.TransactionAutoConfiguration;
import org.springframework.boot.autoconfigure.transaction.TransactionManagerCustomizers;
import org.springframework.cloud.gcp.data.spanner.core.SpannerTransactionManager;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.transaction.PlatformTransactionManager;

/**
 * Auto-configuration for {@link SpannerTransactionManager}.
 *
 * @author Chengyuan Zhao
 *
 * @since 1.1
 */
@Configuration
@ConditionalOnClass(SpannerTransactionManager.class)
@AutoConfigureBefore(TransactionAutoConfiguration.class)
public class SpannerTransactionManagerAutoConfiguration {

	/**
	 * Config settings.
	 */
	@Configuration
	@ConditionalOnSingleCandidate(DatabaseClient.class)
	static class DatabaseClientTransactionManagerConfiguration {

		private final DatabaseClient databaseClient;

		private final TransactionManagerCustomizers transactionManagerCustomizers;

		DatabaseClientTransactionManagerConfiguration(DatabaseClient databaseClient,
				ObjectProvider<TransactionManagerCustomizers> transactionManagerCustomizers) {
			this.databaseClient = databaseClient;
			this.transactionManagerCustomizers = transactionManagerCustomizers
					.getIfAvailable();
		}

		@Bean
		@ConditionalOnMissingBean(PlatformTransactionManager.class)
		public SpannerTransactionManager spannerTransactionManager() {
			SpannerTransactionManager transactionManager = new SpannerTransactionManager(
					this.databaseClient);
			if (this.transactionManagerCustomizers != null) {
				this.transactionManagerCustomizers.customize(transactionManager);
			}
			return transactionManager;
		}
	}
}
