/*
 * Copyright 2002-2024 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.springframework.cloud.servicebroker.autoconfigure.web.servlet;

import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;

import org.springframework.cloud.servicebroker.exception.ServiceBrokerApiVersionException;
import org.springframework.cloud.servicebroker.exception.ServiceBrokerApiVersionMissingException;
import org.springframework.cloud.servicebroker.model.BrokerApiVersion;
import org.springframework.web.servlet.HandlerInterceptor;

/**
 * {@link HandlerInterceptor} that inspects the service broker API version passed in all
 * request headers and compares it to the API version supported by the broker.
 *
 * @author Scott Frederick
 */
public class ApiVersionInterceptor implements HandlerInterceptor {

	private final BrokerApiVersion version;

	/**
	 * Construct an interceptor that disables API version validation.
	 */
	public ApiVersionInterceptor() {
		this(null);
	}

	/**
	 * Construct an interceptor that validates the API version passed in request headers
	 * to the configured version.
	 * @param version the API version supported by the broker.
	 */
	public ApiVersionInterceptor(BrokerApiVersion version) {
		super();
		this.version = version;
	}

	/**
	 * Compares the service broker API version header to the supported version.
	 * @param request {@inheritDoc}
	 * @param response {@inheritDoc}
	 * @param handler {@inheritDoc}
	 * @throws ServiceBrokerApiVersionException if the API version header value does not
	 * match the version supported by the broker
	 */
	@Override
	public boolean preHandle(HttpServletRequest request, HttpServletResponse response, Object handler) {
		if (this.version != null && !anyVersionAllowed()) {
			String apiVersion = request.getHeader(this.version.getBrokerApiVersionHeader());

			if (apiVersion == null) {
				throw new ServiceBrokerApiVersionMissingException(this.version.getApiVersion());
			}

			if (!this.version.getApiVersion().equals(apiVersion)) {
				throw new ServiceBrokerApiVersionException(this.version.getApiVersion(), apiVersion);
			}
		}
		return true;
	}

	private boolean anyVersionAllowed() {
		return BrokerApiVersion.API_VERSION_ANY.equals(this.version.getApiVersion());
	}

}
