/*
 * Decompiled with CFR 0.152.
 */
package org.springframework.data.neo4j.repository.query;

import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.OffsetTime;
import java.time.ZonedDateTime;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.EnumSet;
import java.util.Set;
import java.util.TreeSet;
import java.util.stream.Collectors;
import org.neo4j.driver.types.Point;
import org.springframework.data.mapping.PersistentPropertyPath;
import org.springframework.data.neo4j.core.mapping.Neo4jMappingContext;
import org.springframework.data.neo4j.core.mapping.Neo4jPersistentProperty;
import org.springframework.data.neo4j.repository.query.Neo4jQueryMethod;
import org.springframework.data.repository.query.parser.Part;
import org.springframework.data.util.ClassTypeInformation;
import org.springframework.util.Assert;

class PartValidator {
    private static final Set<Class<?>> COMPARABLE_TEMPORAL_TYPES;
    private static final EnumSet<Part.Type> TYPES_SUPPORTING_CASE_INSENSITIVITY;
    private static final EnumSet<Part.Type> TYPES_SUPPORTED_FOR_COMPOSITES;
    private final Neo4jMappingContext mappingContext;
    private final Neo4jQueryMethod queryMethod;

    PartValidator(Neo4jMappingContext mappingContext, Neo4jQueryMethod queryMethod) {
        this.mappingContext = mappingContext;
        this.queryMethod = queryMethod;
    }

    void validatePart(Part part) {
        this.validateIgnoreCase(part);
        switch (part.getType()) {
            case AFTER: 
            case BEFORE: {
                this.validateTemporalProperty(part);
                break;
            }
            case IS_EMPTY: 
            case IS_NOT_EMPTY: {
                this.validateCollectionProperty(part);
                break;
            }
            case NEAR: 
            case WITHIN: {
                this.validatePointProperty(part);
            }
        }
        if (!TYPES_SUPPORTED_FOR_COMPOSITES.contains(part.getType())) {
            this.validateNotACompositeProperty(part);
        }
    }

    private void validateNotACompositeProperty(Part part) {
        PersistentPropertyPath path = this.mappingContext.getPersistentPropertyPath(part.getProperty());
        Neo4jPersistentProperty property = (Neo4jPersistentProperty)path.getRequiredLeafProperty();
        Assert.isTrue((!property.isComposite() ? 1 : 0) != 0, (String)"Can not derive query for '%s': Derived queries are not supported for composite properties.");
    }

    private void validateIgnoreCase(Part part) {
        Assert.isTrue((part.shouldIgnoreCase() != Part.IgnoreCaseType.ALWAYS || PartValidator.canIgnoreCase(part) ? 1 : 0) != 0, () -> String.format("Can not derive query for '%s': Only the case of String based properties can be ignored within the following keywords: %s.", new Object[]{this.queryMethod, PartValidator.formatTypes(TYPES_SUPPORTING_CASE_INSENSITIVITY)}));
    }

    private void validateTemporalProperty(Part part) {
        Assert.isTrue((boolean)COMPARABLE_TEMPORAL_TYPES.contains(part.getProperty().getLeafType()), () -> String.format("Can not derive query for '%s': The keywords %s work only with properties with one of the following types: %s.", new Object[]{this.queryMethod, PartValidator.formatTypes(Collections.singletonList(part.getType())), COMPARABLE_TEMPORAL_TYPES}));
    }

    private void validateCollectionProperty(Part part) {
        Assert.isTrue((boolean)part.getProperty().getLeafProperty().isCollection(), () -> String.format("Can not derive query for '%s': The keywords %s work only with collection properties.", new Object[]{this.queryMethod, PartValidator.formatTypes(Collections.singletonList(part.getType()))}));
    }

    private void validatePointProperty(Part part) {
        Assert.isTrue((boolean)ClassTypeInformation.from(Point.class).isAssignableFrom(part.getProperty().getLeafProperty().getTypeInformation()), () -> String.format("Can not derive query for '%s': %s works only with spatial properties.", new Object[]{this.queryMethod, part.getType()}));
    }

    private static String formatTypes(Collection<Part.Type> types) {
        return types.stream().flatMap(t -> t.getKeywords().stream()).collect(Collectors.joining(", ", "[", "]"));
    }

    static boolean canIgnoreCase(Part part) {
        return part.getProperty().getLeafType() == String.class && TYPES_SUPPORTING_CASE_INSENSITIVITY.contains(part.getType());
    }

    static {
        TreeSet<Class> hlp = new TreeSet<Class>(Comparator.comparing(Class::getName));
        hlp.addAll(Arrays.asList(LocalDate.class, OffsetTime.class, ZonedDateTime.class, LocalDateTime.class, Instant.class));
        COMPARABLE_TEMPORAL_TYPES = Collections.unmodifiableSet(hlp);
        TYPES_SUPPORTING_CASE_INSENSITIVITY = EnumSet.of(Part.Type.CONTAINING, new Part.Type[]{Part.Type.ENDING_WITH, Part.Type.LIKE, Part.Type.NEGATING_SIMPLE_PROPERTY, Part.Type.NOT_CONTAINING, Part.Type.NOT_LIKE, Part.Type.SIMPLE_PROPERTY, Part.Type.STARTING_WITH});
        TYPES_SUPPORTED_FOR_COMPOSITES = EnumSet.of(Part.Type.SIMPLE_PROPERTY, Part.Type.NEGATING_SIMPLE_PROPERTY);
    }
}

