/*
 * Copyright 2012 - 2014 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.springframework.data.solr.repository.support;

import java.io.Serializable;

import org.apache.solr.client.solrj.SolrServer;
import org.springframework.beans.factory.FactoryBean;
import org.springframework.data.repository.Repository;
import org.springframework.data.repository.core.support.RepositoryFactorySupport;
import org.springframework.data.repository.core.support.TransactionalRepositoryFactoryBeanSupport;
import org.springframework.data.solr.core.SolrOperations;
import org.springframework.util.Assert;

/**
 * Spring {@link FactoryBean} implementation to ease container based configuration for XML namespace and JavaConfig.
 * 
 * @author Oliver Gierke
 * @author Christoph Strobl
 */
public class SolrRepositoryFactoryBean<T extends Repository<S, ID>, S, ID extends Serializable> extends
		TransactionalRepositoryFactoryBeanSupport<T, S, ID> {

	private SolrServer solrServer;
	private SolrOperations operations;

	/**
	 * Configures the {@link SolrOperations} to be used to create Solr repositories.
	 * 
	 * @param operations the operations to set
	 */
	public void setSolrOperations(SolrOperations operations) {
		this.operations = operations;
	}

	public void setSolrServer(SolrServer solrServer) {
		this.solrServer = solrServer;
	}

	/**
	 * @return SolrOperations to be used for eg. custom implementation
	 */
	protected SolrOperations getSolrOperations() {
		return this.operations;
	}

	/* 
	 * (non-Javadoc)
	 * @see org.springframework.data.repository.core.support.RepositoryFactoryBeanSupport#afterPropertiesSet()
	 */
	@Override
	public void afterPropertiesSet() {
		super.afterPropertiesSet();
		Assert.isTrue((operations != null || solrServer != null), "SolrOperations or SolrServer must be configured!");
	}

	@Override
	protected RepositoryFactorySupport doCreateRepositoryFactory() {
		if (operations != null) {
			return new SolrRepositoryFactory(this.operations);
		}
		return new SolrRepositoryFactory(this.solrServer);
	}
}
