/*
 * Copyright 2013 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.springframework.yarn.integration.convert;

import java.util.Set;

import org.springframework.beans.factory.BeanFactory;
import org.springframework.beans.factory.BeanFactoryAware;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.core.convert.ConversionService;
import org.springframework.core.convert.converter.Converter;
import org.springframework.core.convert.support.ConversionServiceFactory;
import org.springframework.core.convert.support.GenericConversionService;
import org.springframework.util.Assert;
import org.springframework.yarn.integration.support.IntegrationContextUtils;

/**
 * Utility class that keeps track of a set of Converters in order to register
 * them with the "integrationConversionService" upon initialization.
 *
 * @author Oleg Zhurakousky
 * @author Mark Fisher
 * @author Janne Valkealahti
 *
 */
public class ConverterRegistrar implements InitializingBean, BeanFactoryAware {

	private final Set<Converter<?, ?>> converters;

	private BeanFactory beanFactory;

	public ConverterRegistrar(Set<Converter<?, ?>> converters) {
		this.converters = converters;
	}

	public void setBeanFactory(BeanFactory beanFactory) {
		this.beanFactory = beanFactory;
	}

	public void afterPropertiesSet() throws Exception {
		Assert.notNull(beanFactory, "BeanFactory is required");
		ConversionService conversionService = IntegrationContextUtils.getConversionService(beanFactory);
		if (conversionService instanceof GenericConversionService) {
			ConversionServiceFactory.registerConverters(converters, (GenericConversionService) conversionService);
		} else {
			Assert.notNull(conversionService, "Failed to locate '"
					+ IntegrationContextUtils.YARN_INTEGRATION_CONVERSION_SERVICE_BEAN_NAME + "'");
		}
	}

}
