/*
 * Copyright 2019-2023 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.springframework.pulsar.support.header;

import org.springframework.util.ClassUtils;

import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.MapperFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.json.JsonMapper;

/**
 * The utilities for Jackson {@link ObjectMapper} instances.
 *
 * @author Artem Bilan
 * @author Chris Bono
 */
public final class JacksonUtils {

	private static final ClassLoader classLoader = ClassUtils.getDefaultClassLoader();

	private static final boolean JACKSON_PRESENT = ClassUtils.isPresent("com.fasterxml.jackson.databind.ObjectMapper",
			classLoader) && ClassUtils.isPresent("com.fasterxml.jackson.core.JsonGenerator", classLoader);

	private static final boolean JDK8_MODULE_PRESENT = ClassUtils
			.isPresent("com.fasterxml.jackson.datatype.jdk8.Jdk8Module", null);

	private static final boolean JAVA_TIME_MODULE_PRESENT = ClassUtils
			.isPresent("com.fasterxml.jackson.datatype.jsr310.JavaTimeModule", null);

	private static final boolean JODA_MODULE_PRESENT = ClassUtils
			.isPresent("com.fasterxml.jackson.datatype.joda.JodaModule", null);

	/**
	 * Determines if the Jackson JSON processor is on the classpath.
	 * @return whether Jackson JSON processor is available on the classpath
	 */
	public static boolean isJacksonPresent() {
		return JACKSON_PRESENT;
	}

	/**
	 * Factory for {@link ObjectMapper} instances with registered well-known modules and
	 * disabled {@link MapperFeature#DEFAULT_VIEW_INCLUSION} and
	 * {@link DeserializationFeature#FAIL_ON_UNKNOWN_PROPERTIES} features.
	 * @return the {@link ObjectMapper} instance.
	 */
	public static ObjectMapper enhancedObjectMapper() {
		ObjectMapper objectMapper = JsonMapper.builder().configure(MapperFeature.DEFAULT_VIEW_INCLUSION, false)
				.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false).build();
		if (JDK8_MODULE_PRESENT) {
			objectMapper.registerModule(Jdk8ModuleProvider.MODULE);
		}
		if (JAVA_TIME_MODULE_PRESENT) {
			objectMapper.registerModule(JavaTimeModuleProvider.MODULE);
		}
		if (JODA_MODULE_PRESENT) {
			objectMapper.registerModule(JodaModuleProvider.MODULE);
		}
		return objectMapper;
	}

	private JacksonUtils() {
	}

	private static final class Jdk8ModuleProvider {

		static final com.fasterxml.jackson.databind.Module MODULE = new com.fasterxml.jackson.datatype.jdk8.Jdk8Module();

	}

	private static final class JavaTimeModuleProvider {

		static final com.fasterxml.jackson.databind.Module MODULE = new com.fasterxml.jackson.datatype.jsr310.JavaTimeModule();

	}

	private static final class JodaModuleProvider {

		static final com.fasterxml.jackson.databind.Module MODULE = new com.fasterxml.jackson.datatype.joda.JodaModule();

	}

}
