/*
 * Decompiled with CFR 0.152.
 */
package org.springframework.security.saml2.provider.service.authentication;

import java.security.PrivateKey;
import java.security.cert.X509Certificate;
import java.time.Duration;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.opensaml.core.xml.XMLObject;
import org.opensaml.saml.common.SignableSAMLObject;
import org.opensaml.saml.common.assertion.AssertionValidationException;
import org.opensaml.saml.common.assertion.ValidationContext;
import org.opensaml.saml.common.assertion.ValidationResult;
import org.opensaml.saml.saml2.assertion.SAML20AssertionValidator;
import org.opensaml.saml.saml2.assertion.impl.AudienceRestrictionConditionValidator;
import org.opensaml.saml.saml2.assertion.impl.BearerSubjectConfirmationValidator;
import org.opensaml.saml.saml2.core.Assertion;
import org.opensaml.saml.saml2.core.EncryptedAssertion;
import org.opensaml.saml.saml2.core.EncryptedID;
import org.opensaml.saml.saml2.core.NameID;
import org.opensaml.saml.saml2.core.Response;
import org.opensaml.saml.saml2.core.Subject;
import org.opensaml.saml.saml2.core.SubjectConfirmation;
import org.opensaml.saml.saml2.encryption.Decrypter;
import org.opensaml.saml.security.impl.SAMLSignatureProfileValidator;
import org.opensaml.security.credential.Credential;
import org.opensaml.security.credential.CredentialResolver;
import org.opensaml.security.credential.CredentialSupport;
import org.opensaml.security.credential.impl.CollectionCredentialResolver;
import org.opensaml.security.x509.BasicX509Credential;
import org.opensaml.xmlsec.config.impl.DefaultSecurityConfigurationBootstrap;
import org.opensaml.xmlsec.encryption.support.DecryptionException;
import org.opensaml.xmlsec.keyinfo.KeyInfoCredentialResolver;
import org.opensaml.xmlsec.keyinfo.impl.StaticKeyInfoCredentialResolver;
import org.opensaml.xmlsec.signature.Signature;
import org.opensaml.xmlsec.signature.support.SignatureException;
import org.opensaml.xmlsec.signature.support.SignaturePrevalidator;
import org.opensaml.xmlsec.signature.support.SignatureTrustEngine;
import org.opensaml.xmlsec.signature.support.SignatureValidator;
import org.opensaml.xmlsec.signature.support.impl.ExplicitKeySignatureTrustEngine;
import org.springframework.core.convert.converter.Converter;
import org.springframework.security.authentication.AuthenticationProvider;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.AuthenticationException;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.authority.SimpleGrantedAuthority;
import org.springframework.security.core.authority.mapping.GrantedAuthoritiesMapper;
import org.springframework.security.saml2.Saml2Exception;
import org.springframework.security.saml2.credentials.Saml2X509Credential;
import org.springframework.security.saml2.provider.service.authentication.OpenSamlImplementation;
import org.springframework.security.saml2.provider.service.authentication.Saml2Authentication;
import org.springframework.security.saml2.provider.service.authentication.Saml2AuthenticationException;
import org.springframework.security.saml2.provider.service.authentication.Saml2AuthenticationToken;
import org.springframework.security.saml2.provider.service.authentication.Saml2Error;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;

public final class OpenSamlAuthenticationProvider
implements AuthenticationProvider {
    private static Log logger = LogFactory.getLog(OpenSamlAuthenticationProvider.class);
    private final OpenSamlImplementation saml = OpenSamlImplementation.getInstance();
    private Converter<Assertion, Collection<? extends GrantedAuthority>> authoritiesExtractor = a -> Collections.singletonList(new SimpleGrantedAuthority("ROLE_USER"));
    private GrantedAuthoritiesMapper authoritiesMapper = a -> a;
    private Duration responseTimeValidationSkew = Duration.ofMinutes(5L);

    public void setAuthoritiesExtractor(Converter<Assertion, Collection<? extends GrantedAuthority>> authoritiesExtractor) {
        Assert.notNull(authoritiesExtractor, (String)"authoritiesExtractor cannot be null");
        this.authoritiesExtractor = authoritiesExtractor;
    }

    public void setAuthoritiesMapper(GrantedAuthoritiesMapper authoritiesMapper) {
        Assert.notNull((Object)authoritiesMapper, (String)"authoritiesMapper cannot be null");
        this.authoritiesMapper = authoritiesMapper;
    }

    public void setResponseTimeValidationSkew(Duration responseTimeValidationSkew) {
        this.responseTimeValidationSkew = responseTimeValidationSkew;
    }

    public Authentication authenticate(Authentication authentication) throws AuthenticationException {
        try {
            Saml2AuthenticationToken token = (Saml2AuthenticationToken)authentication;
            Response samlResponse = this.getSaml2Response(token);
            Assertion assertion = this.validateSaml2Response(token, token.getRecipientUri(), samlResponse);
            String username = this.getUsername(token, assertion);
            return new Saml2Authentication(() -> username, token.getSaml2Response(), this.authoritiesMapper.mapAuthorities(this.getAssertionAuthorities(assertion)));
        }
        catch (Saml2AuthenticationException e) {
            throw e;
        }
        catch (Exception e) {
            throw this.authException("internal_validation_error", e.getMessage(), e);
        }
    }

    public boolean supports(Class<?> authentication) {
        return authentication != null && Saml2AuthenticationToken.class.isAssignableFrom(authentication);
    }

    private Collection<? extends GrantedAuthority> getAssertionAuthorities(Assertion assertion) {
        return (Collection)this.authoritiesExtractor.convert((Object)assertion);
    }

    private String getUsername(Saml2AuthenticationToken token, Assertion assertion) throws Saml2AuthenticationException {
        String username = null;
        Subject subject = assertion.getSubject();
        if (subject == null) {
            throw this.authException("subject_not_found", "Assertion [" + assertion.getID() + "] is missing a subject");
        }
        if (subject.getNameID() != null) {
            username = subject.getNameID().getValue();
        } else if (subject.getEncryptedID() != null) {
            NameID nameId = this.decrypt(token, subject.getEncryptedID());
            username = nameId.getValue();
        }
        if (username == null) {
            throw this.authException("username_not_found", "Assertion [" + assertion.getID() + "] is missing a user identifier");
        }
        return username;
    }

    private Assertion validateSaml2Response(Saml2AuthenticationToken token, String recipient, Response samlResponse) throws Saml2AuthenticationException {
        if (StringUtils.hasText((String)samlResponse.getDestination()) && !recipient.equals(samlResponse.getDestination())) {
            throw this.authException("invalid_destination", "Invalid SAML response destination: " + samlResponse.getDestination());
        }
        String issuer = samlResponse.getIssuer().getValue();
        if (logger.isDebugEnabled()) {
            logger.debug((Object)("Validating SAML response from " + issuer));
        }
        if (!StringUtils.hasText((String)issuer) || !issuer.equals(token.getIdpEntityId())) {
            String message = String.format("Response issuer '%s' doesn't match '%s'", issuer, token.getIdpEntityId());
            throw this.authException("invalid_issuer", message);
        }
        Saml2AuthenticationException lastValidationError = null;
        boolean responseSigned = this.hasValidSignature((SignableSAMLObject)samlResponse, token);
        for (Assertion a : samlResponse.getAssertions()) {
            if (logger.isDebugEnabled()) {
                logger.debug((Object)("Checking plain assertion validity " + a));
            }
            try {
                this.validateAssertion(recipient, a, token, !responseSigned);
                return a;
            }
            catch (Saml2AuthenticationException e) {
                lastValidationError = e;
            }
        }
        for (EncryptedAssertion ea : samlResponse.getEncryptedAssertions()) {
            if (logger.isDebugEnabled()) {
                logger.debug((Object)("Checking encrypted assertion validity " + ea));
            }
            try {
                Assertion a = this.decrypt(token, ea);
                this.validateAssertion(recipient, a, token, !responseSigned);
                return a;
            }
            catch (Saml2AuthenticationException e) {
                lastValidationError = e;
            }
        }
        if (lastValidationError != null) {
            throw lastValidationError;
        }
        throw this.authException("malformed_response_data", "No assertions found in response.");
    }

    private boolean hasValidSignature(SignableSAMLObject samlObject, Saml2AuthenticationToken token) {
        if (!samlObject.isSigned()) {
            if (logger.isDebugEnabled()) {
                logger.debug((Object)"SAML object is not signed, no signatures found");
            }
            return false;
        }
        List<X509Certificate> verificationKeys = this.getVerificationCertificates(token);
        if (verificationKeys.isEmpty()) {
            return false;
        }
        for (X509Certificate certificate : verificationKeys) {
            Credential credential = this.getVerificationCredential(certificate);
            try {
                SignatureValidator.validate((Signature)samlObject.getSignature(), (Credential)credential);
                if (logger.isDebugEnabled()) {
                    logger.debug((Object)("Valid signature found in SAML object:" + samlObject.getClass().getName()));
                }
                return true;
            }
            catch (SignatureException ignored) {
                if (logger.isTraceEnabled()) {
                    logger.trace((Object)("Signature validation failed with cert:" + certificate.toString()), (Throwable)ignored);
                    continue;
                }
                if (!logger.isDebugEnabled()) continue;
                logger.debug((Object)("Signature validation failed with cert:" + certificate.toString()));
            }
        }
        return false;
    }

    private void validateAssertion(String recipient, Assertion a, Saml2AuthenticationToken token, boolean signatureRequired) {
        SAML20AssertionValidator validator = this.getAssertionValidator(token);
        HashMap<String, Object> validationParams = new HashMap<String, Object>();
        validationParams.put("saml2.SignatureRequired", false);
        validationParams.put("saml2.ClockSkew", this.responseTimeValidationSkew.toMillis());
        validationParams.put("saml2.Conditions.ValidAudiences", Collections.singleton(token.getLocalSpEntityId()));
        if (StringUtils.hasText((String)recipient)) {
            validationParams.put("saml2.SubjectConfirmation.ValidRecipients", Collections.singleton(recipient));
        }
        if (signatureRequired && !this.hasValidSignature((SignableSAMLObject)a, token)) {
            if (logger.isDebugEnabled()) {
                logger.debug((Object)String.format("Assertion [%s] does not a valid signature.", a.getID()));
            }
            throw this.authException("invalid_signature", "Assertion doesn't have a valid signature.");
        }
        a.setSignature(null);
        if (a.getSubject() != null) {
            for (SubjectConfirmation sc : a.getSubject().getSubjectConfirmations()) {
                if (sc.getSubjectConfirmationData() == null) continue;
                sc.getSubjectConfirmationData().setAddress(null);
            }
        }
        ValidationContext vctx = new ValidationContext(validationParams);
        try {
            ValidationResult result = validator.validate(a, vctx);
            boolean valid = result.equals((Object)ValidationResult.VALID);
            if (!valid) {
                if (logger.isDebugEnabled()) {
                    logger.debug((Object)String.format("Failed to validate assertion from %s", token.getIdpEntityId()));
                }
                throw this.authException("invalid_assertion", vctx.getValidationFailureMessage());
            }
        }
        catch (AssertionValidationException e) {
            if (logger.isDebugEnabled()) {
                logger.debug((Object)"Failed to validate assertion:", (Throwable)e);
            }
            throw this.authException("internal_validation_error", e.getMessage(), (Exception)((Object)e));
        }
    }

    private Response getSaml2Response(Saml2AuthenticationToken token) throws Saml2Exception, Saml2AuthenticationException {
        try {
            XMLObject result = this.saml.resolve(token.getSaml2Response());
            if (result instanceof Response) {
                return (Response)result;
            }
            throw this.authException("unknown_response_class", "Invalid response class:" + result.getClass().getName());
        }
        catch (Saml2Exception x) {
            throw this.authException("malformed_response_data", x.getMessage(), x);
        }
    }

    private Saml2Error validationError(String code, String description) {
        return new Saml2Error(code, description);
    }

    private Saml2AuthenticationException authException(String code, String description) throws Saml2AuthenticationException {
        return new Saml2AuthenticationException(this.validationError(code, description));
    }

    private Saml2AuthenticationException authException(String code, String description, Exception cause) throws Saml2AuthenticationException {
        return new Saml2AuthenticationException(this.validationError(code, description), cause);
    }

    private SAML20AssertionValidator getAssertionValidator(Saml2AuthenticationToken provider) {
        List<AudienceRestrictionConditionValidator> conditions = Collections.singletonList(new AudienceRestrictionConditionValidator());
        BearerSubjectConfirmationValidator subjectConfirmationValidator = new BearerSubjectConfirmationValidator();
        List<BearerSubjectConfirmationValidator> subjects = Collections.singletonList(subjectConfirmationValidator);
        List statements = Collections.emptyList();
        HashSet<Credential> credentials = new HashSet<Credential>();
        for (X509Certificate key : this.getVerificationCertificates(provider)) {
            Credential cred = this.getVerificationCredential(key);
            credentials.add(cred);
        }
        CollectionCredentialResolver credentialsResolver = new CollectionCredentialResolver(credentials);
        ExplicitKeySignatureTrustEngine signatureTrustEngine = new ExplicitKeySignatureTrustEngine((CredentialResolver)credentialsResolver, DefaultSecurityConfigurationBootstrap.buildBasicInlineKeyInfoCredentialResolver());
        SAMLSignatureProfileValidator signaturePrevalidator = new SAMLSignatureProfileValidator();
        return new SAML20AssertionValidator(conditions, subjects, statements, (SignatureTrustEngine)signatureTrustEngine, (SignaturePrevalidator)signaturePrevalidator);
    }

    private Credential getVerificationCredential(X509Certificate certificate) {
        return CredentialSupport.getSimpleCredential((X509Certificate)certificate, null);
    }

    private Decrypter getDecrypter(Saml2X509Credential key) {
        BasicX509Credential credential = CredentialSupport.getSimpleCredential((X509Certificate)key.getCertificate(), (PrivateKey)key.getPrivateKey());
        StaticKeyInfoCredentialResolver resolver = new StaticKeyInfoCredentialResolver((Credential)credential);
        Decrypter decrypter = new Decrypter(null, (KeyInfoCredentialResolver)resolver, this.saml.getEncryptedKeyResolver());
        decrypter.setRootInNewDocument(true);
        return decrypter;
    }

    private Assertion decrypt(Saml2AuthenticationToken token, EncryptedAssertion assertion) throws Saml2AuthenticationException {
        Saml2AuthenticationException last = null;
        List<Saml2X509Credential> decryptionCredentials = this.getDecryptionCredentials(token);
        if (decryptionCredentials.isEmpty()) {
            throw this.authException("decryption_error", "No valid decryption credentials found.");
        }
        for (Saml2X509Credential key : decryptionCredentials) {
            Decrypter decrypter = this.getDecrypter(key);
            try {
                return decrypter.decrypt(assertion);
            }
            catch (DecryptionException e) {
                last = this.authException("decryption_error", e.getMessage(), (Exception)((Object)e));
            }
        }
        throw last;
    }

    private NameID decrypt(Saml2AuthenticationToken token, EncryptedID assertion) throws Saml2AuthenticationException {
        Saml2AuthenticationException last = null;
        List<Saml2X509Credential> decryptionCredentials = this.getDecryptionCredentials(token);
        if (decryptionCredentials.isEmpty()) {
            throw this.authException("decryption_error", "No valid decryption credentials found.");
        }
        for (Saml2X509Credential key : decryptionCredentials) {
            Decrypter decrypter = this.getDecrypter(key);
            try {
                return (NameID)decrypter.decrypt(assertion);
            }
            catch (DecryptionException e) {
                last = this.authException("decryption_error", e.getMessage(), (Exception)((Object)e));
            }
        }
        throw last;
    }

    private List<Saml2X509Credential> getDecryptionCredentials(Saml2AuthenticationToken token) {
        LinkedList<Saml2X509Credential> result = new LinkedList<Saml2X509Credential>();
        for (Saml2X509Credential c : token.getX509Credentials()) {
            if (!c.isDecryptionCredential()) continue;
            result.add(c);
        }
        return result;
    }

    private List<X509Certificate> getVerificationCertificates(Saml2AuthenticationToken token) {
        LinkedList<X509Certificate> result = new LinkedList<X509Certificate>();
        for (Saml2X509Credential c : token.getX509Credentials()) {
            if (!c.isSignatureVerficationCredential()) continue;
            result.add(c.getCertificate());
        }
        return result;
    }
}

