// Copyright (c) 2019-2020 by Rob Norris and Contributors
// This software is licensed under the MIT License (MIT).
// For more information see LICENSE or https://opensource.org/licenses/MIT

package natchez
package xray

import cats.data.OptionT
import cats.effect._
import cats.effect.std.Random
import cats.syntax.all._
import io.circe._
import io.circe.syntax._
import com.comcast.ip4s._
import fs2.io.net.{Datagram, DatagramSocket}
import scodec.bits.ByteVector

final class XRayEntryPoint[F[_]: Concurrent: Clock: Random: XRayEnvironment](
    socket: DatagramSocket[F],
    daemonAddress: SocketAddress[IpAddress],
    useEnvironmentFallback: Boolean
) extends EntryPoint[F] {

  def sendSegment(foo: JsonObject): F[Unit] = {
    val body = io.circe.Printer.noSpaces.printToByteBuffer(foo.asJson)
    val payload = XRayEntryPoint.header ++ ByteVector.apply(body)
    val datagram = Datagram(daemonAddress, fs2.Chunk.byteVector(payload))
    socket.write(datagram)
  }

  def make(span: F[XRaySpan[F]]): Resource[F, Span[F]] =
    Resource.makeCase(span)(XRaySpan.finish(_, this, _)).widen

  override def root(name: String, options: Span.Options): Resource[F, Span[F]] =
    make(XRaySpan.root(name, this, options))

  override def continue(name: String, kernel: Kernel, options: Span.Options): Resource[F, Span[F]] =
    make(
      OptionT(XRaySpan.fromKernel(name, kernel, this, useEnvironmentFallback, options))
        .getOrElseF(new NoSuchElementException().raiseError)
    )

  override def continueOrElseRoot(
      name: String,
      kernel: Kernel,
      options: Span.Options
  ): Resource[F, Span[F]] =
    make(XRaySpan.fromKernelOrElseRoot(name, kernel, this, useEnvironmentFallback, options))
}

object XRayEntryPoint {
  val header = ByteVector
    .encodeUtf8("{\"format\": \"json\", \"version\": 1}\n")
    .getOrElse(ByteVector.empty)
}
