// ============================================================================
//   Copyright 2006-2009 Daniel W. Dyer
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.
// ============================================================================
package org.uncommons.watchmaker.framework;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Random;
import org.uncommons.maths.statistics.DataSet;

/**
 * Base class for {@link EvolutionEngine} implementations.
 * @param <T> The type of entity evolved by the evolution engine.
 * @author Daniel Dyer
 * @see CandidateFactory
 * @see FitnessEvaluator
 * @see SelectionStrategy
 * @see EvolutionaryOperator 
 */
public abstract class AbstractEvolutionEngine<T> implements EvolutionEngine<T>
{
    private final List<EvolutionObserver<? super T>> observers = new LinkedList<EvolutionObserver<? super T>>();
    private final Random rng;
    private final CandidateFactory<T> candidateFactory;
    private final EvolutionaryOperator<T> evolutionScheme;
    private final FitnessEvaluator<? super T> fitnessEvaluator;
    private final SelectionStrategy<? super T> selectionStrategy;

    private long startTime;
    private int currentGenerationIndex;

    private List<TerminationCondition> satisfiedTerminationConditions;


    /**
     * Creates a new evolution engine by specifying the various components required by
     * an evolutionary algorithm.
     * @param candidateFactory Factory used to create the initial population that is
     * iteratively evolved.
     * @param evolutionScheme The combination of evolutionary operators used to evolve
     * the population at each generation.
     * @param fitnessEvaluator A function for assigning fitness scores to candidate
     * solutions.
     * @param selectionStrategy A strategy for selecting which candidates survive to
     * be evolved.
     * @param rng The source of randomness used by all stochastic processes (including
     * evolutionary operators and selection strategies).
     */
    protected AbstractEvolutionEngine(CandidateFactory<T> candidateFactory,
                                      EvolutionaryOperator<T> evolutionScheme,
                                      FitnessEvaluator<? super T> fitnessEvaluator,
                                      SelectionStrategy<? super T> selectionStrategy,
                                      Random rng)
    {
        this.candidateFactory = candidateFactory;
        this.evolutionScheme = evolutionScheme;
        this.fitnessEvaluator = fitnessEvaluator;
        this.selectionStrategy = selectionStrategy;
        this.rng = rng;
    }


    /**
     * Provides sub-classes with access to the fitness evaluator.
     * @return A reference to the fitness evaluator configured for this engine.
     */
    protected final FitnessEvaluator<? super T> getFitnessEvaluator()
    {
        return fitnessEvaluator;
    }


    /**
     * {@inheritDoc}
     *
     * <em>If you interrupt the request thread before this method returns, the
     * method will return prematurely (with the best individual found so far).
     * After returning in this way, the current thread's interrupted flag
     * will be set.  It is preferable to use an appropritate
     * {@link TerminationCondition} rather than interrupting the evolution in
     * this way.</em>
     */
    public T evolve(int populationSize,
                    int eliteCount,
                    TerminationCondition... conditions)
    {
        return evolve(populationSize,
                      eliteCount,
                      Collections.<T>emptySet(),
                      conditions);
    }


    /**
     * {@inheritDoc}
     *
     * <em>If you interrupt the request thread before this method returns, the
     * method will return prematurely (with the best individual found so far).
     * After returning in this way, the current thread's interrupted flag
     * will be set.  It is preferable to use an appropritate
     * {@link TerminationCondition} rather than interrupting the evolution in
     * this way.</em>
     */
    public T evolve(int populationSize,
                    int eliteCount,
                    Collection<T> seedCandidates,
                    TerminationCondition... conditions)
    {
        List<EvaluatedCandidate<T>> evaluatedPopulation = evolvePopulation(populationSize,
                                                                           eliteCount,
                                                                           seedCandidates,
                                                                           conditions);
        // Once we have completed the final generation, we need to pick one of
        // the individuals in the population to return as the result of the
        // algorithm.  Usually we would just need to pick the fittest individual
        // and return that.  However, this doesn't work very well with interactive
        // evolutionary algorithms because all individuals have a nominal fitness
        // of zero and the population has been evolved one final time since the
        // user last expressed a selection preference.
        // 
        // The solution is to always use the selection strategy to pick the candidate
        // to return.  We only let it select from the set of candidates with the
        // highest fitness score.  In the case that there is a clear fittest
        // individual, there will be only one member of this set and the "winner" is
        // clear.
        //
        // In other situations, there may be multiple "best" individuals with equal
        // fitness scores.  In this case, any other individuals (those with lesser
        // fitness scores) are discarded and selection is applied to the remainder.
        // In the case of non-interactive selection, it doesn't really matter which
        // of these fittest individuals is returned since they are all equivalent
        // from a fitness perspective.
        //
        // This approach works well for interactive evolutionary algorithms.  Because
        // all fitness scores are equal, no individuals are discared before selection
        // and the user gets to have the final say over which individual is chosen as
        // the "best" from the final evolved generation.

        // The evaluated population is sorted in order of fitness, so we can just scan
        // the list and retain all individuals with a fitness equal to the first
        // individual.
        List<EvaluatedCandidate<T>> fittest = new ArrayList<EvaluatedCandidate<T>>(evaluatedPopulation.size());
        double bestFitness = evaluatedPopulation.get(0).getFitness();
        for (EvaluatedCandidate<T> candidate : evaluatedPopulation)
        {
            if ((fitnessEvaluator.isNatural() && candidate.getFitness() >= bestFitness)
                || (!fitnessEvaluator.isNatural() && candidate.getFitness() <= bestFitness))
            {
                fittest.add(candidate);
            }
            else
            {
                break;
            }
        }
        return selectionStrategy.select(fittest, fitnessEvaluator.isNatural(), 1, rng).get(0);
    }



    /**
     * {@inheritDoc}
     *
     * <em>If you interrupt the request thread before this method returns, the
     * method will return prematurely (with the members of the most recent
     * generation).
     * After returning in this way, the current thread's interrupted flag
     * will be set.  It is preferable to use an appropritate
     * {@link TerminationCondition} rather than interrupting the evolution in
     * this way.</em>
     */
    public List<EvaluatedCandidate<T>> evolvePopulation(int populationSize,
                                                        int eliteCount,
                                                        TerminationCondition... conditions)
    {
        return evolvePopulation(populationSize,
                                eliteCount,
                                Collections.<T>emptySet(),
                                conditions);
    }



    /**
     * {@inheritDoc}
     *
     * <em>If you interrupt the request thread before this method returns, the
     * method will return prematurely (with the members of the most recent
     * generation).
     * After returning in this way, the current thread's interrupted flag
     * will be set.  It is preferable to use an appropritate
     * {@link TerminationCondition} rather than interrupting the evolution in
     * this way.</em>
     */
    public List<EvaluatedCandidate<T>> evolvePopulation(int populationSize,
                                                        int eliteCount,
                                                        Collection<T> seedCandidates,
                                                        TerminationCondition... conditions)
    {
        if (eliteCount < 0 || eliteCount >= populationSize)
        {
            throw new IllegalArgumentException("Elite count must be non-negative and less than population size.");
        }
        if (conditions.length == 0)
        {
            throw new IllegalArgumentException("At least one TerminationCondition must be specified.");
        }

        satisfiedTerminationConditions = null;
        currentGenerationIndex = 0;
        startTime = System.currentTimeMillis();

        List<T> population = candidateFactory.generateInitialPopulation(populationSize,
                                                                        seedCandidates,
                                                                        rng);
        
        // Calculate the fitness scores for each member of the initial population.
        List<EvaluatedCandidate<T>> evaluatedPopulation = evaluatePopulation(population);
        sortEvaluatedPopulation(evaluatedPopulation);
        PopulationData<T> data = getPopulationData(evaluatedPopulation, eliteCount);
        // Notify observers of the state of the population.
        notifyPopulationChange(data);

        List<TerminationCondition> satisfiedConditions = shouldContinue(data, conditions);
        while (satisfiedConditions == null)
        {
            ++currentGenerationIndex;
            population = createNextGeneration(evaluatedPopulation, eliteCount);
            evaluatedPopulation = evaluatePopulation(population);
            sortEvaluatedPopulation(evaluatedPopulation);
            data = getPopulationData(evaluatedPopulation, eliteCount);
            // Notify observers of the state of the population.
            notifyPopulationChange(data);
            satisfiedConditions = shouldContinue(data, conditions);
        }
        this.satisfiedTerminationConditions = satisfiedConditions;
        return evaluatedPopulation;
    }



    private List<TerminationCondition> shouldContinue(PopulationData<T> data,
                                                      TerminationCondition... conditions)
    {
        // If the thread has been interrupted, we should abort and return whatever
        // result we currently have.
        if (Thread.currentThread().isInterrupted())
        {
            return Collections.emptyList();
        }
        // Otherwise check the termination conditions for the evolution.
        List<TerminationCondition> satisfiedConditions = new LinkedList<TerminationCondition>();
        for (TerminationCondition condition : conditions)
        {
            if (condition.shouldTerminate(data))
            {
                satisfiedConditions.add(condition);
            }
        }
        return satisfiedConditions.isEmpty() ? null : satisfiedConditions;
    }


    /**
     * {@inheritDoc}
     */
    public List<TerminationCondition> getSatisfiedTerminationConditions()
    {
        if (satisfiedTerminationConditions == null)
        {
            throw new IllegalStateException("EvolutionEngine has not terminated.");
        }
        else
        {
            return Collections.unmodifiableList(satisfiedTerminationConditions);
        }
    }


    /**
     * Takes a population, assigns a fitness score to each member and returns
     * the members with their scores attached.  Order is not important since
     * the returned population will be sorted later if required.
     * @param population The population of evolved candidate to be evaluated.
     * @return A list containing each of the candidates with an attached fitness
     * score.
     */
    protected abstract List<EvaluatedCandidate<T>> evaluatePopulation(List<T> population);

    
    /**
     * Sorts an evaluated population in descending order of fitness
     * (descending order of fitness score for natural scores, ascending
     * order of scores for non-natural scores).
     * @param evaluatedPopulation The population to be sorted (in-place).
     */
    private void sortEvaluatedPopulation(List<EvaluatedCandidate<T>> evaluatedPopulation)
    {
        // Sort candidates in descending order according to fitness.
        if (getFitnessEvaluator().isNatural()) // Descending values for natural fitness.
        {
            Collections.sort(evaluatedPopulation, Collections.reverseOrder());
        }
        else // Ascending values for non-natural fitness.
        {
            Collections.sort(evaluatedPopulation);
        }
    }


    /**
     * Evolve the specified evaluated population (the current generation).
     * and return the resultant population (the next generation).
     * @param evaluatedPopulation The population of evolved candidates, with
     * fitness scores attached.
     * @param eliteCount The number of the most fit candidates that will be
     * preserved unchanged in the next generation.
     * @return The next generation of evolved individuals.
     */
    private List<T> createNextGeneration(List<EvaluatedCandidate<T>> evaluatedPopulation,
                                         int eliteCount)
    {
        List<T> population = new ArrayList<T>(evaluatedPopulation.size());

        // First perform any elitist selection.
        List<T> elite = new ArrayList<T>(eliteCount);
        Iterator<EvaluatedCandidate<T>> iterator = evaluatedPopulation.iterator();
        while (elite.size() < eliteCount)
        {
            elite.add(iterator.next().getCandidate());
        }
        // Then select candidates that will be operated on to create the evolved
        // portion of the next generation.
        population.addAll(selectionStrategy.select(evaluatedPopulation,
                                                   fitnessEvaluator.isNatural(),
                                                   evaluatedPopulation.size() - eliteCount,
                                                   rng));
        // Then evolve the population.
        population = evolutionScheme.apply(population, rng);
        // When the evolution is finished, add the elite to the population.
        population.addAll(elite);
        return population;
    }


    /**
     * {@inheritDoc}
     *
     * Updates are dispatched synchronously on the request thread.  Observers should
     * complete their processing and return in a timely manner to avoid holding up
     * the evolution.
     */
    public void addEvolutionObserver(EvolutionObserver<? super T> observer)
    {
        observers.add(observer);
    }


    /**
     * {@inheritDoc}
     */
    public void removeEvolutionObserver(EvolutionObserver<? super T> observer)
    {
        observers.remove(observer);
    }


    /**
     * Send the population data to all registered observers.
     * @param data Information about the current state of the population.
     */
    private void notifyPopulationChange(PopulationData<T> data)
    {
        for (EvolutionObserver<? super T> observer : observers)
        {
            observer.populationUpdate(data);
        }
    }


    /**
     * Gets data about the current population, including the fittest candidate
     * and statistics about the population as a whole.
     * @param evaluatedPopulation Population of candidate solutions with their
     * associated fitness scores.
     * @param eliteCount The number of candidates preserved via elitism.
     * @return Statistics about the current generation of evolved individuals.
     */
    private PopulationData<T> getPopulationData(List<EvaluatedCandidate<T>> evaluatedPopulation,
                                                int eliteCount)
    {
        DataSet stats = new DataSet(evaluatedPopulation.size());
        for (EvaluatedCandidate<T> candidate : evaluatedPopulation)
        {
            stats.addValue(candidate.getFitness());
        }
        return new PopulationData<T>(evaluatedPopulation.get(0).getCandidate(),
                                     evaluatedPopulation.get(0).getFitness(),
                                     stats.getArithmeticMean(),
                                     stats.getStandardDeviation(),
                                     getFitnessEvaluator().isNatural(),
                                     stats.getSize(),
                                     eliteCount,
                                     currentGenerationIndex,
                                     System.currentTimeMillis() - startTime);
    }
}
