/**
 * This file is part of veraPDF Parser, a module of the veraPDF project.
 * Copyright (c) 2015, veraPDF Consortium <info@verapdf.org>
 * All rights reserved.
 *
 * veraPDF Parser is free software: you can redistribute it and/or modify
 * it under the terms of either:
 *
 * The GNU General public license GPLv3+.
 * You should have received a copy of the GNU General Public License
 * along with veraPDF Parser as the LICENSE.GPL file in the root of the source
 * tree.  If not, see http://www.gnu.org/licenses/ or
 * https://www.gnu.org/licenses/gpl-3.0.en.html.
 *
 * The Mozilla Public License MPLv2+.
 * You should have received a copy of the Mozilla Public License along with
 * veraPDF Parser as the LICENSE.MPL file in the root of the source tree.
 * If a copy of the MPL was not distributed with this file, you can obtain one at
 * http://mozilla.org/MPL/2.0/.
 */
package org.verapdf.tools;

import org.verapdf.as.ASAtom;
import org.verapdf.cos.COSKey;
import org.verapdf.cos.COSObjType;
import org.verapdf.cos.COSObject;
import org.verapdf.pd.structure.PDNameSpaceRoleMapping;
import org.verapdf.pd.structure.PDStructElem;
import org.verapdf.pd.structure.PDStructureNameSpace;
import org.verapdf.pd.structure.StructureType;

import java.util.*;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * @author Maksim Bezrukov
 */
public class TaggedPDFHelper {

	private static final Logger LOGGER = Logger.getLogger(TaggedPDFHelper.class.getCanonicalName());

	private static Set<String> PDF_1_7_STANDART_ROLE_TYPES;
	private static Set<String> PDF_2_0_STANDART_ROLE_TYPES;

	static {
		Set<String> tempSet = new HashSet<>();
		// Common standard structure types for PDF 1.7 and 2.0
		tempSet.add(TaggedPDFConstants.DOCUMENT);
		tempSet.add(TaggedPDFConstants.PART);
		tempSet.add(TaggedPDFConstants.DIV);
		tempSet.add(TaggedPDFConstants.CAPTION);
		tempSet.add(TaggedPDFConstants.THEAD);
		tempSet.add(TaggedPDFConstants.TBODY);
		tempSet.add(TaggedPDFConstants.TFOOT);
		tempSet.add(TaggedPDFConstants.H);
		tempSet.add(TaggedPDFConstants.P);
		tempSet.add(TaggedPDFConstants.L);
		tempSet.add(TaggedPDFConstants.LI);
		tempSet.add(TaggedPDFConstants.LBL);
		tempSet.add(TaggedPDFConstants.LBODY);
		tempSet.add(TaggedPDFConstants.TABLE);
		tempSet.add(TaggedPDFConstants.TR);
		tempSet.add(TaggedPDFConstants.TH);
		tempSet.add(TaggedPDFConstants.TD);
		tempSet.add(TaggedPDFConstants.SPAN);
		tempSet.add(TaggedPDFConstants.LINK);
		tempSet.add(TaggedPDFConstants.ANNOT);
		tempSet.add(TaggedPDFConstants.RUBY);
		tempSet.add(TaggedPDFConstants.WARICHU);
		tempSet.add(TaggedPDFConstants.FIGURE);
		tempSet.add(TaggedPDFConstants.FORMULA);
		tempSet.add(TaggedPDFConstants.FORM);
		tempSet.add(TaggedPDFConstants.RB);
		tempSet.add(TaggedPDFConstants.RT);
		tempSet.add(TaggedPDFConstants.RP);
		tempSet.add(TaggedPDFConstants.WT);
		tempSet.add(TaggedPDFConstants.WP);

		Set<String> pdf_1_7 = new HashSet<>(tempSet);

		// Standart structure types present in 1.7
		pdf_1_7.add(TaggedPDFConstants.ART);
		pdf_1_7.add(TaggedPDFConstants.SECT);
		pdf_1_7.add(TaggedPDFConstants.BLOCK_QUOTE);
		pdf_1_7.add(TaggedPDFConstants.TOC);
		pdf_1_7.add(TaggedPDFConstants.TOCI);
		pdf_1_7.add(TaggedPDFConstants.INDEX);
		pdf_1_7.add(TaggedPDFConstants.NON_STRUCT);
		pdf_1_7.add(TaggedPDFConstants.PRIVATE);
		pdf_1_7.add(TaggedPDFConstants.QUOTE);
		pdf_1_7.add(TaggedPDFConstants.NOTE);
		pdf_1_7.add(TaggedPDFConstants.REFERENCE);
		pdf_1_7.add(TaggedPDFConstants.BIB_ENTRY);
		pdf_1_7.add(TaggedPDFConstants.CODE);
		pdf_1_7.add(TaggedPDFConstants.H1);
		pdf_1_7.add(TaggedPDFConstants.H2);
		pdf_1_7.add(TaggedPDFConstants.H3);
		pdf_1_7.add(TaggedPDFConstants.H4);
		pdf_1_7.add(TaggedPDFConstants.H5);
		pdf_1_7.add(TaggedPDFConstants.H6);

		Set<String> pdf_2_0 = new HashSet<>(tempSet);

		pdf_2_0.add(TaggedPDFConstants.DOCUMENT_FRAGMENT);
		pdf_2_0.add(TaggedPDFConstants.ASIDE);
		pdf_2_0.add(TaggedPDFConstants.TITLE);
		pdf_2_0.add(TaggedPDFConstants.FENOTE);
		pdf_2_0.add(TaggedPDFConstants.SUB);
		pdf_2_0.add(TaggedPDFConstants.EM);
		pdf_2_0.add(TaggedPDFConstants.STRONG);
		pdf_2_0.add(TaggedPDFConstants.ARTIFACT);

		PDF_1_7_STANDART_ROLE_TYPES = Collections.unmodifiableSet(pdf_1_7);
		PDF_2_0_STANDART_ROLE_TYPES = Collections.unmodifiableSet(pdf_2_0);
	}

	private static final int MAX_NUMBER_OF_ELEMENTS = 1;
	private static Map<ASAtom, Set<COSKey>> visitedWithNS = new HashMap<>();
	private static Set<ASAtom> visitedWithoutNS = new HashSet<>();

	private TaggedPDFHelper() {
		// disable default constructor
	}

	public static StructureType getDefaultStructureType(StructureType type, Map<ASAtom, ASAtom> rootRoleMap) {
		visitedWithNS.clear();
		visitedWithoutNS.clear();
		addVisited(type);
		StructureType curr = getEquivalent(type, rootRoleMap);
		if (curr == null || isVisited(curr)) {
			return isStandardType(type) ? type : null;
		}
		while (curr != null && !isVisited(curr)) {
			if (isStandardType(curr)) {
				return curr;
			}
			addVisited(curr);
			curr = getEquivalent(curr, rootRoleMap);
		}
		return null;
	}

	private static StructureType getEquivalent(StructureType type, Map<ASAtom, ASAtom> rootRoleMap) {
		PDStructureNameSpace nameSpace = type.getNameSpace();
		if (nameSpace != null) {
			PDNameSpaceRoleMapping nameSpaceMapping = nameSpace.getNameSpaceMapping();
			if (nameSpaceMapping != null) {
				return nameSpaceMapping.getEquivalentType(type.getType());
			} else if (!TaggedPDFConstants.PDF_NAMESPACE.equals(nameSpace.getNS())) {
				return null;
			}
		}
		ASAtom equiv = rootRoleMap.get(type.getType());
		return equiv == null ? null : StructureType.createStructureType(equiv);
	}

	private static boolean isStandardType(StructureType type) {
		String structureType = type.getType().getValue();
		PDStructureNameSpace nameSpace = type.getNameSpace();
		if (nameSpace != null) {
			switch (nameSpace.getNS()) {
				case TaggedPDFConstants.PDF_NAMESPACE:
					return PDF_1_7_STANDART_ROLE_TYPES.contains(structureType);
				case TaggedPDFConstants.PDF2_NAMESPACE:
					return PDF_2_0_STANDART_ROLE_TYPES.contains(structureType)
							|| structureType.matches(TaggedPDFConstants.HN_REGEXP);
				case TaggedPDFConstants.MATH_ML_NAMESPACE:
					return true;
				default:
					return false;
			}
		} else {
			return PDF_1_7_STANDART_ROLE_TYPES.contains(structureType);
		}
	}

	private static void addVisited(StructureType type) {
		ASAtom structType = type.getType();
		PDStructureNameSpace nameSpace = type.getNameSpace();
		if (nameSpace != null) {
			COSKey key = nameSpace.getObject().getObjectKey();
			Set<COSKey> nameSpaces;
			if (visitedWithNS.containsKey(structType)) {
				nameSpaces = visitedWithNS.get(structType);
			} else {
				nameSpaces = new HashSet<>();
				visitedWithNS.put(structType, nameSpaces);
			}
			nameSpaces.add(key);
		} else {
			visitedWithoutNS.add(structType);
		}
	}

	private static boolean isVisited(StructureType type) {
		ASAtom structType = type.getType();
		PDStructureNameSpace nameSpace = type.getNameSpace();
		if (nameSpace != null) {
			if (visitedWithNS.containsKey(structType)) {
				Set<COSKey> nameSpaces = visitedWithNS.get(structType);
				COSKey key = nameSpace.getObject().getObjectKey();
				return nameSpaces.contains(key);
			} else {
				return false;
			}
		} else {
			return visitedWithoutNS.contains(structType);
		}
	}

	public static List<PDStructElem> getStructTreeRootChildren(COSObject parent, Map<ASAtom, ASAtom> roleMap) {
		return getChildren(parent, roleMap, false);
	}

	public static List<PDStructElem> getStructElemChildren(COSObject parent, Map<ASAtom, ASAtom> roleMap) {
		return getChildren(parent, roleMap, true);
	}

	/**
	 * Get all structure elements for current dictionary
	 *
	 * @param parent parent dictionary
	 * @return list of structure elements
	 */
	private static List<PDStructElem> getChildren(COSObject parent, Map<ASAtom, ASAtom> roleMap, boolean checkType) {
		if (parent == null || parent.getType() != COSObjType.COS_DICT) {
			LOGGER.log(Level.FINE, "Parent element for struct elements is null or not a COSDictionary");
			return Collections.emptyList();
		}

		COSObject children = parent.getKey(ASAtom.K);
		if (children != null) {
			if (isStructElem(children, checkType)) {
				List<PDStructElem> list = new ArrayList<>(MAX_NUMBER_OF_ELEMENTS);
				list.add(new PDStructElem(children, roleMap));
				return Collections.unmodifiableList(list);
			} else if (children.getType() == COSObjType.COS_ARRAY) {
				return getChildrenFromArray(children, roleMap, checkType);
			}
		}
		return Collections.emptyList();
	}

	/**
	 * Transform array of dictionaries to list of structure elements
	 *
	 * @param children array of children structure elements
	 * @return list of structure elements
	 */
	private static List<PDStructElem> getChildrenFromArray(COSObject children, Map<ASAtom, ASAtom> roleMap, boolean checkType) {
		if (children.size().intValue() > 0) {
			List<PDStructElem> list = new ArrayList<>();
			for (int i = 0; i < children.size().intValue(); ++i) {
				COSObject elem = children.at(i);
				if (isStructElem(elem, checkType)) {
					list.add(new PDStructElem(elem, roleMap));
				}
			}
			return Collections.unmodifiableList(list);
		}
		return Collections.emptyList();
	}

	private static boolean isStructElem(COSObject dictionary, boolean checkType) {
		if (dictionary == null || dictionary.getType() != COSObjType.COS_DICT) {
			return false;
		}
		ASAtom type = dictionary.getNameKey(ASAtom.TYPE);
		return !checkType || type == null || type.equals(ASAtom.STRUCT_ELEM);
	}

	public static boolean isContentItem(COSObject obj) {
		if (obj == null || obj.empty()) {
			return false;
		}
		if (obj.getType() == COSObjType.COS_INTEGER) {
			return true;
		}
		if (!obj.getType().isDictionaryBased()) {
			return false;
		}
		ASAtom type = obj.getNameKey(ASAtom.TYPE);
		return type == ASAtom.MCR || type == ASAtom.OBJR;
	}
}
