/*
 * Decompiled with CFR 0.152.
 */
package org.vibur.dbcp.pool;

import java.sql.Connection;
import java.sql.SQLException;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.vibur.dbcp.ViburConfig;
import org.vibur.dbcp.ViburDBCPException;
import org.vibur.dbcp.pool.ConnHolder;
import org.vibur.dbcp.pool.ViburObjectFactory;
import org.vibur.dbcp.util.JdbcUtils;

public class ConnectionFactory
implements ViburObjectFactory {
    private static final Logger logger = LoggerFactory.getLogger(ConnectionFactory.class);
    private final JdbcUtils.Connector connector;
    private final ViburConfig config;
    private final AtomicInteger version = new AtomicInteger(1);

    public ConnectionFactory(ViburConfig config) throws ViburDBCPException {
        this.connector = config.getConnector();
        this.config = config;
        JdbcUtils.initLoginTimeout(config);
    }

    public ConnHolder create() throws ViburDBCPException {
        return this.create(this.config.getUsername(), this.config.getPassword());
    }

    @Override
    public ConnHolder create(String username, String password) throws ViburDBCPException {
        int attempt = 0;
        Connection rawConnection = null;
        while (rawConnection == null) {
            try {
                rawConnection = Objects.requireNonNull(this.connector.connect());
            }
            catch (SQLException e) {
                logger.debug("Couldn't create a java.sql.Connection, attempt {}", (Object)attempt, (Object)e);
                if (attempt++ >= this.config.getAcquireRetryAttempts()) {
                    throw new ViburDBCPException(e);
                }
                try {
                    TimeUnit.MILLISECONDS.sleep(this.config.getAcquireRetryDelayInMs());
                }
                catch (InterruptedException interruptedException) {}
            }
        }
        try {
            if (this.config.getInitConnectionHook() != null) {
                this.config.getInitConnectionHook().on(rawConnection);
            }
            this.ensureInitialized(rawConnection);
            JdbcUtils.setDefaultValues(rawConnection, this.config);
        }
        catch (SQLException e) {
            JdbcUtils.quietClose(rawConnection);
            throw new ViburDBCPException(e);
        }
        logger.debug("Created {}", (Object)rawConnection);
        return this.prepareTracking(new ConnHolder(rawConnection, this.version(), this.config.getConnectionIdleLimitInSeconds() >= 0 ? System.currentTimeMillis() : 0L));
    }

    private void ensureInitialized(Connection rawConnection) throws SQLException {
        if (!JdbcUtils.validateConnection(rawConnection, this.config.getInitSQL(), this.config)) {
            throw new SQLException("Couldn't initialize " + rawConnection, "VI003");
        }
    }

    public boolean readyToTake(ConnHolder conn) {
        if (conn.version() != this.version()) {
            return false;
        }
        Connection rawConnection = conn.value();
        try {
            int idle;
            int idleLimit = this.config.getConnectionIdleLimitInSeconds();
            if (idleLimit >= 0 && (idle = (int)TimeUnit.MILLISECONDS.toSeconds(System.currentTimeMillis() - conn.getRestoredTime())) >= idleLimit && !JdbcUtils.validateConnection(rawConnection, this.config.getTestConnectionQuery(), this.config)) {
                return false;
            }
            if (this.config.getConnectionHook() != null) {
                this.config.getConnectionHook().on(rawConnection);
            }
            this.prepareTracking(conn);
            return true;
        }
        catch (SQLException e) {
            logger.debug("Couldn't validate {}", (Object)rawConnection, (Object)e);
            return false;
        }
    }

    public boolean readyToRestore(ConnHolder conn) {
        this.clearTracking(conn);
        Connection rawConnection = conn.value();
        try {
            if (this.config.getCloseConnectionHook() != null) {
                this.config.getCloseConnectionHook().on(rawConnection);
            }
            if (this.config.isClearSQLWarnings()) {
                JdbcUtils.clearWarnings(rawConnection);
            }
            if (this.config.isResetDefaultsAfterUse()) {
                JdbcUtils.setDefaultValues(rawConnection, this.config);
            }
            if (this.config.getConnectionIdleLimitInSeconds() >= 0) {
                conn.setRestoredTime(System.currentTimeMillis());
            }
            return true;
        }
        catch (SQLException e) {
            logger.debug("Couldn't reset {}", (Object)rawConnection, (Object)e);
            return false;
        }
    }

    private ConnHolder prepareTracking(ConnHolder conn) {
        if (this.config.isPoolEnableConnectionTracking()) {
            conn.setTakenTime(System.currentTimeMillis());
            conn.setThread(Thread.currentThread());
            conn.setLocation(new Throwable());
        }
        return conn;
    }

    private void clearTracking(ConnHolder conn) {
        if (this.config.isPoolEnableConnectionTracking()) {
            conn.setThread(null);
            conn.setLocation(null);
        }
    }

    public void destroy(ConnHolder conn) {
        Connection rawConnection = conn.value();
        logger.debug("Destroying {}", (Object)rawConnection);
        this.closeStatements(rawConnection);
        JdbcUtils.quietClose(rawConnection);
    }

    private void closeStatements(Connection rawConnection) {
        if (this.config.getStatementCache() != null) {
            this.config.getStatementCache().removeAll(rawConnection);
        }
    }

    @Override
    public int version() {
        return this.version.get();
    }

    @Override
    public boolean compareAndSetVersion(int expect, int update) {
        return this.version.compareAndSet(expect, update);
    }
}

