@file:Suppress("NAME_SHADOWING", "DEPRECATION")

package com.pulumi.azurenative.sql.kotlin

import com.pulumi.azurenative.sql.DatabaseArgs.builder
import com.pulumi.azurenative.sql.kotlin.enums.BackupStorageRedundancy
import com.pulumi.azurenative.sql.kotlin.enums.CatalogCollationType
import com.pulumi.azurenative.sql.kotlin.enums.CreateMode
import com.pulumi.azurenative.sql.kotlin.enums.DatabaseLicenseType
import com.pulumi.azurenative.sql.kotlin.enums.DatabaseReadScale
import com.pulumi.azurenative.sql.kotlin.enums.SampleName
import com.pulumi.azurenative.sql.kotlin.enums.SecondaryType
import com.pulumi.azurenative.sql.kotlin.inputs.DatabaseIdentityArgs
import com.pulumi.azurenative.sql.kotlin.inputs.DatabaseIdentityArgsBuilder
import com.pulumi.azurenative.sql.kotlin.inputs.SkuArgs
import com.pulumi.azurenative.sql.kotlin.inputs.SkuArgsBuilder
import com.pulumi.core.Either
import com.pulumi.core.Output
import com.pulumi.core.Output.of
import com.pulumi.kotlin.ConvertibleToJava
import com.pulumi.kotlin.PulumiTagMarker
import com.pulumi.kotlin.applySuspend
import kotlin.Boolean
import kotlin.Double
import kotlin.Int
import kotlin.Pair
import kotlin.String
import kotlin.Suppress
import kotlin.Unit
import kotlin.collections.Map
import kotlin.jvm.JvmName

/**
 * A database resource.
 * Azure REST API version: 2021-11-01. Prior API version in Azure Native 1.x: 2020-11-01-preview.
 * Other available API versions: 2014-04-01, 2019-06-01-preview, 2020-02-02-preview, 2020-08-01-preview, 2022-11-01-preview, 2023-02-01-preview, 2023-05-01-preview.
 * ## Example Usage
 * ### Creates a VCore database by specifying service objective name.
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using AzureNative = Pulumi.AzureNative;
 * return await Deployment.RunAsync(() =>
 * {
 *     var database = new AzureNative.Sql.Database("database", new()
 *     {
 *         DatabaseName = "testdb",
 *         Location = "southeastasia",
 *         ResourceGroupName = "Default-SQL-SouthEastAsia",
 *         ServerName = "testsvr",
 *         Sku = new AzureNative.Sql.Inputs.SkuArgs
 *         {
 *             Capacity = 2,
 *             Family = "Gen4",
 *             Name = "BC",
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-azure-native-sdk/sql/v2"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := sql.NewDatabase(ctx, "database", &sql.DatabaseArgs{
 * 			DatabaseName:      pulumi.String("testdb"),
 * 			Location:          pulumi.String("southeastasia"),
 * 			ResourceGroupName: pulumi.String("Default-SQL-SouthEastAsia"),
 * 			ServerName:        pulumi.String("testsvr"),
 * 			Sku: &sql.SkuArgs{
 * 				Capacity: pulumi.Int(2),
 * 				Family:   pulumi.String("Gen4"),
 * 				Name:     pulumi.String("BC"),
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.azurenative.sql.Database;
 * import com.pulumi.azurenative.sql.DatabaseArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var database = new Database("database", DatabaseArgs.builder()
 *             .databaseName("testdb")
 *             .location("southeastasia")
 *             .resourceGroupName("Default-SQL-SouthEastAsia")
 *             .serverName("testsvr")
 *             .sku(Map.ofEntries(
 *                 Map.entry("capacity", 2),
 *                 Map.entry("family", "Gen4"),
 *                 Map.entry("name", "BC")
 *             ))
 *             .build());
 *     }
 * }
 * ```
 * ### Creates a VCore database by specifying sku name and capacity.
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using AzureNative = Pulumi.AzureNative;
 * return await Deployment.RunAsync(() =>
 * {
 *     var database = new AzureNative.Sql.Database("database", new()
 *     {
 *         DatabaseName = "testdb",
 *         Location = "southeastasia",
 *         ResourceGroupName = "Default-SQL-SouthEastAsia",
 *         ServerName = "testsvr",
 *         Sku = new AzureNative.Sql.Inputs.SkuArgs
 *         {
 *             Capacity = 2,
 *             Name = "BC_Gen4",
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-azure-native-sdk/sql/v2"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := sql.NewDatabase(ctx, "database", &sql.DatabaseArgs{
 * 			DatabaseName:      pulumi.String("testdb"),
 * 			Location:          pulumi.String("southeastasia"),
 * 			ResourceGroupName: pulumi.String("Default-SQL-SouthEastAsia"),
 * 			ServerName:        pulumi.String("testsvr"),
 * 			Sku: &sql.SkuArgs{
 * 				Capacity: pulumi.Int(2),
 * 				Name:     pulumi.String("BC_Gen4"),
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.azurenative.sql.Database;
 * import com.pulumi.azurenative.sql.DatabaseArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var database = new Database("database", DatabaseArgs.builder()
 *             .databaseName("testdb")
 *             .location("southeastasia")
 *             .resourceGroupName("Default-SQL-SouthEastAsia")
 *             .serverName("testsvr")
 *             .sku(Map.ofEntries(
 *                 Map.entry("capacity", 2),
 *                 Map.entry("name", "BC_Gen4")
 *             ))
 *             .build());
 *     }
 * }
 * ```
 * ### Creates a data warehouse database as a cross-subscription restore from a backup of a dropped database.
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using AzureNative = Pulumi.AzureNative;
 * return await Deployment.RunAsync(() =>
 * {
 *     var database = new AzureNative.Sql.Database("database", new()
 *     {
 *         CreateMode = "Restore",
 *         DatabaseName = "testdw",
 *         Location = "southeastasia",
 *         ResourceGroupName = "Default-SQL-SouthEastAsia",
 *         ServerName = "testsvr",
 *         SourceResourceId = "/subscriptions/55555555-6666-7777-8888-999999999999/resourceGroups/Default-SQL-SouthEastAsia/providers/Microsoft.Sql/servers/srcsvr/restorableDroppedDatabases/srcdw,131403269876900000",
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-azure-native-sdk/sql/v2"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := sql.NewDatabase(ctx, "database", &sql.DatabaseArgs{
 * 			CreateMode:        pulumi.String("Restore"),
 * 			DatabaseName:      pulumi.String("testdw"),
 * 			Location:          pulumi.String("southeastasia"),
 * 			ResourceGroupName: pulumi.String("Default-SQL-SouthEastAsia"),
 * 			ServerName:        pulumi.String("testsvr"),
 * 			SourceResourceId:  pulumi.String("/subscriptions/55555555-6666-7777-8888-999999999999/resourceGroups/Default-SQL-SouthEastAsia/providers/Microsoft.Sql/servers/srcsvr/restorableDroppedDatabases/srcdw,131403269876900000"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.azurenative.sql.Database;
 * import com.pulumi.azurenative.sql.DatabaseArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var database = new Database("database", DatabaseArgs.builder()
 *             .createMode("Restore")
 *             .databaseName("testdw")
 *             .location("southeastasia")
 *             .resourceGroupName("Default-SQL-SouthEastAsia")
 *             .serverName("testsvr")
 *             .sourceResourceId("/subscriptions/55555555-6666-7777-8888-999999999999/resourceGroups/Default-SQL-SouthEastAsia/providers/Microsoft.Sql/servers/srcsvr/restorableDroppedDatabases/srcdw,131403269876900000")
 *             .build());
 *     }
 * }
 * ```
 * ### Creates a data warehouse database as a cross-subscription restore from a geo-backup.
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using AzureNative = Pulumi.AzureNative;
 * return await Deployment.RunAsync(() =>
 * {
 *     var database = new AzureNative.Sql.Database("database", new()
 *     {
 *         CreateMode = "Recovery",
 *         DatabaseName = "testdw",
 *         Location = "westus",
 *         ResourceGroupName = "Default-SQL-WestUS",
 *         ServerName = "testsvr",
 *         SourceResourceId = "/subscriptions/55555555-6666-7777-8888-999999999999/resourceGroups/Default-SQL-EastUS/providers/Microsoft.Sql/servers/srcsvr/recoverabledatabases/srcdw",
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-azure-native-sdk/sql/v2"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := sql.NewDatabase(ctx, "database", &sql.DatabaseArgs{
 * 			CreateMode:        pulumi.String("Recovery"),
 * 			DatabaseName:      pulumi.String("testdw"),
 * 			Location:          pulumi.String("westus"),
 * 			ResourceGroupName: pulumi.String("Default-SQL-WestUS"),
 * 			ServerName:        pulumi.String("testsvr"),
 * 			SourceResourceId:  pulumi.String("/subscriptions/55555555-6666-7777-8888-999999999999/resourceGroups/Default-SQL-EastUS/providers/Microsoft.Sql/servers/srcsvr/recoverabledatabases/srcdw"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.azurenative.sql.Database;
 * import com.pulumi.azurenative.sql.DatabaseArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var database = new Database("database", DatabaseArgs.builder()
 *             .createMode("Recovery")
 *             .databaseName("testdw")
 *             .location("westus")
 *             .resourceGroupName("Default-SQL-WestUS")
 *             .serverName("testsvr")
 *             .sourceResourceId("/subscriptions/55555555-6666-7777-8888-999999999999/resourceGroups/Default-SQL-EastUS/providers/Microsoft.Sql/servers/srcsvr/recoverabledatabases/srcdw")
 *             .build());
 *     }
 * }
 * ```
 * ### Creates a data warehouse database as a cross-subscription restore from a restore point of an existing database.
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using AzureNative = Pulumi.AzureNative;
 * return await Deployment.RunAsync(() =>
 * {
 *     var database = new AzureNative.Sql.Database("database", new()
 *     {
 *         CreateMode = "PointInTimeRestore",
 *         DatabaseName = "testdw",
 *         Location = "southeastasia",
 *         ResourceGroupName = "Default-SQL-SouthEastAsia",
 *         RestorePointInTime = "2022-01-22T05:35:31.503Z",
 *         ServerName = "testsvr",
 *         SourceResourceId = "/subscriptions/55555555-6666-7777-8888-999999999999/resourceGroups/Default-SQL-SouthEastAsia/providers/Microsoft.Sql/servers/srcsvr/databases/srcdw",
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-azure-native-sdk/sql/v2"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := sql.NewDatabase(ctx, "database", &sql.DatabaseArgs{
 * 			CreateMode:         pulumi.String("PointInTimeRestore"),
 * 			DatabaseName:       pulumi.String("testdw"),
 * 			Location:           pulumi.String("southeastasia"),
 * 			ResourceGroupName:  pulumi.String("Default-SQL-SouthEastAsia"),
 * 			RestorePointInTime: pulumi.String("2022-01-22T05:35:31.503Z"),
 * 			ServerName:         pulumi.String("testsvr"),
 * 			SourceResourceId:   pulumi.String("/subscriptions/55555555-6666-7777-8888-999999999999/resourceGroups/Default-SQL-SouthEastAsia/providers/Microsoft.Sql/servers/srcsvr/databases/srcdw"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.azurenative.sql.Database;
 * import com.pulumi.azurenative.sql.DatabaseArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var database = new Database("database", DatabaseArgs.builder()
 *             .createMode("PointInTimeRestore")
 *             .databaseName("testdw")
 *             .location("southeastasia")
 *             .resourceGroupName("Default-SQL-SouthEastAsia")
 *             .restorePointInTime("2022-01-22T05:35:31.503Z")
 *             .serverName("testsvr")
 *             .sourceResourceId("/subscriptions/55555555-6666-7777-8888-999999999999/resourceGroups/Default-SQL-SouthEastAsia/providers/Microsoft.Sql/servers/srcsvr/databases/srcdw")
 *             .build());
 *     }
 * }
 * ```
 * ### Creates a database as a copy.
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using AzureNative = Pulumi.AzureNative;
 * return await Deployment.RunAsync(() =>
 * {
 *     var database = new AzureNative.Sql.Database("database", new()
 *     {
 *         CreateMode = "Copy",
 *         DatabaseName = "dbcopy",
 *         Location = "southeastasia",
 *         ResourceGroupName = "Default-SQL-SouthEastAsia",
 *         ServerName = "testsvr",
 *         Sku = new AzureNative.Sql.Inputs.SkuArgs
 *         {
 *             Name = "S0",
 *             Tier = "Standard",
 *         },
 *         SourceDatabaseId = "/subscriptions/00000000-1111-2222-3333-444444444444/resourceGroups/Default-SQL-SouthEastAsia/providers/Microsoft.Sql/servers/testsvr/databases/testdb",
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-azure-native-sdk/sql/v2"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := sql.NewDatabase(ctx, "database", &sql.DatabaseArgs{
 * 			CreateMode:        pulumi.String("Copy"),
 * 			DatabaseName:      pulumi.String("dbcopy"),
 * 			Location:          pulumi.String("southeastasia"),
 * 			ResourceGroupName: pulumi.String("Default-SQL-SouthEastAsia"),
 * 			ServerName:        pulumi.String("testsvr"),
 * 			Sku: &sql.SkuArgs{
 * 				Name: pulumi.String("S0"),
 * 				Tier: pulumi.String("Standard"),
 * 			},
 * 			SourceDatabaseId: pulumi.String("/subscriptions/00000000-1111-2222-3333-444444444444/resourceGroups/Default-SQL-SouthEastAsia/providers/Microsoft.Sql/servers/testsvr/databases/testdb"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.azurenative.sql.Database;
 * import com.pulumi.azurenative.sql.DatabaseArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var database = new Database("database", DatabaseArgs.builder()
 *             .createMode("Copy")
 *             .databaseName("dbcopy")
 *             .location("southeastasia")
 *             .resourceGroupName("Default-SQL-SouthEastAsia")
 *             .serverName("testsvr")
 *             .sku(Map.ofEntries(
 *                 Map.entry("name", "S0"),
 *                 Map.entry("tier", "Standard")
 *             ))
 *             .sourceDatabaseId("/subscriptions/00000000-1111-2222-3333-444444444444/resourceGroups/Default-SQL-SouthEastAsia/providers/Microsoft.Sql/servers/testsvr/databases/testdb")
 *             .build());
 *     }
 * }
 * ```
 * ### Creates a database as an on-line secondary.
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using AzureNative = Pulumi.AzureNative;
 * return await Deployment.RunAsync(() =>
 * {
 *     var database = new AzureNative.Sql.Database("database", new()
 *     {
 *         CreateMode = "Secondary",
 *         DatabaseName = "testdb",
 *         Location = "southeastasia",
 *         ResourceGroupName = "Default-SQL-SouthEastAsia",
 *         SecondaryType = "Geo",
 *         ServerName = "testsvr",
 *         Sku = new AzureNative.Sql.Inputs.SkuArgs
 *         {
 *             Name = "S0",
 *             Tier = "Standard",
 *         },
 *         SourceDatabaseId = "/subscriptions/00000000-1111-2222-3333-444444444444/resourceGroups/Default-SQL-NorthEurope/providers/Microsoft.Sql/servers/testsvr1/databases/testdb",
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-azure-native-sdk/sql/v2"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := sql.NewDatabase(ctx, "database", &sql.DatabaseArgs{
 * 			CreateMode:        pulumi.String("Secondary"),
 * 			DatabaseName:      pulumi.String("testdb"),
 * 			Location:          pulumi.String("southeastasia"),
 * 			ResourceGroupName: pulumi.String("Default-SQL-SouthEastAsia"),
 * 			SecondaryType:     pulumi.String("Geo"),
 * 			ServerName:        pulumi.String("testsvr"),
 * 			Sku: &sql.SkuArgs{
 * 				Name: pulumi.String("S0"),
 * 				Tier: pulumi.String("Standard"),
 * 			},
 * 			SourceDatabaseId: pulumi.String("/subscriptions/00000000-1111-2222-3333-444444444444/resourceGroups/Default-SQL-NorthEurope/providers/Microsoft.Sql/servers/testsvr1/databases/testdb"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.azurenative.sql.Database;
 * import com.pulumi.azurenative.sql.DatabaseArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var database = new Database("database", DatabaseArgs.builder()
 *             .createMode("Secondary")
 *             .databaseName("testdb")
 *             .location("southeastasia")
 *             .resourceGroupName("Default-SQL-SouthEastAsia")
 *             .secondaryType("Geo")
 *             .serverName("testsvr")
 *             .sku(Map.ofEntries(
 *                 Map.entry("name", "S0"),
 *                 Map.entry("tier", "Standard")
 *             ))
 *             .sourceDatabaseId("/subscriptions/00000000-1111-2222-3333-444444444444/resourceGroups/Default-SQL-NorthEurope/providers/Microsoft.Sql/servers/testsvr1/databases/testdb")
 *             .build());
 *     }
 * }
 * ```
 * ### Creates a database as named replica secondary.
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using AzureNative = Pulumi.AzureNative;
 * return await Deployment.RunAsync(() =>
 * {
 *     var database = new AzureNative.Sql.Database("database", new()
 *     {
 *         CreateMode = "Secondary",
 *         DatabaseName = "testdb",
 *         Location = "southeastasia",
 *         ResourceGroupName = "Default-SQL-SouthEastAsia",
 *         SecondaryType = "Named",
 *         ServerName = "testsvr",
 *         Sku = new AzureNative.Sql.Inputs.SkuArgs
 *         {
 *             Capacity = 2,
 *             Name = "HS_Gen4",
 *             Tier = "Hyperscale",
 *         },
 *         SourceDatabaseId = "/subscriptions/00000000-1111-2222-3333-444444444444/resourceGroups/Default-SQL-NorthEurope/providers/Microsoft.Sql/servers/testsvr1/databases/primarydb",
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-azure-native-sdk/sql/v2"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := sql.NewDatabase(ctx, "database", &sql.DatabaseArgs{
 * 			CreateMode:        pulumi.String("Secondary"),
 * 			DatabaseName:      pulumi.String("testdb"),
 * 			Location:          pulumi.String("southeastasia"),
 * 			ResourceGroupName: pulumi.String("Default-SQL-SouthEastAsia"),
 * 			SecondaryType:     pulumi.String("Named"),
 * 			ServerName:        pulumi.String("testsvr"),
 * 			Sku: &sql.SkuArgs{
 * 				Capacity: pulumi.Int(2),
 * 				Name:     pulumi.String("HS_Gen4"),
 * 				Tier:     pulumi.String("Hyperscale"),
 * 			},
 * 			SourceDatabaseId: pulumi.String("/subscriptions/00000000-1111-2222-3333-444444444444/resourceGroups/Default-SQL-NorthEurope/providers/Microsoft.Sql/servers/testsvr1/databases/primarydb"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.azurenative.sql.Database;
 * import com.pulumi.azurenative.sql.DatabaseArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var database = new Database("database", DatabaseArgs.builder()
 *             .createMode("Secondary")
 *             .databaseName("testdb")
 *             .location("southeastasia")
 *             .resourceGroupName("Default-SQL-SouthEastAsia")
 *             .secondaryType("Named")
 *             .serverName("testsvr")
 *             .sku(Map.ofEntries(
 *                 Map.entry("capacity", 2),
 *                 Map.entry("name", "HS_Gen4"),
 *                 Map.entry("tier", "Hyperscale")
 *             ))
 *             .sourceDatabaseId("/subscriptions/00000000-1111-2222-3333-444444444444/resourceGroups/Default-SQL-NorthEurope/providers/Microsoft.Sql/servers/testsvr1/databases/primarydb")
 *             .build());
 *     }
 * }
 * ```
 * ### Creates a database from PointInTimeRestore.
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using AzureNative = Pulumi.AzureNative;
 * return await Deployment.RunAsync(() =>
 * {
 *     var database = new AzureNative.Sql.Database("database", new()
 *     {
 *         CreateMode = "PointInTimeRestore",
 *         DatabaseName = "dbpitr",
 *         Location = "southeastasia",
 *         ResourceGroupName = "Default-SQL-SouthEastAsia",
 *         RestorePointInTime = "2020-10-22T05:35:31.503Z",
 *         ServerName = "testsvr",
 *         SourceDatabaseId = "/subscriptions/00000000-1111-2222-3333-444444444444/resourceGroups/Default-SQL-SoutheastAsia/providers/Microsoft.Sql/servers/testsvr/databases/testdb",
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-azure-native-sdk/sql/v2"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := sql.NewDatabase(ctx, "database", &sql.DatabaseArgs{
 * 			CreateMode:         pulumi.String("PointInTimeRestore"),
 * 			DatabaseName:       pulumi.String("dbpitr"),
 * 			Location:           pulumi.String("southeastasia"),
 * 			ResourceGroupName:  pulumi.String("Default-SQL-SouthEastAsia"),
 * 			RestorePointInTime: pulumi.String("2020-10-22T05:35:31.503Z"),
 * 			ServerName:         pulumi.String("testsvr"),
 * 			SourceDatabaseId:   pulumi.String("/subscriptions/00000000-1111-2222-3333-444444444444/resourceGroups/Default-SQL-SoutheastAsia/providers/Microsoft.Sql/servers/testsvr/databases/testdb"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.azurenative.sql.Database;
 * import com.pulumi.azurenative.sql.DatabaseArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var database = new Database("database", DatabaseArgs.builder()
 *             .createMode("PointInTimeRestore")
 *             .databaseName("dbpitr")
 *             .location("southeastasia")
 *             .resourceGroupName("Default-SQL-SouthEastAsia")
 *             .restorePointInTime("2020-10-22T05:35:31.503Z")
 *             .serverName("testsvr")
 *             .sourceDatabaseId("/subscriptions/00000000-1111-2222-3333-444444444444/resourceGroups/Default-SQL-SoutheastAsia/providers/Microsoft.Sql/servers/testsvr/databases/testdb")
 *             .build());
 *     }
 * }
 * ```
 * ### Creates a database with default mode.
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using AzureNative = Pulumi.AzureNative;
 * return await Deployment.RunAsync(() =>
 * {
 *     var database = new AzureNative.Sql.Database("database", new()
 *     {
 *         Collation = "SQL_Latin1_General_CP1_CI_AS",
 *         CreateMode = "Default",
 *         DatabaseName = "testdb",
 *         Location = "southeastasia",
 *         MaxSizeBytes = 1073741824,
 *         ResourceGroupName = "Default-SQL-SouthEastAsia",
 *         ServerName = "testsvr",
 *         Sku = new AzureNative.Sql.Inputs.SkuArgs
 *         {
 *             Name = "S0",
 *             Tier = "Standard",
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-azure-native-sdk/sql/v2"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := sql.NewDatabase(ctx, "database", &sql.DatabaseArgs{
 * 			Collation:         pulumi.String("SQL_Latin1_General_CP1_CI_AS"),
 * 			CreateMode:        pulumi.String("Default"),
 * 			DatabaseName:      pulumi.String("testdb"),
 * 			Location:          pulumi.String("southeastasia"),
 * 			MaxSizeBytes:      pulumi.Float64(1073741824),
 * 			ResourceGroupName: pulumi.String("Default-SQL-SouthEastAsia"),
 * 			ServerName:        pulumi.String("testsvr"),
 * 			Sku: &sql.SkuArgs{
 * 				Name: pulumi.String("S0"),
 * 				Tier: pulumi.String("Standard"),
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.azurenative.sql.Database;
 * import com.pulumi.azurenative.sql.DatabaseArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var database = new Database("database", DatabaseArgs.builder()
 *             .collation("SQL_Latin1_General_CP1_CI_AS")
 *             .createMode("Default")
 *             .databaseName("testdb")
 *             .location("southeastasia")
 *             .maxSizeBytes(1073741824)
 *             .resourceGroupName("Default-SQL-SouthEastAsia")
 *             .serverName("testsvr")
 *             .sku(Map.ofEntries(
 *                 Map.entry("name", "S0"),
 *                 Map.entry("tier", "Standard")
 *             ))
 *             .build());
 *     }
 * }
 * ```
 * ### Creates a database with ledger on.
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using AzureNative = Pulumi.AzureNative;
 * return await Deployment.RunAsync(() =>
 * {
 *     var database = new AzureNative.Sql.Database("database", new()
 *     {
 *         DatabaseName = "testdb",
 *         IsLedgerOn = true,
 *         Location = "southeastasia",
 *         ResourceGroupName = "Default-SQL-SouthEastAsia",
 *         ServerName = "testsvr",
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-azure-native-sdk/sql/v2"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := sql.NewDatabase(ctx, "database", &sql.DatabaseArgs{
 * 			DatabaseName:      pulumi.String("testdb"),
 * 			IsLedgerOn:        pulumi.Bool(true),
 * 			Location:          pulumi.String("southeastasia"),
 * 			ResourceGroupName: pulumi.String("Default-SQL-SouthEastAsia"),
 * 			ServerName:        pulumi.String("testsvr"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.azurenative.sql.Database;
 * import com.pulumi.azurenative.sql.DatabaseArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var database = new Database("database", DatabaseArgs.builder()
 *             .databaseName("testdb")
 *             .isLedgerOn(true)
 *             .location("southeastasia")
 *             .resourceGroupName("Default-SQL-SouthEastAsia")
 *             .serverName("testsvr")
 *             .build());
 *     }
 * }
 * ```
 * ### Creates a database with minimum number of parameters.
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using AzureNative = Pulumi.AzureNative;
 * return await Deployment.RunAsync(() =>
 * {
 *     var database = new AzureNative.Sql.Database("database", new()
 *     {
 *         DatabaseName = "testdb",
 *         Location = "southeastasia",
 *         ResourceGroupName = "Default-SQL-SouthEastAsia",
 *         ServerName = "testsvr",
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-azure-native-sdk/sql/v2"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := sql.NewDatabase(ctx, "database", &sql.DatabaseArgs{
 * 			DatabaseName:      pulumi.String("testdb"),
 * 			Location:          pulumi.String("southeastasia"),
 * 			ResourceGroupName: pulumi.String("Default-SQL-SouthEastAsia"),
 * 			ServerName:        pulumi.String("testsvr"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.azurenative.sql.Database;
 * import com.pulumi.azurenative.sql.DatabaseArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var database = new Database("database", DatabaseArgs.builder()
 *             .databaseName("testdb")
 *             .location("southeastasia")
 *             .resourceGroupName("Default-SQL-SouthEastAsia")
 *             .serverName("testsvr")
 *             .build());
 *     }
 * }
 * ```
 * ### Creates a database with preferred maintenance window.
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using AzureNative = Pulumi.AzureNative;
 * return await Deployment.RunAsync(() =>
 * {
 *     var database = new AzureNative.Sql.Database("database", new()
 *     {
 *         Collation = "SQL_Latin1_General_CP1_CI_AS",
 *         CreateMode = "Default",
 *         DatabaseName = "testdb",
 *         Location = "southeastasia",
 *         MaintenanceConfigurationId = "/subscriptions/00000000-1111-2222-3333-444444444444/providers/Microsoft.Maintenance/publicMaintenanceConfigurations/SQL_SouthEastAsia_1",
 *         MaxSizeBytes = 1073741824,
 *         ResourceGroupName = "Default-SQL-SouthEastAsia",
 *         ServerName = "testsvr",
 *         Sku = new AzureNative.Sql.Inputs.SkuArgs
 *         {
 *             Name = "S2",
 *             Tier = "Standard",
 *         },
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-azure-native-sdk/sql/v2"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := sql.NewDatabase(ctx, "database", &sql.DatabaseArgs{
 * 			Collation:                  pulumi.String("SQL_Latin1_General_CP1_CI_AS"),
 * 			CreateMode:                 pulumi.String("Default"),
 * 			DatabaseName:               pulumi.String("testdb"),
 * 			Location:                   pulumi.String("southeastasia"),
 * 			MaintenanceConfigurationId: pulumi.String("/subscriptions/00000000-1111-2222-3333-444444444444/providers/Microsoft.Maintenance/publicMaintenanceConfigurations/SQL_SouthEastAsia_1"),
 * 			MaxSizeBytes:               pulumi.Float64(1073741824),
 * 			ResourceGroupName:          pulumi.String("Default-SQL-SouthEastAsia"),
 * 			ServerName:                 pulumi.String("testsvr"),
 * 			Sku: &sql.SkuArgs{
 * 				Name: pulumi.String("S2"),
 * 				Tier: pulumi.String("Standard"),
 * 			},
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.azurenative.sql.Database;
 * import com.pulumi.azurenative.sql.DatabaseArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var database = new Database("database", DatabaseArgs.builder()
 *             .collation("SQL_Latin1_General_CP1_CI_AS")
 *             .createMode("Default")
 *             .databaseName("testdb")
 *             .location("southeastasia")
 *             .maintenanceConfigurationId("/subscriptions/00000000-1111-2222-3333-444444444444/providers/Microsoft.Maintenance/publicMaintenanceConfigurations/SQL_SouthEastAsia_1")
 *             .maxSizeBytes(1073741824)
 *             .resourceGroupName("Default-SQL-SouthEastAsia")
 *             .serverName("testsvr")
 *             .sku(Map.ofEntries(
 *                 Map.entry("name", "S2"),
 *                 Map.entry("tier", "Standard")
 *             ))
 *             .build());
 *     }
 * }
 * ```
 * ### Creates a database with specified backup storage redundancy.
 * ```csharp
 * using System.Collections.Generic;
 * using System.Linq;
 * using Pulumi;
 * using AzureNative = Pulumi.AzureNative;
 * return await Deployment.RunAsync(() =>
 * {
 *     var database = new AzureNative.Sql.Database("database", new()
 *     {
 *         DatabaseName = "testdb",
 *         Location = "southeastasia",
 *         RequestedBackupStorageRedundancy = "Zone",
 *         ResourceGroupName = "Default-SQL-SouthEastAsia",
 *         ServerName = "testsvr",
 *     });
 * });
 * ```
 * ```go
 * package main
 * import (
 * 	"github.com/pulumi/pulumi-azure-native-sdk/sql/v2"
 * 	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
 * )
 * func main() {
 * 	pulumi.Run(func(ctx *pulumi.Context) error {
 * 		_, err := sql.NewDatabase(ctx, "database", &sql.DatabaseArgs{
 * 			DatabaseName:                     pulumi.String("testdb"),
 * 			Location:                         pulumi.String("southeastasia"),
 * 			RequestedBackupStorageRedundancy: pulumi.String("Zone"),
 * 			ResourceGroupName:                pulumi.String("Default-SQL-SouthEastAsia"),
 * 			ServerName:                       pulumi.String("testsvr"),
 * 		})
 * 		if err != nil {
 * 			return err
 * 		}
 * 		return nil
 * 	})
 * }
 * ```
 * ```java
 * package generated_program;
 * import com.pulumi.Context;
 * import com.pulumi.Pulumi;
 * import com.pulumi.core.Output;
 * import com.pulumi.azurenative.sql.Database;
 * import com.pulumi.azurenative.sql.DatabaseArgs;
 * import java.util.List;
 * import java.util.ArrayList;
 * import java.util.Map;
 * import java.io.File;
 * import java.nio.file.Files;
 * import java.nio.file.Paths;
 * public class App {
 *     public static void main(String[] args) {
 *         Pulumi.run(App::stack);
 *     }
 *     public static void stack(Context ctx) {
 *         var database = new Database("database", DatabaseArgs.builder()
 *             .databaseName("testdb")
 *             .location("southeastasia")
 *             .requestedBackupStorageRedundancy("Zone")
 *             .resourceGroupName("Default-SQL-SouthEastAsia")
 *             .serverName("testsvr")
 *             .build());
 *     }
 * }
 * ```
 * ## Import
 * An existing resource can be imported using its type token, name, and identifier, e.g.
 * ```sh
 * $ pulumi import azure-native:sql:Database testdb /subscriptions/{subscriptionId}/resourceGroups/{resourceGroupName}/providers/Microsoft.Sql/servers/{serverName}/databases/{databaseName}
 * ```
 * @property autoPauseDelay Time in minutes after which database is automatically paused. A value of -1 means that automatic pause is disabled
 * @property catalogCollation Collation of the metadata catalog.
 * @property collation The collation of the database.
 * @property createMode Specifies the mode of database creation.
 * Default: regular database creation.
 * Copy: creates a database as a copy of an existing database. sourceDatabaseId must be specified as the resource ID of the source database.
 * Secondary: creates a database as a secondary replica of an existing database. sourceDatabaseId must be specified as the resource ID of the existing primary database.
 * PointInTimeRestore: Creates a database by restoring a point in time backup of an existing database. sourceDatabaseId must be specified as the resource ID of the existing database, and restorePointInTime must be specified.
 * Recovery: Creates a database by restoring a geo-replicated backup. sourceDatabaseId must be specified as the recoverable database resource ID to restore.
 * Restore: Creates a database by restoring a backup of a deleted database. sourceDatabaseId must be specified. If sourceDatabaseId is the database's original resource ID, then sourceDatabaseDeletionDate must be specified. Otherwise sourceDatabaseId must be the restorable dropped database resource ID and sourceDatabaseDeletionDate is ignored. restorePointInTime may also be specified to restore from an earlier point in time.
 * RestoreLongTermRetentionBackup: Creates a database by restoring from a long term retention vault. recoveryServicesRecoveryPointResourceId must be specified as the recovery point resource ID.
 * Copy, Secondary, and RestoreLongTermRetentionBackup are not supported for DataWarehouse edition.
 * @property databaseName The name of the database.
 * @property elasticPoolId The resource identifier of the elastic pool containing this database.
 * @property federatedClientId The Client id used for cross tenant per database CMK scenario
 * @property highAvailabilityReplicaCount The number of secondary replicas associated with the database that are used to provide high availability. Not applicable to a Hyperscale database within an elastic pool.
 * @property identity The Azure Active Directory identity of the database.
 * @property isLedgerOn Whether or not this database is a ledger database, which means all tables in the database are ledger tables. Note: the value of this property cannot be changed after the database has been created.
 * @property licenseType The license type to apply for this database. `LicenseIncluded` if you need a license, or `BasePrice` if you have a license and are eligible for the Azure Hybrid Benefit.
 * @property location Resource location.
 * @property longTermRetentionBackupResourceId The resource identifier of the long term retention backup associated with create operation of this database.
 * @property maintenanceConfigurationId Maintenance configuration id assigned to the database. This configuration defines the period when the maintenance updates will occur.
 * @property maxSizeBytes The max size of the database expressed in bytes.
 * @property minCapacity Minimal capacity that database will always have allocated, if not paused
 * @property readScale The state of read-only routing. If enabled, connections that have application intent set to readonly in their connection string may be routed to a readonly secondary replica in the same region. Not applicable to a Hyperscale database within an elastic pool.
 * @property recoverableDatabaseId The resource identifier of the recoverable database associated with create operation of this database.
 * @property recoveryServicesRecoveryPointId The resource identifier of the recovery point associated with create operation of this database.
 * @property requestedBackupStorageRedundancy The storage account type to be used to store backups for this database.
 * @property resourceGroupName The name of the resource group that contains the resource. You can obtain this value from the Azure Resource Manager API or the portal.
 * @property restorableDroppedDatabaseId The resource identifier of the restorable dropped database associated with create operation of this database.
 * @property restorePointInTime Specifies the point in time (ISO8601 format) of the source database that will be restored to create the new database.
 * @property sampleName The name of the sample schema to apply when creating this database.
 * @property secondaryType The secondary type of the database if it is a secondary.  Valid values are Geo and Named.
 * @property serverName The name of the server.
 * @property sku The database SKU.
 * The list of SKUs may vary by region and support offer. To determine the SKUs (including the SKU name, tier/edition, family, and capacity) that are available to your subscription in an Azure region, use the `Capabilities_ListByLocation` REST API or one of the following commands:
 * ```azurecli
 * az sql db list-editions -l <location> -o table
 * ````
 * ```powershell
 * Get-AzSqlServerServiceObjective -Location <location>
 * ````
 * @property sourceDatabaseDeletionDate Specifies the time that the database was deleted.
 * @property sourceDatabaseId The resource identifier of the source database associated with create operation of this database.
 * @property sourceResourceId The resource identifier of the source associated with the create operation of this database.
 * This property is only supported for DataWarehouse edition and allows to restore across subscriptions.
 * When sourceResourceId is specified, sourceDatabaseId, recoverableDatabaseId, restorableDroppedDatabaseId and sourceDatabaseDeletionDate must not be specified and CreateMode must be PointInTimeRestore, Restore or Recover.
 * When createMode is PointInTimeRestore, sourceResourceId must be the resource ID of the existing database or existing sql pool, and restorePointInTime must be specified.
 * When createMode is Restore, sourceResourceId must be the resource ID of restorable dropped database or restorable dropped sql pool.
 * When createMode is Recover, sourceResourceId must be the resource ID of recoverable database or recoverable sql pool.
 * When source subscription belongs to a different tenant than target subscription, “x-ms-authorization-auxiliary” header must contain authentication token for the source tenant. For more details about “x-ms-authorization-auxiliary” header see https://docs.microsoft.com/en-us/azure/azure-resource-manager/management/authenticate-multi-tenant
 * @property tags Resource tags.
 * @property zoneRedundant Whether or not this database is zone redundant, which means the replicas of this database will be spread across multiple availability zones.
 */
public data class DatabaseArgs(
    public val autoPauseDelay: Output<Int>? = null,
    public val catalogCollation: Output<Either<String, CatalogCollationType>>? = null,
    public val collation: Output<String>? = null,
    public val createMode: Output<Either<String, CreateMode>>? = null,
    public val databaseName: Output<String>? = null,
    public val elasticPoolId: Output<String>? = null,
    public val federatedClientId: Output<String>? = null,
    public val highAvailabilityReplicaCount: Output<Int>? = null,
    public val identity: Output<DatabaseIdentityArgs>? = null,
    public val isLedgerOn: Output<Boolean>? = null,
    public val licenseType: Output<Either<String, DatabaseLicenseType>>? = null,
    public val location: Output<String>? = null,
    public val longTermRetentionBackupResourceId: Output<String>? = null,
    public val maintenanceConfigurationId: Output<String>? = null,
    public val maxSizeBytes: Output<Double>? = null,
    public val minCapacity: Output<Double>? = null,
    public val readScale: Output<Either<String, DatabaseReadScale>>? = null,
    public val recoverableDatabaseId: Output<String>? = null,
    public val recoveryServicesRecoveryPointId: Output<String>? = null,
    public val requestedBackupStorageRedundancy: Output<Either<String, BackupStorageRedundancy>>? =
        null,
    public val resourceGroupName: Output<String>? = null,
    public val restorableDroppedDatabaseId: Output<String>? = null,
    public val restorePointInTime: Output<String>? = null,
    public val sampleName: Output<Either<String, SampleName>>? = null,
    public val secondaryType: Output<Either<String, SecondaryType>>? = null,
    public val serverName: Output<String>? = null,
    public val sku: Output<SkuArgs>? = null,
    public val sourceDatabaseDeletionDate: Output<String>? = null,
    public val sourceDatabaseId: Output<String>? = null,
    public val sourceResourceId: Output<String>? = null,
    public val tags: Output<Map<String, String>>? = null,
    public val zoneRedundant: Output<Boolean>? = null,
) : ConvertibleToJava<com.pulumi.azurenative.sql.DatabaseArgs> {
    override fun toJava(): com.pulumi.azurenative.sql.DatabaseArgs =
        com.pulumi.azurenative.sql.DatabaseArgs.builder()
            .autoPauseDelay(autoPauseDelay?.applyValue({ args0 -> args0 }))
            .catalogCollation(
                catalogCollation?.applyValue({ args0 ->
                    args0.transform(
                        { args0 -> args0 },
                        { args0 -> args0.let({ args0 -> args0.toJava() }) },
                    )
                }),
            )
            .collation(collation?.applyValue({ args0 -> args0 }))
            .createMode(
                createMode?.applyValue({ args0 ->
                    args0.transform({ args0 -> args0 }, { args0 ->
                        args0.let({ args0 -> args0.toJava() })
                    })
                }),
            )
            .databaseName(databaseName?.applyValue({ args0 -> args0 }))
            .elasticPoolId(elasticPoolId?.applyValue({ args0 -> args0 }))
            .federatedClientId(federatedClientId?.applyValue({ args0 -> args0 }))
            .highAvailabilityReplicaCount(highAvailabilityReplicaCount?.applyValue({ args0 -> args0 }))
            .identity(identity?.applyValue({ args0 -> args0.let({ args0 -> args0.toJava() }) }))
            .isLedgerOn(isLedgerOn?.applyValue({ args0 -> args0 }))
            .licenseType(
                licenseType?.applyValue({ args0 ->
                    args0.transform({ args0 -> args0 }, { args0 ->
                        args0.let({ args0 -> args0.toJava() })
                    })
                }),
            )
            .location(location?.applyValue({ args0 -> args0 }))
            .longTermRetentionBackupResourceId(
                longTermRetentionBackupResourceId?.applyValue({ args0 ->
                    args0
                }),
            )
            .maintenanceConfigurationId(maintenanceConfigurationId?.applyValue({ args0 -> args0 }))
            .maxSizeBytes(maxSizeBytes?.applyValue({ args0 -> args0 }))
            .minCapacity(minCapacity?.applyValue({ args0 -> args0 }))
            .readScale(
                readScale?.applyValue({ args0 ->
                    args0.transform({ args0 -> args0 }, { args0 ->
                        args0.let({ args0 -> args0.toJava() })
                    })
                }),
            )
            .recoverableDatabaseId(recoverableDatabaseId?.applyValue({ args0 -> args0 }))
            .recoveryServicesRecoveryPointId(recoveryServicesRecoveryPointId?.applyValue({ args0 -> args0 }))
            .requestedBackupStorageRedundancy(
                requestedBackupStorageRedundancy?.applyValue({ args0 ->
                    args0.transform({ args0 -> args0 }, { args0 -> args0.let({ args0 -> args0.toJava() }) })
                }),
            )
            .resourceGroupName(resourceGroupName?.applyValue({ args0 -> args0 }))
            .restorableDroppedDatabaseId(restorableDroppedDatabaseId?.applyValue({ args0 -> args0 }))
            .restorePointInTime(restorePointInTime?.applyValue({ args0 -> args0 }))
            .sampleName(
                sampleName?.applyValue({ args0 ->
                    args0.transform({ args0 -> args0 }, { args0 ->
                        args0.let({ args0 -> args0.toJava() })
                    })
                }),
            )
            .secondaryType(
                secondaryType?.applyValue({ args0 ->
                    args0.transform({ args0 -> args0 }, { args0 ->
                        args0.let({ args0 -> args0.toJava() })
                    })
                }),
            )
            .serverName(serverName?.applyValue({ args0 -> args0 }))
            .sku(sku?.applyValue({ args0 -> args0.let({ args0 -> args0.toJava() }) }))
            .sourceDatabaseDeletionDate(sourceDatabaseDeletionDate?.applyValue({ args0 -> args0 }))
            .sourceDatabaseId(sourceDatabaseId?.applyValue({ args0 -> args0 }))
            .sourceResourceId(sourceResourceId?.applyValue({ args0 -> args0 }))
            .tags(tags?.applyValue({ args0 -> args0.map({ args0 -> args0.key.to(args0.value) }).toMap() }))
            .zoneRedundant(zoneRedundant?.applyValue({ args0 -> args0 })).build()
}

/**
 * Builder for [DatabaseArgs].
 */
@PulumiTagMarker
public class DatabaseArgsBuilder internal constructor() {
    private var autoPauseDelay: Output<Int>? = null

    private var catalogCollation: Output<Either<String, CatalogCollationType>>? = null

    private var collation: Output<String>? = null

    private var createMode: Output<Either<String, CreateMode>>? = null

    private var databaseName: Output<String>? = null

    private var elasticPoolId: Output<String>? = null

    private var federatedClientId: Output<String>? = null

    private var highAvailabilityReplicaCount: Output<Int>? = null

    private var identity: Output<DatabaseIdentityArgs>? = null

    private var isLedgerOn: Output<Boolean>? = null

    private var licenseType: Output<Either<String, DatabaseLicenseType>>? = null

    private var location: Output<String>? = null

    private var longTermRetentionBackupResourceId: Output<String>? = null

    private var maintenanceConfigurationId: Output<String>? = null

    private var maxSizeBytes: Output<Double>? = null

    private var minCapacity: Output<Double>? = null

    private var readScale: Output<Either<String, DatabaseReadScale>>? = null

    private var recoverableDatabaseId: Output<String>? = null

    private var recoveryServicesRecoveryPointId: Output<String>? = null

    private var requestedBackupStorageRedundancy: Output<Either<String, BackupStorageRedundancy>>? =
        null

    private var resourceGroupName: Output<String>? = null

    private var restorableDroppedDatabaseId: Output<String>? = null

    private var restorePointInTime: Output<String>? = null

    private var sampleName: Output<Either<String, SampleName>>? = null

    private var secondaryType: Output<Either<String, SecondaryType>>? = null

    private var serverName: Output<String>? = null

    private var sku: Output<SkuArgs>? = null

    private var sourceDatabaseDeletionDate: Output<String>? = null

    private var sourceDatabaseId: Output<String>? = null

    private var sourceResourceId: Output<String>? = null

    private var tags: Output<Map<String, String>>? = null

    private var zoneRedundant: Output<Boolean>? = null

    /**
     * @param value Time in minutes after which database is automatically paused. A value of -1 means that automatic pause is disabled
     */
    @JvmName("ctsnvyevjobvoshq")
    public suspend fun autoPauseDelay(`value`: Output<Int>) {
        this.autoPauseDelay = value
    }

    /**
     * @param value Collation of the metadata catalog.
     */
    @JvmName("kaqnemfkxdpymmyn")
    public suspend fun catalogCollation(`value`: Output<Either<String, CatalogCollationType>>) {
        this.catalogCollation = value
    }

    /**
     * @param value The collation of the database.
     */
    @JvmName("dkqawxjnrkeuhufb")
    public suspend fun collation(`value`: Output<String>) {
        this.collation = value
    }

    /**
     * @param value Specifies the mode of database creation.
     * Default: regular database creation.
     * Copy: creates a database as a copy of an existing database. sourceDatabaseId must be specified as the resource ID of the source database.
     * Secondary: creates a database as a secondary replica of an existing database. sourceDatabaseId must be specified as the resource ID of the existing primary database.
     * PointInTimeRestore: Creates a database by restoring a point in time backup of an existing database. sourceDatabaseId must be specified as the resource ID of the existing database, and restorePointInTime must be specified.
     * Recovery: Creates a database by restoring a geo-replicated backup. sourceDatabaseId must be specified as the recoverable database resource ID to restore.
     * Restore: Creates a database by restoring a backup of a deleted database. sourceDatabaseId must be specified. If sourceDatabaseId is the database's original resource ID, then sourceDatabaseDeletionDate must be specified. Otherwise sourceDatabaseId must be the restorable dropped database resource ID and sourceDatabaseDeletionDate is ignored. restorePointInTime may also be specified to restore from an earlier point in time.
     * RestoreLongTermRetentionBackup: Creates a database by restoring from a long term retention vault. recoveryServicesRecoveryPointResourceId must be specified as the recovery point resource ID.
     * Copy, Secondary, and RestoreLongTermRetentionBackup are not supported for DataWarehouse edition.
     */
    @JvmName("eryfpuajkehvmfil")
    public suspend fun createMode(`value`: Output<Either<String, CreateMode>>) {
        this.createMode = value
    }

    /**
     * @param value The name of the database.
     */
    @JvmName("jlkqbhvlcrxumond")
    public suspend fun databaseName(`value`: Output<String>) {
        this.databaseName = value
    }

    /**
     * @param value The resource identifier of the elastic pool containing this database.
     */
    @JvmName("bjlkjwgntlwaruco")
    public suspend fun elasticPoolId(`value`: Output<String>) {
        this.elasticPoolId = value
    }

    /**
     * @param value The Client id used for cross tenant per database CMK scenario
     */
    @JvmName("wbsiebhhudduqiof")
    public suspend fun federatedClientId(`value`: Output<String>) {
        this.federatedClientId = value
    }

    /**
     * @param value The number of secondary replicas associated with the database that are used to provide high availability. Not applicable to a Hyperscale database within an elastic pool.
     */
    @JvmName("yprgfqsktnnbljfv")
    public suspend fun highAvailabilityReplicaCount(`value`: Output<Int>) {
        this.highAvailabilityReplicaCount = value
    }

    /**
     * @param value The Azure Active Directory identity of the database.
     */
    @JvmName("odvqpvtxathrfrgl")
    public suspend fun identity(`value`: Output<DatabaseIdentityArgs>) {
        this.identity = value
    }

    /**
     * @param value Whether or not this database is a ledger database, which means all tables in the database are ledger tables. Note: the value of this property cannot be changed after the database has been created.
     */
    @JvmName("boxjqoqubwavwrvn")
    public suspend fun isLedgerOn(`value`: Output<Boolean>) {
        this.isLedgerOn = value
    }

    /**
     * @param value The license type to apply for this database. `LicenseIncluded` if you need a license, or `BasePrice` if you have a license and are eligible for the Azure Hybrid Benefit.
     */
    @JvmName("looshkylrqycsbfj")
    public suspend fun licenseType(`value`: Output<Either<String, DatabaseLicenseType>>) {
        this.licenseType = value
    }

    /**
     * @param value Resource location.
     */
    @JvmName("bxcdjowsdubskkgk")
    public suspend fun location(`value`: Output<String>) {
        this.location = value
    }

    /**
     * @param value The resource identifier of the long term retention backup associated with create operation of this database.
     */
    @JvmName("gofrvguosslwqkht")
    public suspend fun longTermRetentionBackupResourceId(`value`: Output<String>) {
        this.longTermRetentionBackupResourceId = value
    }

    /**
     * @param value Maintenance configuration id assigned to the database. This configuration defines the period when the maintenance updates will occur.
     */
    @JvmName("vvhvjeciukecuacx")
    public suspend fun maintenanceConfigurationId(`value`: Output<String>) {
        this.maintenanceConfigurationId = value
    }

    /**
     * @param value The max size of the database expressed in bytes.
     */
    @JvmName("hinavxqvsjoxfotf")
    public suspend fun maxSizeBytes(`value`: Output<Double>) {
        this.maxSizeBytes = value
    }

    /**
     * @param value Minimal capacity that database will always have allocated, if not paused
     */
    @JvmName("aprxbxqbkcowoyyw")
    public suspend fun minCapacity(`value`: Output<Double>) {
        this.minCapacity = value
    }

    /**
     * @param value The state of read-only routing. If enabled, connections that have application intent set to readonly in their connection string may be routed to a readonly secondary replica in the same region. Not applicable to a Hyperscale database within an elastic pool.
     */
    @JvmName("qrlmquwqrertyapa")
    public suspend fun readScale(`value`: Output<Either<String, DatabaseReadScale>>) {
        this.readScale = value
    }

    /**
     * @param value The resource identifier of the recoverable database associated with create operation of this database.
     */
    @JvmName("hbbalvmterrlhkkr")
    public suspend fun recoverableDatabaseId(`value`: Output<String>) {
        this.recoverableDatabaseId = value
    }

    /**
     * @param value The resource identifier of the recovery point associated with create operation of this database.
     */
    @JvmName("fbwqipdryopsvdse")
    public suspend fun recoveryServicesRecoveryPointId(`value`: Output<String>) {
        this.recoveryServicesRecoveryPointId = value
    }

    /**
     * @param value The storage account type to be used to store backups for this database.
     */
    @JvmName("aevebmrotqllrobu")
    public suspend
    fun requestedBackupStorageRedundancy(`value`: Output<Either<String, BackupStorageRedundancy>>) {
        this.requestedBackupStorageRedundancy = value
    }

    /**
     * @param value The name of the resource group that contains the resource. You can obtain this value from the Azure Resource Manager API or the portal.
     */
    @JvmName("impglhrkbdbbnljh")
    public suspend fun resourceGroupName(`value`: Output<String>) {
        this.resourceGroupName = value
    }

    /**
     * @param value The resource identifier of the restorable dropped database associated with create operation of this database.
     */
    @JvmName("njxuqbeyqjvqikoy")
    public suspend fun restorableDroppedDatabaseId(`value`: Output<String>) {
        this.restorableDroppedDatabaseId = value
    }

    /**
     * @param value Specifies the point in time (ISO8601 format) of the source database that will be restored to create the new database.
     */
    @JvmName("njvlwgtjtdsutsey")
    public suspend fun restorePointInTime(`value`: Output<String>) {
        this.restorePointInTime = value
    }

    /**
     * @param value The name of the sample schema to apply when creating this database.
     */
    @JvmName("uirycbwnqouyrlbr")
    public suspend fun sampleName(`value`: Output<Either<String, SampleName>>) {
        this.sampleName = value
    }

    /**
     * @param value The secondary type of the database if it is a secondary.  Valid values are Geo and Named.
     */
    @JvmName("nrbqlnbiittpgiqc")
    public suspend fun secondaryType(`value`: Output<Either<String, SecondaryType>>) {
        this.secondaryType = value
    }

    /**
     * @param value The name of the server.
     */
    @JvmName("jgwvmevfihqvysbi")
    public suspend fun serverName(`value`: Output<String>) {
        this.serverName = value
    }

    /**
     * @param value The database SKU.
     * The list of SKUs may vary by region and support offer. To determine the SKUs (including the SKU name, tier/edition, family, and capacity) that are available to your subscription in an Azure region, use the `Capabilities_ListByLocation` REST API or one of the following commands:
     * ```azurecli
     * az sql db list-editions -l <location> -o table
     * ````
     * ```powershell
     * Get-AzSqlServerServiceObjective -Location <location>
     * ````
     */
    @JvmName("vqggadroqmetkyyj")
    public suspend fun sku(`value`: Output<SkuArgs>) {
        this.sku = value
    }

    /**
     * @param value Specifies the time that the database was deleted.
     */
    @JvmName("dhnwsdfwrafdppkf")
    public suspend fun sourceDatabaseDeletionDate(`value`: Output<String>) {
        this.sourceDatabaseDeletionDate = value
    }

    /**
     * @param value The resource identifier of the source database associated with create operation of this database.
     */
    @JvmName("jgdmxfkmkorosgxo")
    public suspend fun sourceDatabaseId(`value`: Output<String>) {
        this.sourceDatabaseId = value
    }

    /**
     * @param value The resource identifier of the source associated with the create operation of this database.
     * This property is only supported for DataWarehouse edition and allows to restore across subscriptions.
     * When sourceResourceId is specified, sourceDatabaseId, recoverableDatabaseId, restorableDroppedDatabaseId and sourceDatabaseDeletionDate must not be specified and CreateMode must be PointInTimeRestore, Restore or Recover.
     * When createMode is PointInTimeRestore, sourceResourceId must be the resource ID of the existing database or existing sql pool, and restorePointInTime must be specified.
     * When createMode is Restore, sourceResourceId must be the resource ID of restorable dropped database or restorable dropped sql pool.
     * When createMode is Recover, sourceResourceId must be the resource ID of recoverable database or recoverable sql pool.
     * When source subscription belongs to a different tenant than target subscription, “x-ms-authorization-auxiliary” header must contain authentication token for the source tenant. For more details about “x-ms-authorization-auxiliary” header see https://docs.microsoft.com/en-us/azure/azure-resource-manager/management/authenticate-multi-tenant
     */
    @JvmName("sbodwkddtistdxvu")
    public suspend fun sourceResourceId(`value`: Output<String>) {
        this.sourceResourceId = value
    }

    /**
     * @param value Resource tags.
     */
    @JvmName("ihjtqjrsyldehnbu")
    public suspend fun tags(`value`: Output<Map<String, String>>) {
        this.tags = value
    }

    /**
     * @param value Whether or not this database is zone redundant, which means the replicas of this database will be spread across multiple availability zones.
     */
    @JvmName("agdaruwhsyjvuqpb")
    public suspend fun zoneRedundant(`value`: Output<Boolean>) {
        this.zoneRedundant = value
    }

    /**
     * @param value Time in minutes after which database is automatically paused. A value of -1 means that automatic pause is disabled
     */
    @JvmName("hnpdflepklcvvgau")
    public suspend fun autoPauseDelay(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.autoPauseDelay = mapped
    }

    /**
     * @param value Collation of the metadata catalog.
     */
    @JvmName("ddcbhkksjvtfjarh")
    public suspend fun catalogCollation(`value`: Either<String, CatalogCollationType>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.catalogCollation = mapped
    }

    /**
     * @param value Collation of the metadata catalog.
     */
    @JvmName("sfpntsjyoopajwob")
    public fun catalogCollation(`value`: String) {
        val toBeMapped = Either.ofLeft<String, CatalogCollationType>(value)
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.catalogCollation = mapped
    }

    /**
     * @param value Collation of the metadata catalog.
     */
    @JvmName("htqlwtwleniwguva")
    public fun catalogCollation(`value`: CatalogCollationType) {
        val toBeMapped = Either.ofRight<String, CatalogCollationType>(value)
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.catalogCollation = mapped
    }

    /**
     * @param value The collation of the database.
     */
    @JvmName("eeshflcndovvpffp")
    public suspend fun collation(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.collation = mapped
    }

    /**
     * @param value Specifies the mode of database creation.
     * Default: regular database creation.
     * Copy: creates a database as a copy of an existing database. sourceDatabaseId must be specified as the resource ID of the source database.
     * Secondary: creates a database as a secondary replica of an existing database. sourceDatabaseId must be specified as the resource ID of the existing primary database.
     * PointInTimeRestore: Creates a database by restoring a point in time backup of an existing database. sourceDatabaseId must be specified as the resource ID of the existing database, and restorePointInTime must be specified.
     * Recovery: Creates a database by restoring a geo-replicated backup. sourceDatabaseId must be specified as the recoverable database resource ID to restore.
     * Restore: Creates a database by restoring a backup of a deleted database. sourceDatabaseId must be specified. If sourceDatabaseId is the database's original resource ID, then sourceDatabaseDeletionDate must be specified. Otherwise sourceDatabaseId must be the restorable dropped database resource ID and sourceDatabaseDeletionDate is ignored. restorePointInTime may also be specified to restore from an earlier point in time.
     * RestoreLongTermRetentionBackup: Creates a database by restoring from a long term retention vault. recoveryServicesRecoveryPointResourceId must be specified as the recovery point resource ID.
     * Copy, Secondary, and RestoreLongTermRetentionBackup are not supported for DataWarehouse edition.
     */
    @JvmName("wsiovipjvhhvxnaj")
    public suspend fun createMode(`value`: Either<String, CreateMode>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.createMode = mapped
    }

    /**
     * @param value Specifies the mode of database creation.
     * Default: regular database creation.
     * Copy: creates a database as a copy of an existing database. sourceDatabaseId must be specified as the resource ID of the source database.
     * Secondary: creates a database as a secondary replica of an existing database. sourceDatabaseId must be specified as the resource ID of the existing primary database.
     * PointInTimeRestore: Creates a database by restoring a point in time backup of an existing database. sourceDatabaseId must be specified as the resource ID of the existing database, and restorePointInTime must be specified.
     * Recovery: Creates a database by restoring a geo-replicated backup. sourceDatabaseId must be specified as the recoverable database resource ID to restore.
     * Restore: Creates a database by restoring a backup of a deleted database. sourceDatabaseId must be specified. If sourceDatabaseId is the database's original resource ID, then sourceDatabaseDeletionDate must be specified. Otherwise sourceDatabaseId must be the restorable dropped database resource ID and sourceDatabaseDeletionDate is ignored. restorePointInTime may also be specified to restore from an earlier point in time.
     * RestoreLongTermRetentionBackup: Creates a database by restoring from a long term retention vault. recoveryServicesRecoveryPointResourceId must be specified as the recovery point resource ID.
     * Copy, Secondary, and RestoreLongTermRetentionBackup are not supported for DataWarehouse edition.
     */
    @JvmName("qnvieiwqysxyxwtk")
    public fun createMode(`value`: String) {
        val toBeMapped = Either.ofLeft<String, CreateMode>(value)
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.createMode = mapped
    }

    /**
     * @param value Specifies the mode of database creation.
     * Default: regular database creation.
     * Copy: creates a database as a copy of an existing database. sourceDatabaseId must be specified as the resource ID of the source database.
     * Secondary: creates a database as a secondary replica of an existing database. sourceDatabaseId must be specified as the resource ID of the existing primary database.
     * PointInTimeRestore: Creates a database by restoring a point in time backup of an existing database. sourceDatabaseId must be specified as the resource ID of the existing database, and restorePointInTime must be specified.
     * Recovery: Creates a database by restoring a geo-replicated backup. sourceDatabaseId must be specified as the recoverable database resource ID to restore.
     * Restore: Creates a database by restoring a backup of a deleted database. sourceDatabaseId must be specified. If sourceDatabaseId is the database's original resource ID, then sourceDatabaseDeletionDate must be specified. Otherwise sourceDatabaseId must be the restorable dropped database resource ID and sourceDatabaseDeletionDate is ignored. restorePointInTime may also be specified to restore from an earlier point in time.
     * RestoreLongTermRetentionBackup: Creates a database by restoring from a long term retention vault. recoveryServicesRecoveryPointResourceId must be specified as the recovery point resource ID.
     * Copy, Secondary, and RestoreLongTermRetentionBackup are not supported for DataWarehouse edition.
     */
    @JvmName("wiyejktkxxrkjous")
    public fun createMode(`value`: CreateMode) {
        val toBeMapped = Either.ofRight<String, CreateMode>(value)
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.createMode = mapped
    }

    /**
     * @param value The name of the database.
     */
    @JvmName("ubnaraqirnehbhog")
    public suspend fun databaseName(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.databaseName = mapped
    }

    /**
     * @param value The resource identifier of the elastic pool containing this database.
     */
    @JvmName("aibvdkyafxwibxvn")
    public suspend fun elasticPoolId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.elasticPoolId = mapped
    }

    /**
     * @param value The Client id used for cross tenant per database CMK scenario
     */
    @JvmName("luqqblfbhikqxerh")
    public suspend fun federatedClientId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.federatedClientId = mapped
    }

    /**
     * @param value The number of secondary replicas associated with the database that are used to provide high availability. Not applicable to a Hyperscale database within an elastic pool.
     */
    @JvmName("ajfhrnamxyrhtpby")
    public suspend fun highAvailabilityReplicaCount(`value`: Int?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.highAvailabilityReplicaCount = mapped
    }

    /**
     * @param value The Azure Active Directory identity of the database.
     */
    @JvmName("yssjxcgcxbwerbhe")
    public suspend fun identity(`value`: DatabaseIdentityArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.identity = mapped
    }

    /**
     * @param argument The Azure Active Directory identity of the database.
     */
    @JvmName("mkqxaisbbbboqydf")
    public suspend fun identity(argument: suspend DatabaseIdentityArgsBuilder.() -> Unit) {
        val toBeMapped = DatabaseIdentityArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.identity = mapped
    }

    /**
     * @param value Whether or not this database is a ledger database, which means all tables in the database are ledger tables. Note: the value of this property cannot be changed after the database has been created.
     */
    @JvmName("nisbmsmvwbaokfyr")
    public suspend fun isLedgerOn(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.isLedgerOn = mapped
    }

    /**
     * @param value The license type to apply for this database. `LicenseIncluded` if you need a license, or `BasePrice` if you have a license and are eligible for the Azure Hybrid Benefit.
     */
    @JvmName("iqbevvrvtwvtyohl")
    public suspend fun licenseType(`value`: Either<String, DatabaseLicenseType>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.licenseType = mapped
    }

    /**
     * @param value The license type to apply for this database. `LicenseIncluded` if you need a license, or `BasePrice` if you have a license and are eligible for the Azure Hybrid Benefit.
     */
    @JvmName("nieiuungslbkpjav")
    public fun licenseType(`value`: String) {
        val toBeMapped = Either.ofLeft<String, DatabaseLicenseType>(value)
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.licenseType = mapped
    }

    /**
     * @param value The license type to apply for this database. `LicenseIncluded` if you need a license, or `BasePrice` if you have a license and are eligible for the Azure Hybrid Benefit.
     */
    @JvmName("mfhlepsgxvehltht")
    public fun licenseType(`value`: DatabaseLicenseType) {
        val toBeMapped = Either.ofRight<String, DatabaseLicenseType>(value)
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.licenseType = mapped
    }

    /**
     * @param value Resource location.
     */
    @JvmName("ymlhnwhuuqtqurfk")
    public suspend fun location(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.location = mapped
    }

    /**
     * @param value The resource identifier of the long term retention backup associated with create operation of this database.
     */
    @JvmName("ciglrdhmhxsnmvdi")
    public suspend fun longTermRetentionBackupResourceId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.longTermRetentionBackupResourceId = mapped
    }

    /**
     * @param value Maintenance configuration id assigned to the database. This configuration defines the period when the maintenance updates will occur.
     */
    @JvmName("ttvrcvarbhcyocng")
    public suspend fun maintenanceConfigurationId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.maintenanceConfigurationId = mapped
    }

    /**
     * @param value The max size of the database expressed in bytes.
     */
    @JvmName("qhbinubgyqdkohds")
    public suspend fun maxSizeBytes(`value`: Double?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.maxSizeBytes = mapped
    }

    /**
     * @param value Minimal capacity that database will always have allocated, if not paused
     */
    @JvmName("gifsguujrackfboy")
    public suspend fun minCapacity(`value`: Double?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.minCapacity = mapped
    }

    /**
     * @param value The state of read-only routing. If enabled, connections that have application intent set to readonly in their connection string may be routed to a readonly secondary replica in the same region. Not applicable to a Hyperscale database within an elastic pool.
     */
    @JvmName("dtuthhfillcbsgxn")
    public suspend fun readScale(`value`: Either<String, DatabaseReadScale>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.readScale = mapped
    }

    /**
     * @param value The state of read-only routing. If enabled, connections that have application intent set to readonly in their connection string may be routed to a readonly secondary replica in the same region. Not applicable to a Hyperscale database within an elastic pool.
     */
    @JvmName("mdityudcvmwsefvk")
    public fun readScale(`value`: String) {
        val toBeMapped = Either.ofLeft<String, DatabaseReadScale>(value)
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.readScale = mapped
    }

    /**
     * @param value The state of read-only routing. If enabled, connections that have application intent set to readonly in their connection string may be routed to a readonly secondary replica in the same region. Not applicable to a Hyperscale database within an elastic pool.
     */
    @JvmName("brjsnccfcxbelbho")
    public fun readScale(`value`: DatabaseReadScale) {
        val toBeMapped = Either.ofRight<String, DatabaseReadScale>(value)
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.readScale = mapped
    }

    /**
     * @param value The resource identifier of the recoverable database associated with create operation of this database.
     */
    @JvmName("waxkfbroocbprtct")
    public suspend fun recoverableDatabaseId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.recoverableDatabaseId = mapped
    }

    /**
     * @param value The resource identifier of the recovery point associated with create operation of this database.
     */
    @JvmName("rtbhucbefaphrakq")
    public suspend fun recoveryServicesRecoveryPointId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.recoveryServicesRecoveryPointId = mapped
    }

    /**
     * @param value The storage account type to be used to store backups for this database.
     */
    @JvmName("siaygryjybcmlhsb")
    public suspend
    fun requestedBackupStorageRedundancy(`value`: Either<String, BackupStorageRedundancy>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.requestedBackupStorageRedundancy = mapped
    }

    /**
     * @param value The storage account type to be used to store backups for this database.
     */
    @JvmName("qwqrjpbnexipvauw")
    public fun requestedBackupStorageRedundancy(`value`: String) {
        val toBeMapped = Either.ofLeft<String, BackupStorageRedundancy>(value)
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.requestedBackupStorageRedundancy = mapped
    }

    /**
     * @param value The storage account type to be used to store backups for this database.
     */
    @JvmName("yqambveopbadhrlm")
    public fun requestedBackupStorageRedundancy(`value`: BackupStorageRedundancy) {
        val toBeMapped = Either.ofRight<String, BackupStorageRedundancy>(value)
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.requestedBackupStorageRedundancy = mapped
    }

    /**
     * @param value The name of the resource group that contains the resource. You can obtain this value from the Azure Resource Manager API or the portal.
     */
    @JvmName("qiaisyaatjvsrpcy")
    public suspend fun resourceGroupName(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.resourceGroupName = mapped
    }

    /**
     * @param value The resource identifier of the restorable dropped database associated with create operation of this database.
     */
    @JvmName("viakbduufburqnbl")
    public suspend fun restorableDroppedDatabaseId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.restorableDroppedDatabaseId = mapped
    }

    /**
     * @param value Specifies the point in time (ISO8601 format) of the source database that will be restored to create the new database.
     */
    @JvmName("bdjyxxjvgharvljf")
    public suspend fun restorePointInTime(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.restorePointInTime = mapped
    }

    /**
     * @param value The name of the sample schema to apply when creating this database.
     */
    @JvmName("vwjroesiotcwikah")
    public suspend fun sampleName(`value`: Either<String, SampleName>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.sampleName = mapped
    }

    /**
     * @param value The name of the sample schema to apply when creating this database.
     */
    @JvmName("qgawpadmcbadfjfm")
    public fun sampleName(`value`: String) {
        val toBeMapped = Either.ofLeft<String, SampleName>(value)
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.sampleName = mapped
    }

    /**
     * @param value The name of the sample schema to apply when creating this database.
     */
    @JvmName("dqdugbopkhcxyskb")
    public fun sampleName(`value`: SampleName) {
        val toBeMapped = Either.ofRight<String, SampleName>(value)
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.sampleName = mapped
    }

    /**
     * @param value The secondary type of the database if it is a secondary.  Valid values are Geo and Named.
     */
    @JvmName("dqutstdtblasdves")
    public suspend fun secondaryType(`value`: Either<String, SecondaryType>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.secondaryType = mapped
    }

    /**
     * @param value The secondary type of the database if it is a secondary.  Valid values are Geo and Named.
     */
    @JvmName("hahuptgrgdthppaa")
    public fun secondaryType(`value`: String) {
        val toBeMapped = Either.ofLeft<String, SecondaryType>(value)
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.secondaryType = mapped
    }

    /**
     * @param value The secondary type of the database if it is a secondary.  Valid values are Geo and Named.
     */
    @JvmName("tjxiwygnihpwvtbj")
    public fun secondaryType(`value`: SecondaryType) {
        val toBeMapped = Either.ofRight<String, SecondaryType>(value)
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.secondaryType = mapped
    }

    /**
     * @param value The name of the server.
     */
    @JvmName("exrqadwhlcsdjvil")
    public suspend fun serverName(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.serverName = mapped
    }

    /**
     * @param value The database SKU.
     * The list of SKUs may vary by region and support offer. To determine the SKUs (including the SKU name, tier/edition, family, and capacity) that are available to your subscription in an Azure region, use the `Capabilities_ListByLocation` REST API or one of the following commands:
     * ```azurecli
     * az sql db list-editions -l <location> -o table
     * ````
     * ```powershell
     * Get-AzSqlServerServiceObjective -Location <location>
     * ````
     */
    @JvmName("aefjepeixiqxildg")
    public suspend fun sku(`value`: SkuArgs?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.sku = mapped
    }

    /**
     * @param argument The database SKU.
     * The list of SKUs may vary by region and support offer. To determine the SKUs (including the SKU name, tier/edition, family, and capacity) that are available to your subscription in an Azure region, use the `Capabilities_ListByLocation` REST API or one of the following commands:
     * ```azurecli
     * az sql db list-editions -l <location> -o table
     * ````
     * ```powershell
     * Get-AzSqlServerServiceObjective -Location <location>
     * ````
     */
    @JvmName("cfrkqoklmyhqjmvl")
    public suspend fun sku(argument: suspend SkuArgsBuilder.() -> Unit) {
        val toBeMapped = SkuArgsBuilder().applySuspend { argument() }.build()
        val mapped = of(toBeMapped)
        this.sku = mapped
    }

    /**
     * @param value Specifies the time that the database was deleted.
     */
    @JvmName("qegtuskspudcotbu")
    public suspend fun sourceDatabaseDeletionDate(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.sourceDatabaseDeletionDate = mapped
    }

    /**
     * @param value The resource identifier of the source database associated with create operation of this database.
     */
    @JvmName("gdadbedakxhlcopu")
    public suspend fun sourceDatabaseId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.sourceDatabaseId = mapped
    }

    /**
     * @param value The resource identifier of the source associated with the create operation of this database.
     * This property is only supported for DataWarehouse edition and allows to restore across subscriptions.
     * When sourceResourceId is specified, sourceDatabaseId, recoverableDatabaseId, restorableDroppedDatabaseId and sourceDatabaseDeletionDate must not be specified and CreateMode must be PointInTimeRestore, Restore or Recover.
     * When createMode is PointInTimeRestore, sourceResourceId must be the resource ID of the existing database or existing sql pool, and restorePointInTime must be specified.
     * When createMode is Restore, sourceResourceId must be the resource ID of restorable dropped database or restorable dropped sql pool.
     * When createMode is Recover, sourceResourceId must be the resource ID of recoverable database or recoverable sql pool.
     * When source subscription belongs to a different tenant than target subscription, “x-ms-authorization-auxiliary” header must contain authentication token for the source tenant. For more details about “x-ms-authorization-auxiliary” header see https://docs.microsoft.com/en-us/azure/azure-resource-manager/management/authenticate-multi-tenant
     */
    @JvmName("npdcpekohnuipylf")
    public suspend fun sourceResourceId(`value`: String?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.sourceResourceId = mapped
    }

    /**
     * @param value Resource tags.
     */
    @JvmName("gasktebhvxdfykre")
    public suspend fun tags(`value`: Map<String, String>?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param values Resource tags.
     */
    @JvmName("lvmffwqwrpclymfh")
    public fun tags(vararg values: Pair<String, String>) {
        val toBeMapped = values.toMap()
        val mapped = toBeMapped.let({ args0 -> of(args0) })
        this.tags = mapped
    }

    /**
     * @param value Whether or not this database is zone redundant, which means the replicas of this database will be spread across multiple availability zones.
     */
    @JvmName("wxryryiydmwkrfgc")
    public suspend fun zoneRedundant(`value`: Boolean?) {
        val toBeMapped = value
        val mapped = toBeMapped?.let({ args0 -> of(args0) })
        this.zoneRedundant = mapped
    }

    internal fun build(): DatabaseArgs = DatabaseArgs(
        autoPauseDelay = autoPauseDelay,
        catalogCollation = catalogCollation,
        collation = collation,
        createMode = createMode,
        databaseName = databaseName,
        elasticPoolId = elasticPoolId,
        federatedClientId = federatedClientId,
        highAvailabilityReplicaCount = highAvailabilityReplicaCount,
        identity = identity,
        isLedgerOn = isLedgerOn,
        licenseType = licenseType,
        location = location,
        longTermRetentionBackupResourceId = longTermRetentionBackupResourceId,
        maintenanceConfigurationId = maintenanceConfigurationId,
        maxSizeBytes = maxSizeBytes,
        minCapacity = minCapacity,
        readScale = readScale,
        recoverableDatabaseId = recoverableDatabaseId,
        recoveryServicesRecoveryPointId = recoveryServicesRecoveryPointId,
        requestedBackupStorageRedundancy = requestedBackupStorageRedundancy,
        resourceGroupName = resourceGroupName,
        restorableDroppedDatabaseId = restorableDroppedDatabaseId,
        restorePointInTime = restorePointInTime,
        sampleName = sampleName,
        secondaryType = secondaryType,
        serverName = serverName,
        sku = sku,
        sourceDatabaseDeletionDate = sourceDatabaseDeletionDate,
        sourceDatabaseId = sourceDatabaseId,
        sourceResourceId = sourceResourceId,
        tags = tags,
        zoneRedundant = zoneRedundant,
    )
}
