/***
 *
 * Copyright (c) 2007 Paul Hammant
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from
 *    this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

package org.weakref.jmx.internal.paranamer;

import org.weakref.jmx.internal.paranamer.BytecodeReadingParanamer;

import java.lang.reflect.AccessibleObject;
import java.lang.reflect.Method;
import java.lang.reflect.Constructor;

/**
 * Implementation of Paranamer which automatically chooses between two Paranamer implementations depending on which can supply data
 *
 * @author Paul Hammant
 * @author Mauro Talevi
 */
public class AdaptiveParanamer implements Paranamer {

    public static final String __PARANAMER_DATA = "v1.0 \n"
        + "org.weakref.jmx.internal.paranamer.AdaptiveParanamer AdaptiveParanamer \n"
        + "org.weakref.jmx.internal.paranamer.AdaptiveParanamer AdaptiveParanamer org.weakref.jmx.internal.paranamer.Paranamer,org.weakref.jmx.internal.paranamer.Paranamer delegate,fallback\n"
        + "org.weakref.jmx.internal.paranamer.AdaptiveParanamer toString \n"
        + "org.weakref.jmx.internal.paranamer.AdaptiveParanamer lookupParameterNames java.lang.AccessibleObject methodOrCtor \n";

    private Paranamer delegate;
    private Paranamer fallback;

    /**
     * Cache a DefaultParanamer's lookups.
     */
    public AdaptiveParanamer() {
        this(new DefaultParanamer(), new BytecodeReadingParanamer());
    }


    /**
     * Cache a primary and secondary Paranamer instance (the second is a fallback to the first)
     * @param delegate first
     * @param fallback second
     */
    public AdaptiveParanamer(Paranamer delegate, Paranamer fallback) {
        this.delegate = delegate;
        this.fallback = fallback;
        if (delegate == null || fallback == null || delegate == fallback) {
            throw new RuntimeException("must supply delegate and fallback (which must be different)");
        }
    }

    public String[] lookupParameterNames(AccessibleObject methodOrConstructor) {
        return lookupParameterNames(methodOrConstructor, true);
    }

    public String[] lookupParameterNames(AccessibleObject methodOrCtor, boolean throwExceptionIfMissing) {

        Class declaringClass = null;
        String name = null;
        if (methodOrCtor instanceof Method) {
            Method method = (Method) methodOrCtor;
            declaringClass = method.getDeclaringClass();
            name = method.getName();
        } else {
            Constructor constructor = (Constructor) methodOrCtor;
            declaringClass = constructor.getDeclaringClass();
            name = constructor.getName();
        }

        String[] names = delegate.lookupParameterNames(methodOrCtor, false);
        if (names == Paranamer.EMPTY_NAMES) {
            names = fallback.lookupParameterNames(methodOrCtor, throwExceptionIfMissing);
        }
        return names;

    }

    public String toString() {
         return new StringBuffer("[AdaptiveParanamer delegate=")
         .append(delegate).append(", fallback=").append(fallback).append("]").toString();
     }

}