"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.KeywordCxt = void 0;
const context_1 = __importDefault(require("./compile/context"));
exports.KeywordCxt = context_1.default;
const cache_1 = __importDefault(require("./cache"));
const error_classes_1 = require("./compile/error_classes");
const rules_1 = require("./compile/rules");
const dataType_1 = require("./compile/validate/dataType");
const compile_1 = require("./compile");
const codegen_1 = require("./compile/codegen");
const resolve_1 = require("./compile/resolve");
const core_1 = __importDefault(require("./vocabularies/core"));
const validation_1 = __importDefault(require("./vocabularies/validation"));
const applicator_1 = __importDefault(require("./vocabularies/applicator"));
const format_1 = __importDefault(require("./vocabularies/format"));
const metadata_1 = require("./vocabularies/metadata");
const fast_json_stable_stringify_1 = __importDefault(require("fast-json-stable-stringify"));
const util_1 = require("./compile/util");
const data_json_1 = __importDefault(require("./refs/data.json"));
const json_schema_draft_07_json_1 = __importDefault(require("./refs/json-schema-draft-07.json"));
const META_SCHEMA_ID = "http://json-schema.org/draft-07/schema";
const META_IGNORE_OPTIONS = ["removeAdditional", "useDefaults", "coerceTypes"];
const META_SUPPORT_DATA = ["/properties"];
const EXT_SCOPE_NAMES = new Set([
    "validate",
    "wrapper",
    "root",
    "schema",
    "keyword",
    "pattern",
    "formats",
    "validate$data",
    "func",
    "Error",
]);
const optsDefaults = {
    strict: true,
    code: {},
    loopRequired: Infinity,
    loopEnum: Infinity,
    addUsedSchema: true,
};
class Ajv {
    constructor(opts = {}) {
        var _a, _b, _c;
        // shared external scope values for compiled functions
        this.scope = new codegen_1.ValueScope({ scope: {}, prefixes: EXT_SCOPE_NAMES });
        this.schemas = {};
        this.refs = {};
        this.formats = {};
        this._compilations = new Set();
        this._loading = {};
        opts = this.opts = {
            ...optsDefaults,
            ...opts,
            serialize: opts.serialize === false ? (x) => x : (_a = opts.serialize) !== null && _a !== void 0 ? _a : fast_json_stable_stringify_1.default,
            addUsedSchema: (_b = opts.addUsedSchema) !== null && _b !== void 0 ? _b : true,
            validateSchema: (_c = opts.validateSchema) !== null && _c !== void 0 ? _c : true,
        };
        this.logger = getLogger(opts.logger);
        const formatOpt = opts.format;
        opts.format = false;
        this._cache = opts.cache || new cache_1.default();
        this.RULES = rules_1.getRules();
        checkDeprecatedOptions.call(this, opts);
        this._metaOpts = getMetaSchemaOptions.call(this);
        if (opts.formats)
            addInitialFormats.call(this);
        this.addVocabulary(["$async"]);
        this.addVocabulary(core_1.default);
        this.addVocabulary(validation_1.default);
        this.addVocabulary(applicator_1.default);
        this.addVocabulary(format_1.default);
        this.addVocabulary(metadata_1.metadataVocabulary);
        this.addVocabulary(metadata_1.contentVocabulary);
        if (opts.keywords)
            addInitialKeywords.call(this, opts.keywords);
        addDefaultMetaSchema.call(this);
        if (typeof opts.meta == "object")
            this.addMetaSchema(opts.meta);
        addInitialSchemas.call(this);
        opts.format = formatOpt;
    }
    validate(schemaKeyRef, // key, ref or schema object
    data // to be validated
    ) {
        let v;
        if (typeof schemaKeyRef == "string") {
            v = this.getSchema(schemaKeyRef);
            if (!v)
                throw new Error('no schema with key or ref "' + schemaKeyRef + '"');
        }
        else {
            const sch = this._addSchema(schemaKeyRef);
            v = sch.validate || this._compileSchemaEnv(sch);
        }
        const valid = v(data);
        if (!("$async" in v))
            this.errors = v.errors;
        return valid;
    }
    compile(schema, _meta) {
        const sch = this._addSchema(schema, _meta);
        return (sch.validate || this._compileSchemaEnv(sch));
    }
    compileAsync(schema, meta) {
        if (typeof this.opts.loadSchema != "function") {
            throw new Error("options.loadSchema should be a function");
        }
        const { loadSchema } = this.opts;
        return runCompileAsync.call(this, schema, meta);
        async function runCompileAsync(_schema, _meta) {
            await loadMetaSchema.call(this, _schema.$schema);
            const sch = this._addSchema(_schema, _meta);
            return sch.validate || _compileAsync.call(this, sch);
        }
        async function loadMetaSchema($ref) {
            if ($ref && !this.getSchema($ref)) {
                await runCompileAsync.call(this, { $ref }, true);
            }
        }
        async function _compileAsync(sch) {
            try {
                return this._compileSchemaEnv(sch);
            }
            catch (e) {
                if (!(e instanceof error_classes_1.MissingRefError))
                    throw e;
                checkLoaded.call(this, e);
                await loadMissingSchema.call(this, e.missingSchema);
                return _compileAsync.call(this, sch);
            }
        }
        function checkLoaded({ missingSchema: ref, missingRef }) {
            if (this.refs[ref]) {
                throw new Error(`AnySchema ${ref} is loaded but ${missingRef} cannot be resolved`);
            }
        }
        async function loadMissingSchema(ref) {
            const _schema = await _loadSchema.call(this, ref);
            if (!this.refs[ref])
                await loadMetaSchema.call(this, _schema.$schema);
            if (!this.refs[ref])
                this.addSchema(_schema, ref, meta);
        }
        async function _loadSchema(ref) {
            const p = this._loading[ref];
            if (p)
                return p;
            try {
                return await (this._loading[ref] = loadSchema(ref));
            }
            finally {
                delete this._loading[ref];
            }
        }
    }
    // Adds schema to the instance
    addSchema(schema, // If array is passed, `key` will be ignored
    key, // Optional schema key. Can be passed to `validate` method instead of schema object or id/ref. One schema per instance can have empty `id` and `key`.
    _meta, // true if schema is a meta-schema. Used internally, addMetaSchema should be used instead.
    _validateSchema = this.opts.validateSchema // false to skip schema validation. Used internally, option validateSchema should be used instead.
    ) {
        if (Array.isArray(schema)) {
            for (const sch of schema)
                this.addSchema(sch, undefined, _meta, _validateSchema);
            return this;
        }
        let id;
        if (typeof schema === "object") {
            id = schema.$id;
            if (id !== undefined && typeof id != "string")
                throw new Error("schema id must be string");
        }
        key = resolve_1.normalizeId(key || id);
        this._checkUnique(key);
        this.schemas[key] = this._addSchema(schema, _meta, _validateSchema, true);
        return this;
    }
    // Add schema that will be used to validate other schemas
    // options in META_IGNORE_OPTIONS are alway set to false
    addMetaSchema(schema, key, // schema key
    _validateSchema = this.opts.validateSchema // false to skip schema validation, can be used to override validateSchema option for meta-schema
    ) {
        this.addSchema(schema, key, true, _validateSchema);
        return this;
    }
    //  Validate schema against its meta-schema
    validateSchema(schema, throwOrLogError) {
        if (typeof schema == "boolean")
            return true;
        let $schema;
        $schema = schema.$schema;
        if ($schema !== undefined && typeof $schema != "string") {
            throw new Error("$schema must be a string");
        }
        $schema = $schema || this.opts.defaultMeta || defaultMeta.call(this);
        if (!$schema) {
            this.logger.warn("meta-schema not available");
            this.errors = null;
            return true;
        }
        const valid = this.validate($schema, schema);
        if (!valid && throwOrLogError) {
            const message = "schema is invalid: " + this.errorsText();
            if (this.opts.validateSchema === "log")
                this.logger.error(message);
            else
                throw new Error(message);
        }
        return valid;
    }
    // Get compiled schema by `key` or `ref`.
    // (`key` that was passed to `addSchema` or full schema reference - `schema.$id` or resolved id)
    getSchema(keyRef) {
        let sch;
        while (typeof (sch = getSchEnv.call(this, keyRef)) == "string")
            keyRef = sch;
        if (sch === undefined) {
            const root = new compile_1.SchemaEnv({ schema: {} });
            sch = compile_1.resolveSchema.call(this, root, keyRef);
            if (!sch)
                return;
            this.refs[keyRef] = sch;
        }
        return (sch.validate || this._compileSchemaEnv(sch));
    }
    // Remove cached schema(s).
    // If no parameter is passed all schemas but meta-schemas are removed.
    // If RegExp is passed all schemas with key/id matching pattern but meta-schemas are removed.
    // Even if schema is referenced by other schemas it still can be removed as other schemas have local references.
    removeSchema(schemaKeyRef) {
        if (schemaKeyRef instanceof RegExp) {
            this._removeAllSchemas(this.schemas, schemaKeyRef);
            this._removeAllSchemas(this.refs, schemaKeyRef);
            return this;
        }
        switch (typeof schemaKeyRef) {
            case "undefined":
                this._removeAllSchemas(this.schemas);
                this._removeAllSchemas(this.refs);
                this._cache.clear();
                return this;
            case "string": {
                const sch = getSchEnv.call(this, schemaKeyRef);
                if (typeof sch == "object")
                    this._cache.del(sch.cacheKey);
                delete this.schemas[schemaKeyRef];
                delete this.refs[schemaKeyRef];
                return this;
            }
            case "object": {
                const cacheKey = this.opts.serialize(schemaKeyRef);
                this._cache.del(cacheKey);
                let id = schemaKeyRef.$id;
                if (id) {
                    id = resolve_1.normalizeId(id);
                    delete this.schemas[id];
                    delete this.refs[id];
                }
                return this;
            }
            default:
                throw new Error("ajv.removeSchema: invalid parameter");
        }
    }
    // add "vocabulary" - a collection of keywords
    addVocabulary(definitions) {
        for (const def of definitions)
            this.addKeyword(def);
        return this;
    }
    addKeyword(kwdOrDef, def // deprecated
    ) {
        let keyword;
        if (typeof kwdOrDef == "string") {
            keyword = kwdOrDef;
            if (typeof def == "object") {
                this.logger.warn("these parameters are deprecated, see docs for addKeyword");
                def.keyword = keyword;
            }
        }
        else if (typeof kwdOrDef == "object" && def === undefined) {
            def = kwdOrDef;
            keyword = def.keyword;
        }
        else {
            throw new Error("invalid addKeywords parameters");
        }
        checkKeyword.call(this, keyword, def);
        if (def)
            keywordMetaschema.call(this, def);
        util_1.eachItem(keyword, (kwd) => {
            util_1.eachItem(def === null || def === void 0 ? void 0 : def.type, (t) => addRule.call(this, kwd, t, def));
        });
        return this;
    }
    getKeyword(keyword) {
        const rule = this.RULES.all[keyword];
        return typeof rule == "object" ? rule.definition : !!rule;
    }
    // Remove keyword
    removeKeyword(keyword) {
        // TODO return type should be Ajv
        const { RULES } = this;
        delete RULES.keywords[keyword];
        delete RULES.all[keyword];
        for (const group of RULES.rules) {
            const i = group.rules.findIndex((rule) => rule.keyword === keyword);
            if (i >= 0)
                group.rules.splice(i, 1);
        }
        return this;
    }
    // Add format
    addFormat(name, format) {
        if (typeof format == "string")
            format = new RegExp(format);
        this.formats[name] = format;
        return this;
    }
    errorsText(errors = this.errors, // optional array of validation errors
    { separator = ", ", dataVar = "data" } = {} // optional options with properties `separator` and `dataVar`
    ) {
        if (!errors || errors.length === 0)
            return "No errors";
        return errors
            .map((e) => `${dataVar}${e.dataPath} ${e.message}`)
            .reduce((text, msg) => text + msg + separator);
    }
    $dataMetaSchema(metaSchema, keywordsJsonPointers) {
        const rules = this.RULES.all;
        for (const jsonPointer of keywordsJsonPointers) {
            metaSchema = JSON.parse(JSON.stringify(metaSchema));
            const segments = jsonPointer.split("/").slice(1); // first segment is an empty string
            let keywords = metaSchema;
            for (const seg of segments)
                keywords = keywords[seg];
            for (const key in rules) {
                const rule = rules[key];
                if (typeof rule != "object")
                    continue;
                const { $data } = rule.definition;
                const schema = keywords[key];
                if ($data && schema)
                    keywords[key] = schemaOrData(schema);
            }
        }
        return metaSchema;
    }
    _removeAllSchemas(schemas, regex) {
        for (const keyRef in schemas) {
            const sch = schemas[keyRef];
            if (!regex || regex.test(keyRef)) {
                if (typeof sch == "string") {
                    delete schemas[keyRef];
                }
                else if (sch && !sch.meta) {
                    this._cache.del(sch.cacheKey);
                    delete schemas[keyRef];
                }
            }
        }
    }
    _addSchema(schema, meta, validateSchema = this.opts.validateSchema, addSchema = this.opts.addUsedSchema) {
        if (typeof schema != "object" && typeof schema != "boolean") {
            throw new Error("schema must be object or boolean");
        }
        const cacheKey = this.opts.serialize(schema);
        let sch = this._cache.get(cacheKey);
        if (sch)
            return sch;
        const localRefs = resolve_1.getSchemaRefs.call(this, schema);
        sch = new compile_1.SchemaEnv({ schema, cacheKey, meta, localRefs });
        this._cache.put(sch.cacheKey, sch);
        const id = sch.baseId;
        if (addSchema && !id.startsWith("#")) {
            // TODO atm it is allowed to overwrite schemas without id (instead of not adding them)
            if (id)
                this._checkUnique(id);
            this.refs[id] = sch;
        }
        if (validateSchema)
            this.validateSchema(schema, true);
        return sch;
    }
    _checkUnique(id) {
        if (this.schemas[id] || this.refs[id]) {
            throw new Error(`schema with key or id "${id}" already exists`);
        }
    }
    _compileSchemaEnv(sch) {
        if (sch.meta)
            this._compileMetaSchema(sch);
        else
            compile_1.compileSchema.call(this, sch);
        if (!sch.validate)
            throw new Error("ajv implementation error");
        return sch.validate;
    }
    _compileMetaSchema(sch) {
        const currentOpts = this.opts;
        this.opts = this._metaOpts;
        try {
            compile_1.compileSchema.call(this, sch);
        }
        finally {
            this.opts = currentOpts;
        }
    }
}
exports.default = Ajv;
Ajv.ValidationError = error_classes_1.ValidationError;
Ajv.MissingRefError = error_classes_1.MissingRefError;
function checkDeprecatedOptions(opts) {
    if (opts.errorDataPath !== undefined)
        this.logger.error("NOT SUPPORTED: option errorDataPath");
    if (opts.schemaId !== undefined)
        this.logger.error("NOT SUPPORTED: option schemaId");
    if (opts.uniqueItems !== undefined)
        this.logger.error("NOT SUPPORTED: option uniqueItems");
    if (opts.jsPropertySyntax !== undefined)
        this.logger.warn("DEPRECATED: option jsPropertySyntax");
    if (opts.unicode !== undefined)
        this.logger.warn("DEPRECATED: option unicode");
}
function defaultMeta() {
    const { meta } = this.opts;
    this.opts.defaultMeta =
        typeof meta == "object"
            ? meta.$id || meta
            : this.getSchema(META_SCHEMA_ID)
                ? META_SCHEMA_ID
                : undefined;
    return this.opts.defaultMeta;
}
function getSchEnv(keyRef) {
    keyRef = resolve_1.normalizeId(keyRef); // TODO tests fail without this line
    return this.schemas[keyRef] || this.refs[keyRef];
}
function addDefaultMetaSchema() {
    const { $data, meta } = this.opts;
    if ($data)
        this.addMetaSchema(data_json_1.default, data_json_1.default.$id, false);
    if (meta === false)
        return;
    const metaSchema = $data
        ? this.$dataMetaSchema(json_schema_draft_07_json_1.default, META_SUPPORT_DATA)
        : json_schema_draft_07_json_1.default;
    this.addMetaSchema(metaSchema, META_SCHEMA_ID, false);
    this.refs["http://json-schema.org/schema"] = META_SCHEMA_ID;
}
function addInitialSchemas() {
    const optsSchemas = this.opts.schemas;
    if (!optsSchemas)
        return;
    if (Array.isArray(optsSchemas))
        this.addSchema(optsSchemas);
    else
        for (const key in optsSchemas)
            this.addSchema(optsSchemas[key], key);
}
function addInitialFormats() {
    for (const name in this.opts.formats) {
        const format = this.opts.formats[name];
        this.addFormat(name, format);
    }
}
function addInitialKeywords(defs) {
    if (Array.isArray(defs)) {
        this.addVocabulary(defs);
        return;
    }
    this.logger.warn("keywords option as map is deprecated, pass array");
    for (const keyword in defs) {
        const def = defs[keyword];
        if (!def.keyword)
            def.keyword = keyword;
        this.addKeyword(def);
    }
}
function getMetaSchemaOptions() {
    const metaOpts = { ...this.opts };
    for (const opt of META_IGNORE_OPTIONS)
        delete metaOpts[opt];
    return metaOpts;
}
const noLogs = { log() { }, warn() { }, error() { } };
function getLogger(logger) {
    if (logger === false)
        return noLogs;
    if (logger === undefined)
        return console;
    if (logger.log && logger.warn && logger.error)
        return logger;
    throw new Error("logger must implement log, warn and error methods");
}
const KEYWORD_NAME = /^[a-z_$][a-z0-9_$-]*$/i;
function checkKeyword(keyword, def) {
    const { RULES } = this;
    util_1.eachItem(keyword, (kwd) => {
        if (RULES.keywords[kwd])
            throw new Error(`Keyword ${kwd} is already defined`);
        if (!KEYWORD_NAME.test(kwd))
            throw new Error(`Keyword ${kwd} has invalid name`);
    });
    if (!def)
        return;
    if (def.type)
        util_1.eachItem(def.type, (t) => dataType_1.checkType(t, RULES));
    if (def.$data && !("code" in def || "validate" in def)) {
        throw new Error('$data keyword must have "code" or "validate" function');
    }
}
function addRule(keyword, dataType, definition) {
    var _a;
    const { RULES } = this;
    let ruleGroup = RULES.rules.find(({ type: t }) => t === dataType);
    if (!ruleGroup) {
        ruleGroup = { type: dataType, rules: [] };
        RULES.rules.push(ruleGroup);
    }
    RULES.keywords[keyword] = true;
    if (!definition)
        return;
    const rule = { keyword, definition };
    if (definition.before)
        addBeforeRule.call(this, ruleGroup, rule, definition.before);
    else
        ruleGroup.rules.push(rule);
    RULES.all[keyword] = rule;
    (_a = definition.implements) === null || _a === void 0 ? void 0 : _a.forEach((kwd) => this.addKeyword(kwd));
}
function addBeforeRule(ruleGroup, rule, before) {
    const i = ruleGroup.rules.findIndex((_rule) => _rule.keyword === before);
    if (i >= 0) {
        ruleGroup.rules.splice(i, 0, rule);
    }
    else {
        ruleGroup.rules.push(rule);
        this.logger.warn(`rule ${before} is not defined`);
    }
}
function keywordMetaschema(def) {
    let { metaSchema } = def;
    if (metaSchema === undefined)
        return;
    if (def.$data && this.opts.$data)
        metaSchema = schemaOrData(metaSchema);
    def.validateSchema = this.compile(metaSchema, true);
}
const $dataRef = {
    $ref: "https://raw.githubusercontent.com/ajv-validator/ajv/master/lib/refs/data.json#",
};
function schemaOrData(schema) {
    return { anyOf: [schema, $dataRef] };
}
module.exports = Ajv;
module.exports.default = Ajv;
//# sourceMappingURL=ajv.js.map