"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.or = exports.and = exports.CodeGen = exports.varKinds = exports.operators = exports.ValueScope = exports.Scope = exports.Name = exports.stringify = exports.getProperty = exports.nil = exports.str = exports._ = void 0;
const code_1 = require("./code");
const scope_1 = require("./scope");
var code_2 = require("./code");
Object.defineProperty(exports, "_", { enumerable: true, get: function () { return code_2._; } });
Object.defineProperty(exports, "str", { enumerable: true, get: function () { return code_2.str; } });
Object.defineProperty(exports, "nil", { enumerable: true, get: function () { return code_2.nil; } });
Object.defineProperty(exports, "getProperty", { enumerable: true, get: function () { return code_2.getProperty; } });
Object.defineProperty(exports, "stringify", { enumerable: true, get: function () { return code_2.stringify; } });
Object.defineProperty(exports, "Name", { enumerable: true, get: function () { return code_2.Name; } });
var scope_2 = require("./scope");
Object.defineProperty(exports, "Scope", { enumerable: true, get: function () { return scope_2.Scope; } });
Object.defineProperty(exports, "ValueScope", { enumerable: true, get: function () { return scope_2.ValueScope; } });
var BlockKind;
(function (BlockKind) {
    BlockKind[BlockKind["If"] = 0] = "If";
    BlockKind[BlockKind["Else"] = 1] = "Else";
    BlockKind[BlockKind["For"] = 2] = "For";
    BlockKind[BlockKind["Func"] = 3] = "Func";
})(BlockKind || (BlockKind = {}));
exports.operators = {
    GT: new code_1._Code(">"),
    GTE: new code_1._Code(">="),
    LT: new code_1._Code("<"),
    LTE: new code_1._Code("<="),
    EQ: new code_1._Code("==="),
    NEQ: new code_1._Code("!=="),
    NOT: new code_1._Code("!"),
    OR: new code_1._Code("||"),
    AND: new code_1._Code("&&"),
};
exports.varKinds = {
    const: new code_1.Name("const"),
    let: new code_1.Name("let"),
    var: new code_1.Name("var"),
};
class CodeGen {
    constructor(extScope, opts = {}) {
        this._values = {};
        this._blocks = [];
        this._blockStarts = [];
        this._out = "";
        this.opts = opts;
        this._extScope = extScope;
        this._scope = new scope_1.Scope({ parent: extScope });
        this._n = opts.lines ? "\n" : "";
    }
    toString() {
        return this._out;
    }
    name(prefix) {
        return this._scope.name(prefix);
    }
    scopeName(prefix) {
        return this._extScope.name(prefix);
    }
    scopeValue(prefixOrName, value) {
        const name = this._extScope.value(prefixOrName, value);
        const vs = this._values[name.prefix] || (this._values[name.prefix] = new Set());
        vs.add(name);
        return name;
    }
    getScopeValue(prefix, keyOrRef) {
        return this._extScope.getValue(prefix, keyOrRef);
    }
    scopeRefs(scopeName) {
        return this._extScope.scopeRefs(scopeName, this._values);
    }
    _def(varKind, nameOrPrefix, rhs) {
        const name = this._scope.toName(nameOrPrefix);
        if (this.opts.es5)
            varKind = exports.varKinds.var;
        if (rhs === undefined)
            this._out += `${varKind} ${name};` + this._n;
        else
            this._out += `${varKind} ${name} = ${rhs};` + this._n;
        return name;
    }
    const(nameOrPrefix, rhs) {
        return this._def(exports.varKinds.const, nameOrPrefix, rhs);
    }
    let(nameOrPrefix, rhs) {
        return this._def(exports.varKinds.let, nameOrPrefix, rhs);
    }
    var(nameOrPrefix, rhs) {
        return this._def(exports.varKinds.var, nameOrPrefix, rhs);
    }
    assign(name, rhs) {
        this._out += `${name} = ${rhs};` + this._n;
        return this;
    }
    code(c) {
        if (typeof c == "function")
            c();
        else
            this._out += `${c};${this._n}`;
        return this;
    }
    if(condition, thenBody, elseBody) {
        this._blocks.push(BlockKind.If);
        this._out += `if(${condition}){` + this._n;
        if (thenBody && elseBody) {
            this.code(thenBody).else().code(elseBody).endIf();
        }
        else if (thenBody) {
            this.code(thenBody).endIf();
        }
        else if (elseBody) {
            throw new Error('CodeGen: "else" body without "then" body');
        }
        return this;
    }
    ifNot(condition, thenBody, elseBody) {
        const cond = new code_1._Code(condition instanceof code_1.Name ? `!${condition}` : `!(${condition})`);
        return this.if(cond, thenBody, elseBody);
    }
    elseIf(condition) {
        if (this._lastBlock !== BlockKind.If)
            throw new Error('CodeGen: "else if" without "if"');
        this._out += `}else if(${condition}){` + this._n;
        return this;
    }
    else() {
        if (this._lastBlock !== BlockKind.If)
            throw new Error('CodeGen: "else" without "if"');
        this._lastBlock = BlockKind.Else;
        this._out += "}else{" + this._n;
        return this;
    }
    endIf() {
        // TODO possibly remove empty branches here
        const b = this._lastBlock;
        if (b !== BlockKind.If && b !== BlockKind.Else)
            throw new Error('CodeGen: "endIf" without "if"');
        this._blocks.pop();
        this._out += "}" + this._n;
        return this;
    }
    for(iteration, forBody) {
        this._blocks.push(BlockKind.For);
        this._out += `for(${iteration}){` + this._n;
        if (forBody)
            this.code(forBody).endFor();
        return this;
    }
    forRange(nameOrPrefix, from, to, forBody, varKind = exports.varKinds.let) {
        const i = this._scope.toName(nameOrPrefix);
        if (this.opts.es5)
            varKind = exports.varKinds.var;
        return this._loop(code_1._ `for(${varKind} ${i}=${from}; ${i}<${to}; ${i}++){`, i, forBody);
    }
    forOf(nameOrPrefix, iterable, forBody, varKind = exports.varKinds.const) {
        const name = this._scope.toName(nameOrPrefix);
        if (this.opts.es5) {
            const arr = iterable instanceof code_1.Name ? iterable : this.var("arr", iterable);
            return this.forRange("_i", 0, new code_1._Code(`${arr}.length`), (i) => {
                this.var(name, new code_1._Code(`${arr}[${i}]`));
                forBody(name);
            });
        }
        return this._loop(code_1._ `for(${varKind} ${name} of ${iterable}){`, name, forBody);
    }
    forIn(nameOrPrefix, obj, forBody, varKind = exports.varKinds.const) {
        if (this.opts.forInOwn) {
            return this.forOf(nameOrPrefix, new code_1._Code(`Object.keys(${obj})`), forBody);
        }
        const name = this._scope.toName(nameOrPrefix);
        return this._loop(code_1._ `for(${varKind} ${name} in ${obj}){`, name, forBody);
    }
    _loop(forCode, name, forBody) {
        this._blocks.push(BlockKind.For);
        this._out += `${forCode}${this._n}`;
        forBody(name);
        this.endFor();
        return this;
    }
    endFor() {
        const b = this._lastBlock;
        if (b !== BlockKind.For)
            throw new Error('CodeGen: "endFor" without "for"');
        this._blocks.pop();
        this._out += "}" + this._n;
        return this;
    }
    label(label) {
        this._out += `${label}:${this._n}`;
        return this;
    }
    break(label) {
        this._out += (label ? `break ${label};` : "break;") + this._n;
        return this;
    }
    return(value) {
        this._out += "return ";
        this.code(value);
        this._out += ";" + this._n;
        return this;
    }
    try(tryBody, catchCode, finallyCode) {
        if (!catchCode && !finallyCode)
            throw new Error('CodeGen: "try" without "catch" and "finally"');
        this._out += "try{" + this._n;
        this.code(tryBody);
        if (catchCode) {
            const err = this.name("e");
            this._out += `}catch(${err}){` + this._n;
            catchCode(err);
        }
        if (finallyCode) {
            this._out += "}finally{" + this._n;
            this.code(finallyCode);
        }
        this._out += "}" + this._n;
        return this;
    }
    throw(err) {
        this._out += `throw ${err};` + this._n;
        return this;
    }
    block(body, expectedToClose) {
        this._blockStarts.push(this._blocks.length);
        if (body)
            this.code(body).endBlock(expectedToClose);
        return this;
    }
    endBlock(expectedToClose) {
        // TODO maybe close blocks one by one, eliminating empty branches
        const len = this._blockStarts.pop();
        if (len === undefined)
            throw new Error("CodeGen: not in block sequence");
        const toClose = this._blocks.length - len;
        if (toClose < 0 || (expectedToClose !== undefined && toClose !== expectedToClose)) {
            throw new Error("CodeGen: block sequence already ended or incorrect number of blocks");
        }
        this._blocks.length = len;
        if (toClose > 0)
            this._out += "}".repeat(toClose) + this._n;
        return this;
    }
    func(name, args = code_1.nil, async, funcBody) {
        this._blocks.push(BlockKind.Func);
        this._out += `${async ? "async " : ""}function ${name}(${args}){` + this._n;
        if (funcBody)
            this.code(funcBody).endFunc();
        return this;
    }
    endFunc() {
        const b = this._lastBlock;
        if (b !== BlockKind.Func)
            throw new Error('CodeGen: "endFunc" without "func"');
        this._blocks.pop();
        this._out += "}" + this._n;
        return this;
    }
    get _lastBlock() {
        return this._blocks[this._last()];
    }
    set _lastBlock(b) {
        this._blocks[this._last()] = b;
    }
    _last() {
        const len = this._blocks.length;
        if (len === 0)
            throw new Error("CodeGen: not in block");
        return len - 1;
    }
}
exports.CodeGen = CodeGen;
const andCode = mappend(exports.operators.AND);
function and(...args) {
    return args.reduce(andCode);
}
exports.and = and;
const orCode = mappend(exports.operators.OR);
function or(...args) {
    return args.reduce(orCode);
}
exports.or = or;
function mappend(op) {
    return (x, y) => (x === code_1.nil ? y : y === code_1.nil ? x : new code_1._Code(`${x} ${op} ${y}`));
}
//# sourceMappingURL=index.js.map