"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.subschemaCode = exports.validateFunctionCode = void 0;
const boolSchema_1 = require("./boolSchema");
const dataType_1 = require("./dataType");
const iterate_1 = require("./iterate");
const codegen_1 = require("../codegen");
const names_1 = __importDefault(require("../names"));
const resolve_1 = require("../resolve");
const util_1 = require("../util");
const util_2 = require("../../vocabularies/util");
// schema compilation - generates validation function, subschemaCode (below) is used for subschemas
function validateFunctionCode(it) {
    if (isSchemaObj(it)) {
        checkKeywords(it);
        if (util_1.schemaCxtHasRules(it)) {
            topSchemaObjCode(it);
            return;
        }
    }
    validateFunction(it, () => boolSchema_1.topBoolOrEmptySchema(it));
}
exports.validateFunctionCode = validateFunctionCode;
function validateFunction({ gen, validateName, schema, schemaEnv, opts }, body) {
    gen.return(() => gen.func(validateName, codegen_1._ `${names_1.default.data}, ${names_1.default.dataPath}, ${names_1.default.parentData}, ${names_1.default.parentDataProperty}, ${names_1.default.rootData}`, schemaEnv.$async, () => gen.code(codegen_1._ `"use strict"; ${funcSourceUrl(schema, opts)}`).code(body)));
}
function topSchemaObjCode(it) {
    const { schema, opts } = it;
    validateFunction(it, () => {
        if (opts.$comment && schema.$comment)
            commentKeyword(it);
        checkNoDefault(it);
        initializeTop(it.gen);
        typeAndKeywords(it);
        returnResults(it);
    });
    return;
}
function funcSourceUrl(schema, opts) {
    return typeof schema == "object" && schema.$id && (opts.sourceCode || opts.processCode)
        ? codegen_1._ `/*# sourceURL=${schema.$id} */`
        : codegen_1.nil;
}
// schema compilation - this function is used recursively to generate code for sub-schemas
function subschemaCode(it, valid) {
    if (isSchemaObj(it)) {
        checkKeywords(it);
        if (util_1.schemaCxtHasRules(it)) {
            subSchemaObjCode(it, valid);
            return;
        }
    }
    boolSchema_1.boolOrEmptySchema(it, valid);
}
exports.subschemaCode = subschemaCode;
function isSchemaObj(it) {
    return typeof it.schema != "boolean";
}
function subSchemaObjCode(it, valid) {
    const { schema, gen, opts } = it;
    if (opts.$comment && schema.$comment)
        commentKeyword(it);
    updateContext(it);
    checkAsync(it);
    // TODO var - async validation fails if var replaced, possibly because of nodent
    const errsCount = gen.var("_errs", names_1.default.errors);
    typeAndKeywords(it, errsCount);
    // TODO var
    gen.var(valid, codegen_1._ `${errsCount} === ${names_1.default.errors}`);
}
function checkKeywords(it) {
    util_2.checkUnknownRules(it);
    checkRefsAndKeywords(it);
}
function typeAndKeywords(it, errsCount) {
    const types = dataType_1.getSchemaTypes(it, it.schema);
    const checkedTypes = dataType_1.coerceAndCheckDataType(it, types);
    iterate_1.schemaKeywords(it, types, !checkedTypes, errsCount);
}
function checkRefsAndKeywords(it) {
    const { schema, errSchemaPath, opts, self } = it;
    if (schema.$ref && util_1.schemaHasRulesButRef(schema, self.RULES)) {
        if (opts.extendRefs === "fail") {
            throw new Error(`$ref: sibling validation keywords at "${errSchemaPath}" (option extendRefs)`);
        }
        else if (opts.extendRefs !== true) {
            self.logger.warn(`$ref: keywords ignored in schema at path "${errSchemaPath}"`);
        }
    }
}
function checkNoDefault(it) {
    const { schema, opts } = it;
    if (schema.default !== undefined && opts.useDefaults && opts.strict) {
        util_2.checkStrictMode(it, "default is ignored in the schema root");
    }
}
function initializeTop(gen) {
    gen.let(names_1.default.vErrors, null);
    gen.let(names_1.default.errors, 0);
    gen.if(codegen_1._ `${names_1.default.rootData} === undefined`, () => gen.assign(names_1.default.rootData, names_1.default.data));
    // gen.if(_`${N.dataPath} === undefined`, () => gen.assign(N.dataPath, _`""`)) // TODO maybe add it
}
function updateContext(it) {
    if (it.schema.$id)
        it.baseId = resolve_1.resolveUrl(it.baseId, it.schema.$id);
}
function checkAsync(it) {
    if (it.schema.$async && !it.schemaEnv.$async)
        throw new Error("async schema in sync schema");
}
function commentKeyword({ gen, schemaEnv, schema, errSchemaPath, opts }) {
    const msg = schema.$comment;
    if (opts.$comment === true) {
        gen.code(codegen_1._ `${names_1.default.self}.logger.log(${msg})`);
    }
    else if (typeof opts.$comment == "function") {
        const schemaPath = codegen_1.str `${errSchemaPath}/$comment`;
        const rootName = gen.scopeValue("root", { ref: schemaEnv.root });
        gen.code(codegen_1._ `${names_1.default.self}.opts.$comment(${msg}, ${schemaPath}, ${rootName}.schema)`);
    }
}
function returnResults({ gen, schemaEnv, validateName, ValidationError }) {
    if (schemaEnv.$async) {
        gen.if(codegen_1._ `${names_1.default.errors} === 0`, () => gen.return(names_1.default.data), codegen_1._ `throw new ${ValidationError}(${names_1.default.vErrors})`);
    }
    else {
        gen.assign(codegen_1._ `${validateName}.errors`, names_1.default.vErrors);
        gen.return(codegen_1._ `${names_1.default.errors} === 0`);
    }
}
//# sourceMappingURL=index.js.map