import { isJsObject, global, isPresent, isBlank, isArray, getSymbolIterator } from 'angular2/src/facade/lang';
export var Map = global.Map;
export var Set = global.Set;
// Safari and Internet Explorer do not support the iterable parameter to the
// Map constructor.  We work around that by manually adding the items.
var createMapFromPairs = (function () {
    try {
        if (new Map([[1, 2]]).size === 1) {
            return function createMapFromPairs(pairs) { return new Map(pairs); };
        }
    }
    catch (e) {
    }
    return function createMapAndPopulateFromPairs(pairs) {
        var map = new Map();
        for (var i = 0; i < pairs.length; i++) {
            var pair = pairs[i];
            map.set(pair[0], pair[1]);
        }
        return map;
    };
})();
var createMapFromMap = (function () {
    try {
        if (new Map(new Map())) {
            return function createMapFromMap(m) { return new Map(m); };
        }
    }
    catch (e) {
    }
    return function createMapAndPopulateFromMap(m) {
        var map = new Map();
        m.forEach((v, k) => { map.set(k, v); });
        return map;
    };
})();
var _clearValues = (function () {
    if ((new Map()).keys().next) {
        return function _clearValues(m) {
            var keyIterator = m.keys();
            var k;
            while (!((k = keyIterator.next()).done)) {
                m.set(k.value, null);
            }
        };
    }
    else {
        return function _clearValuesWithForeEach(m) {
            m.forEach((v, k) => { m.set(k, null); });
        };
    }
})();
// Safari doesn't implement MapIterator.next(), which is used is Traceur's polyfill of Array.from
// TODO(mlaval): remove the work around once we have a working polyfill of Array.from
var _arrayFromMap = (function () {
    try {
        if ((new Map()).values().next) {
            return function createArrayFromMap(m, getValues) {
                return getValues ? Array.from(m.values()) : Array.from(m.keys());
            };
        }
    }
    catch (e) {
    }
    return function createArrayFromMapWithForeach(m, getValues) {
        var res = ListWrapper.createFixedSize(m.size), i = 0;
        m.forEach((v, k) => {
            res[i] = getValues ? v : k;
            i++;
        });
        return res;
    };
})();
export class MapWrapper {
    static clone(m) { return createMapFromMap(m); }
    static createFromStringMap(stringMap) {
        var result = new Map();
        for (var prop in stringMap) {
            result.set(prop, stringMap[prop]);
        }
        return result;
    }
    static toStringMap(m) {
        var r = {};
        m.forEach((v, k) => r[k] = v);
        return r;
    }
    static createFromPairs(pairs) { return createMapFromPairs(pairs); }
    static clearValues(m) { _clearValues(m); }
    static iterable(m) { return m; }
    static keys(m) { return _arrayFromMap(m, false); }
    static values(m) { return _arrayFromMap(m, true); }
}
/**
 * Wraps Javascript Objects
 */
export class StringMapWrapper {
    static create() {
        // Note: We are not using Object.create(null) here due to
        // performance!
        // http://jsperf.com/ng2-object-create-null
        return {};
    }
    static contains(map, key) {
        return map.hasOwnProperty(key);
    }
    static get(map, key) {
        return map.hasOwnProperty(key) ? map[key] : undefined;
    }
    static set(map, key, value) { map[key] = value; }
    static keys(map) { return Object.keys(map); }
    static isEmpty(map) {
        for (var prop in map) {
            return false;
        }
        return true;
    }
    static delete(map, key) { delete map[key]; }
    static forEach(map, callback) {
        for (var prop in map) {
            if (map.hasOwnProperty(prop)) {
                callback(map[prop], prop);
            }
        }
    }
    static merge(m1, m2) {
        var m = {};
        for (var attr in m1) {
            if (m1.hasOwnProperty(attr)) {
                m[attr] = m1[attr];
            }
        }
        for (var attr in m2) {
            if (m2.hasOwnProperty(attr)) {
                m[attr] = m2[attr];
            }
        }
        return m;
    }
    static equals(m1, m2) {
        var k1 = Object.keys(m1);
        var k2 = Object.keys(m2);
        if (k1.length != k2.length) {
            return false;
        }
        var key;
        for (var i = 0; i < k1.length; i++) {
            key = k1[i];
            if (m1[key] !== m2[key]) {
                return false;
            }
        }
        return true;
    }
}
export class ListWrapper {
    // JS has no way to express a statically fixed size list, but dart does so we
    // keep both methods.
    static createFixedSize(size) { return new Array(size); }
    static createGrowableSize(size) { return new Array(size); }
    static clone(array) { return array.slice(0); }
    static forEachWithIndex(array, fn) {
        for (var i = 0; i < array.length; i++) {
            fn(array[i], i);
        }
    }
    static first(array) {
        if (!array)
            return null;
        return array[0];
    }
    static last(array) {
        if (!array || array.length == 0)
            return null;
        return array[array.length - 1];
    }
    static indexOf(array, value, startIndex = 0) {
        return array.indexOf(value, startIndex);
    }
    static contains(list, el) { return list.indexOf(el) !== -1; }
    static reversed(array) {
        var a = ListWrapper.clone(array);
        return a.reverse();
    }
    static concat(a, b) { return a.concat(b); }
    static insert(list, index, value) { list.splice(index, 0, value); }
    static removeAt(list, index) {
        var res = list[index];
        list.splice(index, 1);
        return res;
    }
    static removeAll(list, items) {
        for (var i = 0; i < items.length; ++i) {
            var index = list.indexOf(items[i]);
            list.splice(index, 1);
        }
    }
    static remove(list, el) {
        var index = list.indexOf(el);
        if (index > -1) {
            list.splice(index, 1);
            return true;
        }
        return false;
    }
    static clear(list) { list.length = 0; }
    static isEmpty(list) { return list.length == 0; }
    static fill(list, value, start = 0, end = null) {
        list.fill(value, start, end === null ? list.length : end);
    }
    static equals(a, b) {
        if (a.length != b.length)
            return false;
        for (var i = 0; i < a.length; ++i) {
            if (a[i] !== b[i])
                return false;
        }
        return true;
    }
    static slice(l, from = 0, to = null) {
        return l.slice(from, to === null ? undefined : to);
    }
    static splice(l, from, length) { return l.splice(from, length); }
    static sort(l, compareFn) {
        if (isPresent(compareFn)) {
            l.sort(compareFn);
        }
        else {
            l.sort();
        }
    }
    static toString(l) { return l.toString(); }
    static toJSON(l) { return JSON.stringify(l); }
    static maximum(list, predicate) {
        if (list.length == 0) {
            return null;
        }
        var solution = null;
        var maxValue = -Infinity;
        for (var index = 0; index < list.length; index++) {
            var candidate = list[index];
            if (isBlank(candidate)) {
                continue;
            }
            var candidateValue = predicate(candidate);
            if (candidateValue > maxValue) {
                solution = candidate;
                maxValue = candidateValue;
            }
        }
        return solution;
    }
}
export function isListLikeIterable(obj) {
    if (!isJsObject(obj))
        return false;
    return isArray(obj) ||
        (!(obj instanceof Map) &&
            getSymbolIterator() in obj); // JS Iterable have a Symbol.iterator prop
}
export function iterateListLike(obj, fn) {
    if (isArray(obj)) {
        for (var i = 0; i < obj.length; i++) {
            fn(obj[i]);
        }
    }
    else {
        var iterator = obj[getSymbolIterator()]();
        var item;
        while (!((item = iterator.next()).done)) {
            fn(item.value);
        }
    }
}
// Safari and Internet Explorer do not support the iterable parameter to the
// Set constructor.  We work around that by manually adding the items.
var createSetFromList = (function () {
    var test = new Set([1, 2, 3]);
    if (test.size === 3) {
        return function createSetFromList(lst) { return new Set(lst); };
    }
    else {
        return function createSetAndPopulateFromList(lst) {
            var res = new Set(lst);
            if (res.size !== lst.length) {
                for (var i = 0; i < lst.length; i++) {
                    res.add(lst[i]);
                }
            }
            return res;
        };
    }
})();
export class SetWrapper {
    static createFromList(lst) { return createSetFromList(lst); }
    static has(s, key) { return s.has(key); }
    static delete(m, k) { m.delete(k); }
}
//# sourceMappingURL=data:application/json;base64,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