'use strict';var lang_1 = require('angular2/src/facade/lang');
var codegen_facade_1 = require('./codegen_facade');
var proto_record_1 = require('./proto_record');
var constants_1 = require('./constants');
var exceptions_1 = require('angular2/src/facade/exceptions');
/**
 * Class responsible for providing change detection logic for change detector classes.
 */
var CodegenLogicUtil = (function () {
    function CodegenLogicUtil(_names, _utilName, _changeDetectorStateName, _changeDetection) {
        this._names = _names;
        this._utilName = _utilName;
        this._changeDetectorStateName = _changeDetectorStateName;
        this._changeDetection = _changeDetection;
    }
    /**
     * Generates a statement which updates the local variable representing `protoRec` with the current
     * value of the record. Used by property bindings.
     */
    CodegenLogicUtil.prototype.genPropertyBindingEvalValue = function (protoRec) {
        var _this = this;
        return this._genEvalValue(protoRec, function (idx) { return _this._names.getLocalName(idx); }, this._names.getLocalsAccessorName());
    };
    /**
     * Generates a statement which updates the local variable representing `protoRec` with the current
     * value of the record. Used by event bindings.
     */
    CodegenLogicUtil.prototype.genEventBindingEvalValue = function (eventRecord, protoRec) {
        var _this = this;
        return this._genEvalValue(protoRec, function (idx) { return _this._names.getEventLocalName(eventRecord, idx); }, "locals");
    };
    CodegenLogicUtil.prototype._genEvalValue = function (protoRec, getLocalName, localsAccessor) {
        var context = (protoRec.contextIndex == -1) ?
            this._names.getDirectiveName(protoRec.directiveIndex) :
            getLocalName(protoRec.contextIndex);
        var argString = protoRec.args.map(function (arg) { return getLocalName(arg); }).join(", ");
        var rhs;
        switch (protoRec.mode) {
            case proto_record_1.RecordType.Self:
                rhs = context;
                break;
            case proto_record_1.RecordType.Const:
                rhs = codegen_facade_1.codify(protoRec.funcOrValue);
                break;
            case proto_record_1.RecordType.PropertyRead:
                rhs = this._observe(context + "." + protoRec.name, protoRec);
                break;
            case proto_record_1.RecordType.SafeProperty:
                var read = this._observe(context + "." + protoRec.name, protoRec);
                rhs =
                    this._utilName + ".isValueBlank(" + context + ") ? null : " + this._observe(read, protoRec);
                break;
            case proto_record_1.RecordType.PropertyWrite:
                rhs = context + "." + protoRec.name + " = " + getLocalName(protoRec.args[0]);
                break;
            case proto_record_1.RecordType.Local:
                rhs = this._observe(localsAccessor + ".get(" + codegen_facade_1.rawString(protoRec.name) + ")", protoRec);
                break;
            case proto_record_1.RecordType.InvokeMethod:
                rhs = this._observe(context + "." + protoRec.name + "(" + argString + ")", protoRec);
                break;
            case proto_record_1.RecordType.SafeMethodInvoke:
                var invoke = context + "." + protoRec.name + "(" + argString + ")";
                rhs =
                    this._utilName + ".isValueBlank(" + context + ") ? null : " + this._observe(invoke, protoRec);
                break;
            case proto_record_1.RecordType.InvokeClosure:
                rhs = context + "(" + argString + ")";
                break;
            case proto_record_1.RecordType.PrimitiveOp:
                rhs = this._utilName + "." + protoRec.name + "(" + argString + ")";
                break;
            case proto_record_1.RecordType.CollectionLiteral:
                rhs = this._utilName + "." + protoRec.name + "(" + argString + ")";
                break;
            case proto_record_1.RecordType.Interpolate:
                rhs = this._genInterpolation(protoRec);
                break;
            case proto_record_1.RecordType.KeyedRead:
                rhs = this._observe(context + "[" + getLocalName(protoRec.args[0]) + "]", protoRec);
                break;
            case proto_record_1.RecordType.KeyedWrite:
                rhs = context + "[" + getLocalName(protoRec.args[0]) + "] = " + getLocalName(protoRec.args[1]);
                break;
            case proto_record_1.RecordType.Chain:
                rhs = 'null';
                break;
            default:
                throw new exceptions_1.BaseException("Unknown operation " + protoRec.mode);
        }
        return getLocalName(protoRec.selfIndex) + " = " + rhs + ";";
    };
    /** @internal */
    CodegenLogicUtil.prototype._observe = function (exp, rec) {
        // This is an experimental feature. Works only in Dart.
        if (this._changeDetection === constants_1.ChangeDetectionStrategy.OnPushObserve) {
            return "this.observeValue(" + exp + ", " + rec.selfIndex + ")";
        }
        else {
            return exp;
        }
    };
    CodegenLogicUtil.prototype.genPropertyBindingTargets = function (propertyBindingTargets, genDebugInfo) {
        var _this = this;
        var bs = propertyBindingTargets.map(function (b) {
            if (lang_1.isBlank(b))
                return "null";
            var debug = genDebugInfo ? codegen_facade_1.codify(b.debug) : "null";
            return _this._utilName + ".bindingTarget(" + codegen_facade_1.codify(b.mode) + ", " + b.elementIndex + ", " + codegen_facade_1.codify(b.name) + ", " + codegen_facade_1.codify(b.unit) + ", " + debug + ")";
        });
        return "[" + bs.join(", ") + "]";
    };
    CodegenLogicUtil.prototype.genDirectiveIndices = function (directiveRecords) {
        var _this = this;
        var bs = directiveRecords.map(function (b) {
            return (_this._utilName + ".directiveIndex(" + b.directiveIndex.elementIndex + ", " + b.directiveIndex.directiveIndex + ")");
        });
        return "[" + bs.join(", ") + "]";
    };
    /** @internal */
    CodegenLogicUtil.prototype._genInterpolation = function (protoRec) {
        var iVals = [];
        for (var i = 0; i < protoRec.args.length; ++i) {
            iVals.push(codegen_facade_1.codify(protoRec.fixedArgs[i]));
            iVals.push(this._utilName + ".s(" + this._names.getLocalName(protoRec.args[i]) + ")");
        }
        iVals.push(codegen_facade_1.codify(protoRec.fixedArgs[protoRec.args.length]));
        return codegen_facade_1.combineGeneratedStrings(iVals);
    };
    CodegenLogicUtil.prototype.genHydrateDirectives = function (directiveRecords) {
        var _this = this;
        var res = [];
        for (var i = 0; i < directiveRecords.length; ++i) {
            var r = directiveRecords[i];
            var dirVarName = this._names.getDirectiveName(r.directiveIndex);
            res.push(dirVarName + " = " + this._genReadDirective(i) + ";");
            if (lang_1.isPresent(r.outputs)) {
                r.outputs.forEach(function (output) {
                    var eventHandlerExpr = _this._genEventHandler(r.directiveIndex.elementIndex, output[1]);
                    if (lang_1.IS_DART) {
                        res.push(dirVarName + "." + output[0] + ".listen(" + eventHandlerExpr + ");");
                    }
                    else {
                        res.push(dirVarName + "." + output[0] + ".subscribe({next: " + eventHandlerExpr + "});");
                    }
                });
            }
        }
        return res.join("\n");
    };
    CodegenLogicUtil.prototype.genDirectivesOnDestroy = function (directiveRecords) {
        var res = [];
        for (var i = 0; i < directiveRecords.length; ++i) {
            var r = directiveRecords[i];
            if (r.callOnDestroy) {
                var dirVarName = this._names.getDirectiveName(r.directiveIndex);
                res.push(dirVarName + ".ngOnDestroy();");
            }
        }
        return res.join("\n");
    };
    CodegenLogicUtil.prototype._genEventHandler = function (boundElementIndex, eventName) {
        if (lang_1.IS_DART) {
            return "(event) => this.handleEvent('" + eventName + "', " + boundElementIndex + ", event)";
        }
        else {
            return "(function(event) { return this.handleEvent('" + eventName + "', " + boundElementIndex + ", event); }).bind(this)";
        }
    };
    CodegenLogicUtil.prototype._genReadDirective = function (index) {
        var directiveExpr = "this.getDirectiveFor(directives, " + index + ")";
        // This is an experimental feature. Works only in Dart.
        if (this._changeDetection === constants_1.ChangeDetectionStrategy.OnPushObserve) {
            return "this.observeDirective(" + directiveExpr + ", " + index + ")";
        }
        else {
            return directiveExpr;
        }
    };
    CodegenLogicUtil.prototype.genHydrateDetectors = function (directiveRecords) {
        var res = [];
        for (var i = 0; i < directiveRecords.length; ++i) {
            var r = directiveRecords[i];
            if (!r.isDefaultChangeDetection()) {
                res.push(this._names.getDetectorName(r.directiveIndex) + " = this.getDetectorFor(directives, " + i + ");");
            }
        }
        return res.join("\n");
    };
    CodegenLogicUtil.prototype.genContentLifecycleCallbacks = function (directiveRecords) {
        var res = [];
        var eq = lang_1.IS_DART ? '==' : '===';
        // NOTE(kegluneq): Order is important!
        for (var i = directiveRecords.length - 1; i >= 0; --i) {
            var dir = directiveRecords[i];
            if (dir.callAfterContentInit) {
                res.push("if(" + this._names.getStateName() + " " + eq + " " + this._changeDetectorStateName + ".NeverChecked) " + this._names.getDirectiveName(dir.directiveIndex) + ".ngAfterContentInit();");
            }
            if (dir.callAfterContentChecked) {
                res.push(this._names.getDirectiveName(dir.directiveIndex) + ".ngAfterContentChecked();");
            }
        }
        return res;
    };
    CodegenLogicUtil.prototype.genViewLifecycleCallbacks = function (directiveRecords) {
        var res = [];
        var eq = lang_1.IS_DART ? '==' : '===';
        // NOTE(kegluneq): Order is important!
        for (var i = directiveRecords.length - 1; i >= 0; --i) {
            var dir = directiveRecords[i];
            if (dir.callAfterViewInit) {
                res.push("if(" + this._names.getStateName() + " " + eq + " " + this._changeDetectorStateName + ".NeverChecked) " + this._names.getDirectiveName(dir.directiveIndex) + ".ngAfterViewInit();");
            }
            if (dir.callAfterViewChecked) {
                res.push(this._names.getDirectiveName(dir.directiveIndex) + ".ngAfterViewChecked();");
            }
        }
        return res;
    };
    return CodegenLogicUtil;
})();
exports.CodegenLogicUtil = CodegenLogicUtil;
//# sourceMappingURL=data:application/json;base64,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