'use strict';var lang_1 = require('angular2/src/facade/lang');
var exceptions_1 = require('angular2/src/facade/exceptions');
var collection_1 = require('angular2/src/facade/collection');
var utils_1 = require('../../utils');
var url_parser_1 = require('../../url_parser');
var route_path_1 = require('./route_path');
/**
 * Identified by a `...` URL segment. This indicates that the
 * Route will continue to be matched by child `Router`s.
 */
var ContinuationPathSegment = (function () {
    function ContinuationPathSegment() {
        this.name = '';
        this.specificity = '';
        this.hash = '...';
    }
    ContinuationPathSegment.prototype.generate = function (params) { return ''; };
    ContinuationPathSegment.prototype.match = function (path) { return true; };
    return ContinuationPathSegment;
})();
/**
 * Identified by a string not starting with a `:` or `*`.
 * Only matches the URL segments that equal the segment path
 */
var StaticPathSegment = (function () {
    function StaticPathSegment(path) {
        this.path = path;
        this.name = '';
        this.specificity = '2';
        this.hash = path;
    }
    StaticPathSegment.prototype.match = function (path) { return path == this.path; };
    StaticPathSegment.prototype.generate = function (params) { return this.path; };
    return StaticPathSegment;
})();
/**
 * Identified by a string starting with `:`. Indicates a segment
 * that can contain a value that will be extracted and provided to
 * a matching `Instruction`.
 */
var DynamicPathSegment = (function () {
    function DynamicPathSegment(name) {
        this.name = name;
        this.specificity = '1';
        this.hash = ':';
    }
    DynamicPathSegment.prototype.match = function (path) { return path.length > 0; };
    DynamicPathSegment.prototype.generate = function (params) {
        if (!collection_1.StringMapWrapper.contains(params.map, this.name)) {
            throw new exceptions_1.BaseException("Route generator for '" + this.name + "' was not included in parameters passed.");
        }
        return utils_1.normalizeString(params.get(this.name));
    };
    DynamicPathSegment.paramMatcher = /^:([^\/]+)$/g;
    return DynamicPathSegment;
})();
/**
 * Identified by a string starting with `*` Indicates that all the following
 * segments match this route and that the value of these segments should
 * be provided to a matching `Instruction`.
 */
var StarPathSegment = (function () {
    function StarPathSegment(name) {
        this.name = name;
        this.specificity = '0';
        this.hash = '*';
    }
    StarPathSegment.prototype.match = function (path) { return true; };
    StarPathSegment.prototype.generate = function (params) { return utils_1.normalizeString(params.get(this.name)); };
    StarPathSegment.wildcardMatcher = /^\*([^\/]+)$/g;
    return StarPathSegment;
})();
/**
 * Parses a URL string using a given matcher DSL, and generates URLs from param maps
 */
var ParamRoutePath = (function () {
    /**
     * Takes a string representing the matcher DSL
     */
    function ParamRoutePath(routePath) {
        this.routePath = routePath;
        this.terminal = true;
        this._assertValidPath(routePath);
        this._parsePathString(routePath);
        this.specificity = this._calculateSpecificity();
        this.hash = this._calculateHash();
        var lastSegment = this._segments[this._segments.length - 1];
        this.terminal = !(lastSegment instanceof ContinuationPathSegment);
    }
    ParamRoutePath.prototype.matchUrl = function (url) {
        var nextUrlSegment = url;
        var currentUrlSegment;
        var positionalParams = {};
        var captured = [];
        for (var i = 0; i < this._segments.length; i += 1) {
            var pathSegment = this._segments[i];
            currentUrlSegment = nextUrlSegment;
            if (pathSegment instanceof ContinuationPathSegment) {
                break;
            }
            if (lang_1.isPresent(currentUrlSegment)) {
                // the star segment consumes all of the remaining URL, including matrix params
                if (pathSegment instanceof StarPathSegment) {
                    positionalParams[pathSegment.name] = currentUrlSegment.toString();
                    captured.push(currentUrlSegment.toString());
                    nextUrlSegment = null;
                    break;
                }
                captured.push(currentUrlSegment.path);
                if (pathSegment instanceof DynamicPathSegment) {
                    positionalParams[pathSegment.name] = currentUrlSegment.path;
                }
                else if (!pathSegment.match(currentUrlSegment.path)) {
                    return null;
                }
                nextUrlSegment = currentUrlSegment.child;
            }
            else if (!pathSegment.match('')) {
                return null;
            }
        }
        if (this.terminal && lang_1.isPresent(nextUrlSegment)) {
            return null;
        }
        var urlPath = captured.join('/');
        var auxiliary = [];
        var urlParams = [];
        var allParams = positionalParams;
        if (lang_1.isPresent(currentUrlSegment)) {
            // If this is the root component, read query params. Otherwise, read matrix params.
            var paramsSegment = url instanceof url_parser_1.RootUrl ? url : currentUrlSegment;
            if (lang_1.isPresent(paramsSegment.params)) {
                allParams = collection_1.StringMapWrapper.merge(paramsSegment.params, positionalParams);
                urlParams = url_parser_1.convertUrlParamsToArray(paramsSegment.params);
            }
            else {
                allParams = positionalParams;
            }
            auxiliary = currentUrlSegment.auxiliary;
        }
        return new route_path_1.MatchedUrl(urlPath, urlParams, allParams, auxiliary, nextUrlSegment);
    };
    ParamRoutePath.prototype.generateUrl = function (params) {
        var paramTokens = new utils_1.TouchMap(params);
        var path = [];
        for (var i = 0; i < this._segments.length; i++) {
            var segment = this._segments[i];
            if (!(segment instanceof ContinuationPathSegment)) {
                path.push(segment.generate(paramTokens));
            }
        }
        var urlPath = path.join('/');
        var nonPositionalParams = paramTokens.getUnused();
        var urlParams = nonPositionalParams;
        return new route_path_1.GeneratedUrl(urlPath, urlParams);
    };
    ParamRoutePath.prototype.toString = function () { return this.routePath; };
    ParamRoutePath.prototype._parsePathString = function (routePath) {
        // normalize route as not starting with a "/". Recognition will
        // also normalize.
        if (routePath.startsWith("/")) {
            routePath = routePath.substring(1);
        }
        var segmentStrings = routePath.split('/');
        this._segments = [];
        var limit = segmentStrings.length - 1;
        for (var i = 0; i <= limit; i++) {
            var segment = segmentStrings[i], match;
            if (lang_1.isPresent(match = lang_1.RegExpWrapper.firstMatch(DynamicPathSegment.paramMatcher, segment))) {
                this._segments.push(new DynamicPathSegment(match[1]));
            }
            else if (lang_1.isPresent(match = lang_1.RegExpWrapper.firstMatch(StarPathSegment.wildcardMatcher, segment))) {
                this._segments.push(new StarPathSegment(match[1]));
            }
            else if (segment == '...') {
                if (i < limit) {
                    throw new exceptions_1.BaseException("Unexpected \"...\" before the end of the path for \"" + routePath + "\".");
                }
                this._segments.push(new ContinuationPathSegment());
            }
            else {
                this._segments.push(new StaticPathSegment(segment));
            }
        }
    };
    ParamRoutePath.prototype._calculateSpecificity = function () {
        // The "specificity" of a path is used to determine which route is used when multiple routes
        // match
        // a URL. Static segments (like "/foo") are the most specific, followed by dynamic segments
        // (like
        // "/:id"). Star segments add no specificity. Segments at the start of the path are more
        // specific
        // than proceeding ones.
        //
        // The code below uses place values to combine the different types of segments into a single
        // string that we can sort later. Each static segment is marked as a specificity of "2," each
        // dynamic segment is worth "1" specificity, and stars are worth "0" specificity.
        var i, length = this._segments.length, specificity;
        if (length == 0) {
            // a single slash (or "empty segment" is as specific as a static segment
            specificity += '2';
        }
        else {
            specificity = '';
            for (i = 0; i < length; i++) {
                specificity += this._segments[i].specificity;
            }
        }
        return specificity;
    };
    ParamRoutePath.prototype._calculateHash = function () {
        // this function is used to determine whether a route config path like `/foo/:id` collides with
        // `/foo/:name`
        var i, length = this._segments.length;
        var hashParts = [];
        for (i = 0; i < length; i++) {
            hashParts.push(this._segments[i].hash);
        }
        return hashParts.join('/');
    };
    ParamRoutePath.prototype._assertValidPath = function (path) {
        if (lang_1.StringWrapper.contains(path, '#')) {
            throw new exceptions_1.BaseException("Path \"" + path + "\" should not include \"#\". Use \"HashLocationStrategy\" instead.");
        }
        var illegalCharacter = lang_1.RegExpWrapper.firstMatch(ParamRoutePath.RESERVED_CHARS, path);
        if (lang_1.isPresent(illegalCharacter)) {
            throw new exceptions_1.BaseException("Path \"" + path + "\" contains \"" + illegalCharacter[0] + "\" which is not allowed in a route config.");
        }
    };
    ParamRoutePath.RESERVED_CHARS = lang_1.RegExpWrapper.create('//|\\(|\\)|;|\\?|=');
    return ParamRoutePath;
})();
exports.ParamRoutePath = ParamRoutePath;
//# sourceMappingURL=data:application/json;base64,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