import { isJsObject, global, isPresent, isBlank, isArray, getSymbolIterator } from 'angular2/src/facade/lang';
export var Map = global.Map;
export var Set = global.Set;
// Safari and Internet Explorer do not support the iterable parameter to the
// Map constructor.  We work around that by manually adding the items.
var createMapFromPairs = (function () {
    try {
        if (new Map([[1, 2]]).size === 1) {
            return function createMapFromPairs(pairs) { return new Map(pairs); };
        }
    }
    catch (e) {
    }
    return function createMapAndPopulateFromPairs(pairs) {
        var map = new Map();
        for (var i = 0; i < pairs.length; i++) {
            var pair = pairs[i];
            map.set(pair[0], pair[1]);
        }
        return map;
    };
})();
var createMapFromMap = (function () {
    try {
        if (new Map(new Map())) {
            return function createMapFromMap(m) { return new Map(m); };
        }
    }
    catch (e) {
    }
    return function createMapAndPopulateFromMap(m) {
        var map = new Map();
        m.forEach((v, k) => { map.set(k, v); });
        return map;
    };
})();
var _clearValues = (function () {
    if ((new Map()).keys().next) {
        return function _clearValues(m) {
            var keyIterator = m.keys();
            var k;
            while (!((k = keyIterator.next()).done)) {
                m.set(k.value, null);
            }
        };
    }
    else {
        return function _clearValuesWithForeEach(m) {
            m.forEach((v, k) => { m.set(k, null); });
        };
    }
})();
// Safari doesn't implement MapIterator.next(), which is used is Traceur's polyfill of Array.from
// TODO(mlaval): remove the work around once we have a working polyfill of Array.from
var _arrayFromMap = (function () {
    try {
        if ((new Map()).values().next) {
            return function createArrayFromMap(m, getValues) {
                return getValues ? Array.from(m.values()) : Array.from(m.keys());
            };
        }
    }
    catch (e) {
    }
    return function createArrayFromMapWithForeach(m, getValues) {
        var res = ListWrapper.createFixedSize(m.size), i = 0;
        m.forEach((v, k) => {
            res[i] = getValues ? v : k;
            i++;
        });
        return res;
    };
})();
export class MapWrapper {
    static clone(m) { return createMapFromMap(m); }
    static createFromStringMap(stringMap) {
        var result = new Map();
        for (var prop in stringMap) {
            result.set(prop, stringMap[prop]);
        }
        return result;
    }
    static toStringMap(m) {
        var r = {};
        m.forEach((v, k) => r[k] = v);
        return r;
    }
    static createFromPairs(pairs) { return createMapFromPairs(pairs); }
    static clearValues(m) { _clearValues(m); }
    static iterable(m) { return m; }
    static keys(m) { return _arrayFromMap(m, false); }
    static values(m) { return _arrayFromMap(m, true); }
}
/**
 * Wraps Javascript Objects
 */
export class StringMapWrapper {
    static create() {
        // Note: We are not using Object.create(null) here due to
        // performance!
        // http://jsperf.com/ng2-object-create-null
        return {};
    }
    static contains(map, key) {
        return map.hasOwnProperty(key);
    }
    static get(map, key) {
        return map.hasOwnProperty(key) ? map[key] : undefined;
    }
    static set(map, key, value) { map[key] = value; }
    static keys(map) { return Object.keys(map); }
    static values(map) {
        return Object.keys(map).reduce((r, a) => {
            r.push(map[a]);
            return r;
        }, []);
    }
    static isEmpty(map) {
        for (var prop in map) {
            return false;
        }
        return true;
    }
    static delete(map, key) { delete map[key]; }
    static forEach(map, callback) {
        for (var prop in map) {
            if (map.hasOwnProperty(prop)) {
                callback(map[prop], prop);
            }
        }
    }
    static merge(m1, m2) {
        var m = {};
        for (var attr in m1) {
            if (m1.hasOwnProperty(attr)) {
                m[attr] = m1[attr];
            }
        }
        for (var attr in m2) {
            if (m2.hasOwnProperty(attr)) {
                m[attr] = m2[attr];
            }
        }
        return m;
    }
    static equals(m1, m2) {
        var k1 = Object.keys(m1);
        var k2 = Object.keys(m2);
        if (k1.length != k2.length) {
            return false;
        }
        var key;
        for (var i = 0; i < k1.length; i++) {
            key = k1[i];
            if (m1[key] !== m2[key]) {
                return false;
            }
        }
        return true;
    }
}
export class ListWrapper {
    // JS has no way to express a statically fixed size list, but dart does so we
    // keep both methods.
    static createFixedSize(size) { return new Array(size); }
    static createGrowableSize(size) { return new Array(size); }
    static clone(array) { return array.slice(0); }
    static createImmutable(array) {
        var result = ListWrapper.clone(array);
        Object.seal(result);
        return result;
    }
    static forEachWithIndex(array, fn) {
        for (var i = 0; i < array.length; i++) {
            fn(array[i], i);
        }
    }
    static first(array) {
        if (!array)
            return null;
        return array[0];
    }
    static last(array) {
        if (!array || array.length == 0)
            return null;
        return array[array.length - 1];
    }
    static indexOf(array, value, startIndex = 0) {
        return array.indexOf(value, startIndex);
    }
    static contains(list, el) { return list.indexOf(el) !== -1; }
    static reversed(array) {
        var a = ListWrapper.clone(array);
        return a.reverse();
    }
    static concat(a, b) { return a.concat(b); }
    static insert(list, index, value) { list.splice(index, 0, value); }
    static removeAt(list, index) {
        var res = list[index];
        list.splice(index, 1);
        return res;
    }
    static removeAll(list, items) {
        for (var i = 0; i < items.length; ++i) {
            var index = list.indexOf(items[i]);
            list.splice(index, 1);
        }
    }
    static remove(list, el) {
        var index = list.indexOf(el);
        if (index > -1) {
            list.splice(index, 1);
            return true;
        }
        return false;
    }
    static clear(list) { list.length = 0; }
    static isEmpty(list) { return list.length == 0; }
    static fill(list, value, start = 0, end = null) {
        list.fill(value, start, end === null ? list.length : end);
    }
    static equals(a, b) {
        if (a.length != b.length)
            return false;
        for (var i = 0; i < a.length; ++i) {
            if (a[i] !== b[i])
                return false;
        }
        return true;
    }
    static slice(l, from = 0, to = null) {
        return l.slice(from, to === null ? undefined : to);
    }
    static splice(l, from, length) { return l.splice(from, length); }
    static sort(l, compareFn) {
        if (isPresent(compareFn)) {
            l.sort(compareFn);
        }
        else {
            l.sort();
        }
    }
    static toString(l) { return l.toString(); }
    static toJSON(l) { return JSON.stringify(l); }
    static maximum(list, predicate) {
        if (list.length == 0) {
            return null;
        }
        var solution = null;
        var maxValue = -Infinity;
        for (var index = 0; index < list.length; index++) {
            var candidate = list[index];
            if (isBlank(candidate)) {
                continue;
            }
            var candidateValue = predicate(candidate);
            if (candidateValue > maxValue) {
                solution = candidate;
                maxValue = candidateValue;
            }
        }
        return solution;
    }
    static isImmutable(list) { return Object.isSealed(list); }
    static flatten(array) {
        let res = [];
        array.forEach((a) => res = res.concat(a));
        return res;
    }
}
export function isListLikeIterable(obj) {
    if (!isJsObject(obj))
        return false;
    return isArray(obj) ||
        (!(obj instanceof Map) &&
            getSymbolIterator() in obj); // JS Iterable have a Symbol.iterator prop
}
export function areIterablesEqual(a, b, comparator) {
    var iterator1 = a[getSymbolIterator()]();
    var iterator2 = b[getSymbolIterator()]();
    while (true) {
        let item1 = iterator1.next();
        let item2 = iterator2.next();
        if (item1.done && item2.done)
            return true;
        if (item1.done || item2.done)
            return false;
        if (!comparator(item1.value, item2.value))
            return false;
    }
}
export function iterateListLike(obj, fn) {
    if (isArray(obj)) {
        for (var i = 0; i < obj.length; i++) {
            fn(obj[i]);
        }
    }
    else {
        var iterator = obj[getSymbolIterator()]();
        var item;
        while (!((item = iterator.next()).done)) {
            fn(item.value);
        }
    }
}
// Safari and Internet Explorer do not support the iterable parameter to the
// Set constructor.  We work around that by manually adding the items.
var createSetFromList = (function () {
    var test = new Set([1, 2, 3]);
    if (test.size === 3) {
        return function createSetFromList(lst) { return new Set(lst); };
    }
    else {
        return function createSetAndPopulateFromList(lst) {
            var res = new Set(lst);
            if (res.size !== lst.length) {
                for (var i = 0; i < lst.length; i++) {
                    res.add(lst[i]);
                }
            }
            return res;
        };
    }
})();
export class SetWrapper {
    static createFromList(lst) { return createSetFromList(lst); }
    static has(s, key) { return s.has(key); }
    static delete(m, k) { m.delete(k); }
}
//# sourceMappingURL=data:application/json;base64,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