import { RegExpWrapper, StringWrapper, isPresent, isBlank } from 'angular2/src/facade/lang';
import { BaseException } from 'angular2/src/facade/exceptions';
import { StringMapWrapper } from 'angular2/src/facade/collection';
import { TouchMap, normalizeString } from '../../utils';
import { RootUrl, convertUrlParamsToArray } from '../../url_parser';
import { GeneratedUrl, MatchedUrl } from './route_path';
/**
 * Identified by a `...` URL segment. This indicates that the
 * Route will continue to be matched by child `Router`s.
 */
class ContinuationPathSegment {
    constructor() {
        this.name = '';
        this.specificity = '';
        this.hash = '...';
    }
    generate(params) { return ''; }
    match(path) { return true; }
}
/**
 * Identified by a string not starting with a `:` or `*`.
 * Only matches the URL segments that equal the segment path
 */
class StaticPathSegment {
    constructor(path) {
        this.path = path;
        this.name = '';
        this.specificity = '2';
        this.hash = path;
    }
    match(path) { return path == this.path; }
    generate(params) { return this.path; }
}
/**
 * Identified by a string starting with `:`. Indicates a segment
 * that can contain a value that will be extracted and provided to
 * a matching `Instruction`.
 */
class DynamicPathSegment {
    constructor(name) {
        this.name = name;
        this.specificity = '1';
        this.hash = ':';
    }
    match(path) { return path.length > 0; }
    generate(params) {
        if (!StringMapWrapper.contains(params.map, this.name)) {
            throw new BaseException(`Route generator for '${this.name}' was not included in parameters passed.`);
        }
        return encodeDynamicSegment(normalizeString(params.get(this.name)));
    }
}
DynamicPathSegment.paramMatcher = /^:([^\/]+)$/g;
/**
 * Identified by a string starting with `*` Indicates that all the following
 * segments match this route and that the value of these segments should
 * be provided to a matching `Instruction`.
 */
class StarPathSegment {
    constructor(name) {
        this.name = name;
        this.specificity = '0';
        this.hash = '*';
    }
    match(path) { return true; }
    generate(params) { return normalizeString(params.get(this.name)); }
}
StarPathSegment.wildcardMatcher = /^\*([^\/]+)$/g;
/**
 * Parses a URL string using a given matcher DSL, and generates URLs from param maps
 */
export class ParamRoutePath {
    /**
     * Takes a string representing the matcher DSL
     */
    constructor(routePath) {
        this.routePath = routePath;
        this.terminal = true;
        this._assertValidPath(routePath);
        this._parsePathString(routePath);
        this.specificity = this._calculateSpecificity();
        this.hash = this._calculateHash();
        var lastSegment = this._segments[this._segments.length - 1];
        this.terminal = !(lastSegment instanceof ContinuationPathSegment);
    }
    matchUrl(url) {
        var nextUrlSegment = url;
        var currentUrlSegment;
        var positionalParams = {};
        var captured = [];
        for (var i = 0; i < this._segments.length; i += 1) {
            var pathSegment = this._segments[i];
            currentUrlSegment = nextUrlSegment;
            if (pathSegment instanceof ContinuationPathSegment) {
                break;
            }
            if (isPresent(currentUrlSegment)) {
                // the star segment consumes all of the remaining URL, including matrix params
                if (pathSegment instanceof StarPathSegment) {
                    positionalParams[pathSegment.name] = currentUrlSegment.toString();
                    captured.push(currentUrlSegment.toString());
                    nextUrlSegment = null;
                    break;
                }
                captured.push(currentUrlSegment.path);
                if (pathSegment instanceof DynamicPathSegment) {
                    positionalParams[pathSegment.name] = decodeDynamicSegment(currentUrlSegment.path);
                }
                else if (!pathSegment.match(currentUrlSegment.path)) {
                    return null;
                }
                nextUrlSegment = currentUrlSegment.child;
            }
            else if (!pathSegment.match('')) {
                return null;
            }
        }
        if (this.terminal && isPresent(nextUrlSegment)) {
            return null;
        }
        var urlPath = captured.join('/');
        var auxiliary = [];
        var urlParams = [];
        var allParams = positionalParams;
        if (isPresent(currentUrlSegment)) {
            // If this is the root component, read query params. Otherwise, read matrix params.
            var paramsSegment = url instanceof RootUrl ? url : currentUrlSegment;
            if (isPresent(paramsSegment.params)) {
                allParams = StringMapWrapper.merge(paramsSegment.params, positionalParams);
                urlParams = convertUrlParamsToArray(paramsSegment.params);
            }
            else {
                allParams = positionalParams;
            }
            auxiliary = currentUrlSegment.auxiliary;
        }
        return new MatchedUrl(urlPath, urlParams, allParams, auxiliary, nextUrlSegment);
    }
    generateUrl(params) {
        var paramTokens = new TouchMap(params);
        var path = [];
        for (var i = 0; i < this._segments.length; i++) {
            let segment = this._segments[i];
            if (!(segment instanceof ContinuationPathSegment)) {
                path.push(segment.generate(paramTokens));
            }
        }
        var urlPath = path.join('/');
        var nonPositionalParams = paramTokens.getUnused();
        var urlParams = nonPositionalParams;
        return new GeneratedUrl(urlPath, urlParams);
    }
    toString() { return this.routePath; }
    _parsePathString(routePath) {
        // normalize route as not starting with a "/". Recognition will
        // also normalize.
        if (routePath.startsWith("/")) {
            routePath = routePath.substring(1);
        }
        var segmentStrings = routePath.split('/');
        this._segments = [];
        var limit = segmentStrings.length - 1;
        for (var i = 0; i <= limit; i++) {
            var segment = segmentStrings[i], match;
            if (isPresent(match = RegExpWrapper.firstMatch(DynamicPathSegment.paramMatcher, segment))) {
                this._segments.push(new DynamicPathSegment(match[1]));
            }
            else if (isPresent(match = RegExpWrapper.firstMatch(StarPathSegment.wildcardMatcher, segment))) {
                this._segments.push(new StarPathSegment(match[1]));
            }
            else if (segment == '...') {
                if (i < limit) {
                    throw new BaseException(`Unexpected "..." before the end of the path for "${routePath}".`);
                }
                this._segments.push(new ContinuationPathSegment());
            }
            else {
                this._segments.push(new StaticPathSegment(segment));
            }
        }
    }
    _calculateSpecificity() {
        // The "specificity" of a path is used to determine which route is used when multiple routes
        // match
        // a URL. Static segments (like "/foo") are the most specific, followed by dynamic segments
        // (like
        // "/:id"). Star segments add no specificity. Segments at the start of the path are more
        // specific
        // than proceeding ones.
        //
        // The code below uses place values to combine the different types of segments into a single
        // string that we can sort later. Each static segment is marked as a specificity of "2," each
        // dynamic segment is worth "1" specificity, and stars are worth "0" specificity.
        var i, length = this._segments.length, specificity;
        if (length == 0) {
            // a single slash (or "empty segment" is as specific as a static segment
            specificity += '2';
        }
        else {
            specificity = '';
            for (i = 0; i < length; i++) {
                specificity += this._segments[i].specificity;
            }
        }
        return specificity;
    }
    _calculateHash() {
        // this function is used to determine whether a route config path like `/foo/:id` collides with
        // `/foo/:name`
        var i, length = this._segments.length;
        var hashParts = [];
        for (i = 0; i < length; i++) {
            hashParts.push(this._segments[i].hash);
        }
        return hashParts.join('/');
    }
    _assertValidPath(path) {
        if (StringWrapper.contains(path, '#')) {
            throw new BaseException(`Path "${path}" should not include "#". Use "HashLocationStrategy" instead.`);
        }
        var illegalCharacter = RegExpWrapper.firstMatch(ParamRoutePath.RESERVED_CHARS, path);
        if (isPresent(illegalCharacter)) {
            throw new BaseException(`Path "${path}" contains "${illegalCharacter[0]}" which is not allowed in a route config.`);
        }
    }
}
ParamRoutePath.RESERVED_CHARS = RegExpWrapper.create('//|\\(|\\)|;|\\?|=');
let REGEXP_PERCENT = /%/g;
let REGEXP_SLASH = /\//g;
let REGEXP_OPEN_PARENT = /\(/g;
let REGEXP_CLOSE_PARENT = /\)/g;
let REGEXP_SEMICOLON = /;/g;
function encodeDynamicSegment(value) {
    if (isBlank(value)) {
        return null;
    }
    value = StringWrapper.replaceAll(value, REGEXP_PERCENT, '%25');
    value = StringWrapper.replaceAll(value, REGEXP_SLASH, '%2F');
    value = StringWrapper.replaceAll(value, REGEXP_OPEN_PARENT, '%28');
    value = StringWrapper.replaceAll(value, REGEXP_CLOSE_PARENT, '%29');
    value = StringWrapper.replaceAll(value, REGEXP_SEMICOLON, '%3B');
    return value;
}
let REGEXP_ENC_SEMICOLON = /%3B/ig;
let REGEXP_ENC_CLOSE_PARENT = /%29/ig;
let REGEXP_ENC_OPEN_PARENT = /%28/ig;
let REGEXP_ENC_SLASH = /%2F/ig;
let REGEXP_ENC_PERCENT = /%25/ig;
function decodeDynamicSegment(value) {
    if (isBlank(value)) {
        return null;
    }
    value = StringWrapper.replaceAll(value, REGEXP_ENC_SEMICOLON, ';');
    value = StringWrapper.replaceAll(value, REGEXP_ENC_CLOSE_PARENT, ')');
    value = StringWrapper.replaceAll(value, REGEXP_ENC_OPEN_PARENT, '(');
    value = StringWrapper.replaceAll(value, REGEXP_ENC_SLASH, '/');
    value = StringWrapper.replaceAll(value, REGEXP_ENC_PERCENT, '%');
    return value;
}
//# sourceMappingURL=data:application/json;base64,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