import { assertionsEnabled, isPresent, isBlank } from 'angular2/src/facade/lang';
import { ListWrapper } from 'angular2/src/facade/collection';
import { ChangeDetectionUtil } from './change_detection_util';
import { ChangeDetectorRef_ } from './change_detector_ref';
import { ChangeDetectionError, ExpressionChangedAfterItHasBeenCheckedException, DehydratedException, EventEvaluationErrorContext, EventEvaluationError } from './exceptions';
import { Locals } from './parser/locals';
import { ChangeDetectionStrategy, ChangeDetectorState } from './constants';
import { wtfCreateScope, wtfLeave } from '../profile/profile';
import { ObservableWrapper } from 'angular2/src/facade/async';
var _scope_check = wtfCreateScope(`ChangeDetector#check(ascii id, bool throwOnChange)`);
class _Context {
    constructor(element, componentElement, context, locals, injector, expression) {
        this.element = element;
        this.componentElement = componentElement;
        this.context = context;
        this.locals = locals;
        this.injector = injector;
        this.expression = expression;
    }
}
export class AbstractChangeDetector {
    constructor(id, numberOfPropertyProtoRecords, bindingTargets, directiveIndices, strategy) {
        this.id = id;
        this.numberOfPropertyProtoRecords = numberOfPropertyProtoRecords;
        this.bindingTargets = bindingTargets;
        this.directiveIndices = directiveIndices;
        this.strategy = strategy;
        this.contentChildren = [];
        this.viewChildren = [];
        // The names of the below fields must be kept in sync with codegen_name_util.ts or
        // change detection will fail.
        this.state = ChangeDetectorState.NeverChecked;
        this.locals = null;
        this.mode = null;
        this.pipes = null;
        this.ref = new ChangeDetectorRef_(this);
    }
    addContentChild(cd) {
        this.contentChildren.push(cd);
        cd.parent = this;
    }
    removeContentChild(cd) { ListWrapper.remove(this.contentChildren, cd); }
    addViewChild(cd) {
        this.viewChildren.push(cd);
        cd.parent = this;
    }
    removeViewChild(cd) { ListWrapper.remove(this.viewChildren, cd); }
    remove() { this.parent.removeContentChild(this); }
    handleEvent(eventName, elIndex, event) {
        if (!this.hydrated()) {
            this.throwDehydratedError(`${this.id} -> ${eventName}`);
        }
        try {
            var locals = new Map();
            locals.set('$event', event);
            var res = !this.handleEventInternal(eventName, elIndex, new Locals(this.locals, locals));
            this.markPathToRootAsCheckOnce();
            return res;
        }
        catch (e) {
            var c = this.dispatcher.getDebugContext(null, elIndex, null);
            var context = isPresent(c) ?
                new EventEvaluationErrorContext(c.element, c.componentElement, c.context, c.locals, c.injector) :
                null;
            throw new EventEvaluationError(eventName, e, e.stack, context);
        }
    }
    handleEventInternal(eventName, elIndex, locals) { return false; }
    detectChanges() { this.runDetectChanges(false); }
    checkNoChanges() {
        if (assertionsEnabled()) {
            this.runDetectChanges(true);
        }
    }
    runDetectChanges(throwOnChange) {
        if (this.mode === ChangeDetectionStrategy.Detached ||
            this.mode === ChangeDetectionStrategy.Checked || this.state === ChangeDetectorState.Errored)
            return;
        var s = _scope_check(this.id, throwOnChange);
        this.detectChangesInRecords(throwOnChange);
        this._detectChangesContentChildren(throwOnChange);
        if (!throwOnChange)
            this.afterContentLifecycleCallbacks();
        this._detectChangesInViewChildren(throwOnChange);
        if (!throwOnChange)
            this.afterViewLifecycleCallbacks();
        if (this.mode === ChangeDetectionStrategy.CheckOnce)
            this.mode = ChangeDetectionStrategy.Checked;
        this.state = ChangeDetectorState.CheckedBefore;
        wtfLeave(s);
    }
    // This method is not intended to be overridden. Subclasses should instead provide an
    // implementation of `detectChangesInRecordsInternal` which does the work of detecting changes
    // and which this method will call.
    // This method expects that `detectChangesInRecordsInternal` will set the property
    // `this.propertyBindingIndex` to the propertyBindingIndex of the first proto record. This is to
    // facilitate error reporting.
    detectChangesInRecords(throwOnChange) {
        if (!this.hydrated()) {
            this.throwDehydratedError(this.id);
        }
        try {
            this.detectChangesInRecordsInternal(throwOnChange);
        }
        catch (e) {
            // throwOnChange errors aren't counted as fatal errors.
            if (!(e instanceof ExpressionChangedAfterItHasBeenCheckedException)) {
                this.state = ChangeDetectorState.Errored;
            }
            this._throwError(e, e.stack);
        }
    }
    // Subclasses should override this method to perform any work necessary to detect and report
    // changes. For example, changes should be reported via `ChangeDetectionUtil.addChange`, lifecycle
    // methods should be called, etc.
    // This implementation should also set `this.propertyBindingIndex` to the propertyBindingIndex of
    // the
    // first proto record to facilitate error reporting. See {@link #detectChangesInRecords}.
    detectChangesInRecordsInternal(throwOnChange) { }
    // This method is not intended to be overridden. Subclasses should instead provide an
    // implementation of `hydrateDirectives`.
    hydrate(context, locals, dispatcher, pipes) {
        this.dispatcher = dispatcher;
        this.mode = ChangeDetectionUtil.changeDetectionMode(this.strategy);
        this.context = context;
        this.locals = locals;
        this.pipes = pipes;
        this.hydrateDirectives(dispatcher);
        this.state = ChangeDetectorState.NeverChecked;
    }
    // Subclasses should override this method to hydrate any directives.
    hydrateDirectives(dispatcher) { }
    // This method is not intended to be overridden. Subclasses should instead provide an
    // implementation of `dehydrateDirectives`.
    dehydrate() {
        this.dehydrateDirectives(true);
        this._unsubscribeFromOutputs();
        this.dispatcher = null;
        this.context = null;
        this.locals = null;
        this.pipes = null;
    }
    // Subclasses should override this method to dehydrate any directives. This method should reverse
    // any work done in `hydrateDirectives`.
    dehydrateDirectives(destroyPipes) { }
    hydrated() { return isPresent(this.context); }
    destroyRecursive() {
        this.dispatcher.notifyOnDestroy();
        this.dehydrate();
        var children = this.contentChildren;
        for (var i = 0; i < children.length; i++) {
            children[i].destroyRecursive();
        }
        children = this.viewChildren;
        for (var i = 0; i < children.length; i++) {
            children[i].destroyRecursive();
        }
    }
    afterContentLifecycleCallbacks() {
        this.dispatcher.notifyAfterContentChecked();
        this.afterContentLifecycleCallbacksInternal();
    }
    afterContentLifecycleCallbacksInternal() { }
    afterViewLifecycleCallbacks() {
        this.dispatcher.notifyAfterViewChecked();
        this.afterViewLifecycleCallbacksInternal();
    }
    afterViewLifecycleCallbacksInternal() { }
    /** @internal */
    _detectChangesContentChildren(throwOnChange) {
        var c = this.contentChildren;
        for (var i = 0; i < c.length; ++i) {
            c[i].runDetectChanges(throwOnChange);
        }
    }
    /** @internal */
    _detectChangesInViewChildren(throwOnChange) {
        var c = this.viewChildren;
        for (var i = 0; i < c.length; ++i) {
            c[i].runDetectChanges(throwOnChange);
        }
    }
    markAsCheckOnce() { this.mode = ChangeDetectionStrategy.CheckOnce; }
    markPathToRootAsCheckOnce() {
        var c = this;
        while (isPresent(c) && c.mode !== ChangeDetectionStrategy.Detached) {
            if (c.mode === ChangeDetectionStrategy.Checked)
                c.mode = ChangeDetectionStrategy.CheckOnce;
            c = c.parent;
        }
    }
    _unsubscribeFromOutputs() {
        if (isPresent(this.outputSubscriptions)) {
            for (var i = 0; i < this.outputSubscriptions.length; ++i) {
                ObservableWrapper.dispose(this.outputSubscriptions[i]);
                this.outputSubscriptions[i] = null;
            }
        }
    }
    getDirectiveFor(directives, index) {
        return directives.getDirectiveFor(this.directiveIndices[index]);
    }
    getDetectorFor(directives, index) {
        return directives.getDetectorFor(this.directiveIndices[index]);
    }
    notifyDispatcher(value) {
        this.dispatcher.notifyOnBinding(this._currentBinding(), value);
    }
    logBindingUpdate(value) {
        this.dispatcher.logBindingUpdate(this._currentBinding(), value);
    }
    addChange(changes, oldValue, newValue) {
        if (isBlank(changes)) {
            changes = {};
        }
        changes[this._currentBinding().name] = ChangeDetectionUtil.simpleChange(oldValue, newValue);
        return changes;
    }
    _throwError(exception, stack) {
        var error;
        try {
            var c = this.dispatcher.getDebugContext(null, this._currentBinding().elementIndex, null);
            var context = isPresent(c) ? new _Context(c.element, c.componentElement, c.context, c.locals, c.injector, this._currentBinding().debug) :
                null;
            error = new ChangeDetectionError(this._currentBinding().debug, exception, stack, context);
        }
        catch (e) {
            // if an error happens during getting the debug context, we throw a ChangeDetectionError
            // without the extra information.
            error = new ChangeDetectionError(null, exception, stack, null);
        }
        throw error;
    }
    throwOnChangeError(oldValue, newValue) {
        throw new ExpressionChangedAfterItHasBeenCheckedException(this._currentBinding().debug, oldValue, newValue, null);
    }
    throwDehydratedError(detail) { throw new DehydratedException(detail); }
    _currentBinding() {
        return this.bindingTargets[this.propertyBindingIndex];
    }
}
//# sourceMappingURL=data:application/json;base64,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