import { isPresent } from 'angular2/src/facade/lang';
import { unimplemented } from 'angular2/src/facade/exceptions';
/**
 * A DebugElement contains information from the Angular compiler about an
 * element and provides access to the corresponding ElementInjector and
 * underlying DOM Element, as well as a way to query for children.
 *
 * A DebugElement can be obtained from a {@link ComponentFixture} or from an
 * {@link ElementRef} via {@link inspectElement}.
 */
export class DebugElement {
    /**
     * Return the instance of the component associated with this element, if any.
     */
    get componentInstance() { return unimplemented(); }
    ;
    /**
     * Return the native HTML element for this DebugElement.
     */
    get nativeElement() { return unimplemented(); }
    ;
    /**
     * Return an Angular {@link ElementRef} for this element.
     */
    get elementRef() { return unimplemented(); }
    ;
    /**
     * Get child DebugElements from within the Light DOM.
     *
     * @return {DebugElement[]}
     */
    get children() { return unimplemented(); }
    ;
    /**
     * Get the root DebugElement children of a component. Returns an empty
     * list if the current DebugElement is not a component root.
     *
     * @return {DebugElement[]}
     */
    get componentViewChildren() { return unimplemented(); }
    ;
    /**
     * Return the first descendant TestElement matching the given predicate
     * and scope.
     *
     * @param {Function: boolean} predicate
     * @param {Scope} scope
     *
     * @return {DebugElement}
     */
    query(predicate, scope = Scope.all) {
        var results = this.queryAll(predicate, scope);
        return results.length > 0 ? results[0] : null;
    }
    /**
     * Return descendant TestElememts matching the given predicate
     * and scope.
     *
     * @param {Function: boolean} predicate
     * @param {Scope} scope
     *
     * @return {DebugElement[]}
     */
    queryAll(predicate, scope = Scope.all) {
        var elementsInScope = scope(this);
        return elementsInScope.filter(predicate);
    }
}
export class DebugElement_ extends DebugElement {
    constructor(_appElement) {
        super();
        this._appElement = _appElement;
    }
    get componentInstance() {
        if (!isPresent(this._appElement)) {
            return null;
        }
        return this._appElement.getComponent();
    }
    get nativeElement() { return this.elementRef.nativeElement; }
    get elementRef() { return this._appElement.ref; }
    getDirectiveInstance(directiveIndex) {
        return this._appElement.getDirectiveAtIndex(directiveIndex);
    }
    get children() {
        return this._getChildElements(this._appElement.parentView, this._appElement);
    }
    get componentViewChildren() {
        if (!isPresent(this._appElement.componentView)) {
            // The current element is not a component.
            return [];
        }
        return this._getChildElements(this._appElement.componentView, null);
    }
    triggerEventHandler(eventName, eventObj) {
        this._appElement.parentView.triggerEventHandlers(eventName, eventObj, this._appElement.proto.index);
    }
    hasDirective(type) {
        if (!isPresent(this._appElement)) {
            return false;
        }
        return this._appElement.hasDirective(type);
    }
    inject(type) {
        if (!isPresent(this._appElement)) {
            return null;
        }
        return this._appElement.get(type);
    }
    getLocal(name) { return this._appElement.parentView.locals.get(name); }
    /** @internal */
    _getChildElements(view, parentAppElement) {
        var els = [];
        for (var i = 0; i < view.appElements.length; ++i) {
            var appEl = view.appElements[i];
            if (appEl.parent == parentAppElement) {
                els.push(new DebugElement_(appEl));
                var views = appEl.nestedViews;
                if (isPresent(views)) {
                    views.forEach((nextView) => { els = els.concat(this._getChildElements(nextView, null)); });
                }
            }
        }
        return els;
    }
}
/**
 * Returns a {@link DebugElement} for an {@link ElementRef}.
 *
 * @param {ElementRef}: elementRef
 * @return {DebugElement}
 */
export function inspectElement(elementRef) {
    return new DebugElement_(elementRef.internalElement);
}
/**
 * Maps an array of {@link DebugElement}s to an array of native DOM elements.
 */
export function asNativeElements(arr) {
    return arr.map((debugEl) => debugEl.nativeElement);
}
/**
 * Set of scope functions used with {@link DebugElement}'s query functionality.
 */
export class Scope {
    /**
     * Scope queries to both the light dom and view of an element and its
     * children.
     *
     * ## Example
     *
     * {@example core/debug/ts/debug_element/debug_element.ts region='scope_all'}
     */
    static all(debugElement) {
        var scope = [];
        scope.push(debugElement);
        debugElement.children.forEach(child => scope = scope.concat(Scope.all(child)));
        debugElement.componentViewChildren.forEach(child => scope = scope.concat(Scope.all(child)));
        return scope;
    }
    /**
     * Scope queries to the light dom of an element and its children.
     *
     * ## Example
     *
     * {@example core/debug/ts/debug_element/debug_element.ts region='scope_light'}
     */
    static light(debugElement) {
        var scope = [];
        debugElement.children.forEach(child => {
            scope.push(child);
            scope = scope.concat(Scope.light(child));
        });
        return scope;
    }
    /**
     * Scope queries to the view of an element of its children.
     *
     * ## Example
     *
     * {@example core/debug/ts/debug_element/debug_element.ts region='scope_view'}
     */
    static view(debugElement) {
        var scope = [];
        debugElement.componentViewChildren.forEach(child => {
            scope.push(child);
            scope = scope.concat(Scope.light(child));
        });
        return scope;
    }
}
//# sourceMappingURL=data:application/json;base64,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