import { isJsObject, global, isPresent, isBlank, isArray, getSymbolIterator } from 'angular2/src/facade/lang';
export var Map = global.Map;
export var Set = global.Set;
// Safari and Internet Explorer do not support the iterable parameter to the
// Map constructor.  We work around that by manually adding the items.
var createMapFromPairs = (function () {
    try {
        if (new Map([[1, 2]]).size === 1) {
            return function createMapFromPairs(pairs) { return new Map(pairs); };
        }
    }
    catch (e) {
    }
    return function createMapAndPopulateFromPairs(pairs) {
        var map = new Map();
        for (var i = 0; i < pairs.length; i++) {
            var pair = pairs[i];
            map.set(pair[0], pair[1]);
        }
        return map;
    };
})();
var createMapFromMap = (function () {
    try {
        if (new Map(new Map())) {
            return function createMapFromMap(m) { return new Map(m); };
        }
    }
    catch (e) {
    }
    return function createMapAndPopulateFromMap(m) {
        var map = new Map();
        m.forEach((v, k) => { map.set(k, v); });
        return map;
    };
})();
var _clearValues = (function () {
    if ((new Map()).keys().next) {
        return function _clearValues(m) {
            var keyIterator = m.keys();
            var k;
            while (!((k = keyIterator.next()).done)) {
                m.set(k.value, null);
            }
        };
    }
    else {
        return function _clearValuesWithForeEach(m) {
            m.forEach((v, k) => { m.set(k, null); });
        };
    }
})();
// Safari doesn't implement MapIterator.next(), which is used is Traceur's polyfill of Array.from
// TODO(mlaval): remove the work around once we have a working polyfill of Array.from
var _arrayFromMap = (function () {
    try {
        if ((new Map()).values().next) {
            return function createArrayFromMap(m, getValues) {
                return getValues ? Array.from(m.values()) : Array.from(m.keys());
            };
        }
    }
    catch (e) {
    }
    return function createArrayFromMapWithForeach(m, getValues) {
        var res = ListWrapper.createFixedSize(m.size), i = 0;
        m.forEach((v, k) => {
            res[i] = getValues ? v : k;
            i++;
        });
        return res;
    };
})();
export class MapWrapper {
    static clone(m) { return createMapFromMap(m); }
    static createFromStringMap(stringMap) {
        var result = new Map();
        for (var prop in stringMap) {
            result.set(prop, stringMap[prop]);
        }
        return result;
    }
    static toStringMap(m) {
        var r = {};
        m.forEach((v, k) => r[k] = v);
        return r;
    }
    static createFromPairs(pairs) { return createMapFromPairs(pairs); }
    static clearValues(m) { _clearValues(m); }
    static iterable(m) { return m; }
    static keys(m) { return _arrayFromMap(m, false); }
    static values(m) { return _arrayFromMap(m, true); }
}
/**
 * Wraps Javascript Objects
 */
export class StringMapWrapper {
    static create() {
        // Note: We are not using Object.create(null) here due to
        // performance!
        // http://jsperf.com/ng2-object-create-null
        return {};
    }
    static contains(map, key) {
        return map.hasOwnProperty(key);
    }
    static get(map, key) {
        return map.hasOwnProperty(key) ? map[key] : undefined;
    }
    static set(map, key, value) { map[key] = value; }
    static keys(map) { return Object.keys(map); }
    static values(map) {
        return Object.keys(map).reduce((r, a) => {
            r.push(map[a]);
            return r;
        }, []);
    }
    static isEmpty(map) {
        for (var prop in map) {
            return false;
        }
        return true;
    }
    static delete(map, key) { delete map[key]; }
    static forEach(map, callback) {
        for (var prop in map) {
            if (map.hasOwnProperty(prop)) {
                callback(map[prop], prop);
            }
        }
    }
    static merge(m1, m2) {
        var m = {};
        for (var attr in m1) {
            if (m1.hasOwnProperty(attr)) {
                m[attr] = m1[attr];
            }
        }
        for (var attr in m2) {
            if (m2.hasOwnProperty(attr)) {
                m[attr] = m2[attr];
            }
        }
        return m;
    }
    static equals(m1, m2) {
        var k1 = Object.keys(m1);
        var k2 = Object.keys(m2);
        if (k1.length != k2.length) {
            return false;
        }
        var key;
        for (var i = 0; i < k1.length; i++) {
            key = k1[i];
            if (m1[key] !== m2[key]) {
                return false;
            }
        }
        return true;
    }
}
export class ListWrapper {
    // JS has no way to express a statically fixed size list, but dart does so we
    // keep both methods.
    static createFixedSize(size) { return new Array(size); }
    static createGrowableSize(size) { return new Array(size); }
    static clone(array) { return array.slice(0); }
    static forEachWithIndex(array, fn) {
        for (var i = 0; i < array.length; i++) {
            fn(array[i], i);
        }
    }
    static first(array) {
        if (!array)
            return null;
        return array[0];
    }
    static last(array) {
        if (!array || array.length == 0)
            return null;
        return array[array.length - 1];
    }
    static indexOf(array, value, startIndex = 0) {
        return array.indexOf(value, startIndex);
    }
    static contains(list, el) { return list.indexOf(el) !== -1; }
    static reversed(array) {
        var a = ListWrapper.clone(array);
        return a.reverse();
    }
    static concat(a, b) { return a.concat(b); }
    static insert(list, index, value) { list.splice(index, 0, value); }
    static removeAt(list, index) {
        var res = list[index];
        list.splice(index, 1);
        return res;
    }
    static removeAll(list, items) {
        for (var i = 0; i < items.length; ++i) {
            var index = list.indexOf(items[i]);
            list.splice(index, 1);
        }
    }
    static remove(list, el) {
        var index = list.indexOf(el);
        if (index > -1) {
            list.splice(index, 1);
            return true;
        }
        return false;
    }
    static clear(list) { list.length = 0; }
    static isEmpty(list) { return list.length == 0; }
    static fill(list, value, start = 0, end = null) {
        list.fill(value, start, end === null ? list.length : end);
    }
    static equals(a, b) {
        if (a.length != b.length)
            return false;
        for (var i = 0; i < a.length; ++i) {
            if (a[i] !== b[i])
                return false;
        }
        return true;
    }
    static slice(l, from = 0, to = null) {
        return l.slice(from, to === null ? undefined : to);
    }
    static splice(l, from, length) { return l.splice(from, length); }
    static sort(l, compareFn) {
        if (isPresent(compareFn)) {
            l.sort(compareFn);
        }
        else {
            l.sort();
        }
    }
    static toString(l) { return l.toString(); }
    static toJSON(l) { return JSON.stringify(l); }
    static maximum(list, predicate) {
        if (list.length == 0) {
            return null;
        }
        var solution = null;
        var maxValue = -Infinity;
        for (var index = 0; index < list.length; index++) {
            var candidate = list[index];
            if (isBlank(candidate)) {
                continue;
            }
            var candidateValue = predicate(candidate);
            if (candidateValue > maxValue) {
                solution = candidate;
                maxValue = candidateValue;
            }
        }
        return solution;
    }
    static flatten(list) {
        var target = [];
        _flattenArray(list, target);
        return target;
    }
    static addAll(list, source) {
        for (var i = 0; i < source.length; i++) {
            list.push(source[i]);
        }
    }
}
function _flattenArray(source, target) {
    if (isPresent(source)) {
        for (var i = 0; i < source.length; i++) {
            var item = source[i];
            if (isArray(item)) {
                _flattenArray(item, target);
            }
            else {
                target.push(item);
            }
        }
    }
    return target;
}
export function isListLikeIterable(obj) {
    if (!isJsObject(obj))
        return false;
    return isArray(obj) ||
        (!(obj instanceof Map) &&
            getSymbolIterator() in obj); // JS Iterable have a Symbol.iterator prop
}
export function areIterablesEqual(a, b, comparator) {
    var iterator1 = a[getSymbolIterator()]();
    var iterator2 = b[getSymbolIterator()]();
    while (true) {
        let item1 = iterator1.next();
        let item2 = iterator2.next();
        if (item1.done && item2.done)
            return true;
        if (item1.done || item2.done)
            return false;
        if (!comparator(item1.value, item2.value))
            return false;
    }
}
export function iterateListLike(obj, fn) {
    if (isArray(obj)) {
        for (var i = 0; i < obj.length; i++) {
            fn(obj[i]);
        }
    }
    else {
        var iterator = obj[getSymbolIterator()]();
        var item;
        while (!((item = iterator.next()).done)) {
            fn(item.value);
        }
    }
}
// Safari and Internet Explorer do not support the iterable parameter to the
// Set constructor.  We work around that by manually adding the items.
var createSetFromList = (function () {
    var test = new Set([1, 2, 3]);
    if (test.size === 3) {
        return function createSetFromList(lst) { return new Set(lst); };
    }
    else {
        return function createSetAndPopulateFromList(lst) {
            var res = new Set(lst);
            if (res.size !== lst.length) {
                for (var i = 0; i < lst.length; i++) {
                    res.add(lst[i]);
                }
            }
            return res;
        };
    }
})();
export class SetWrapper {
    static createFromList(lst) { return createSetFromList(lst); }
    static has(s, key) { return s.has(key); }
    static delete(m, k) { m.delete(k); }
}
//# sourceMappingURL=data:application/json;base64,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