import { assertionsEnabled, isPresent, isBlank } from 'angular2/src/facade/lang';
import { ListWrapper } from 'angular2/src/facade/collection';
import { ChangeDetectionUtil } from './change_detection_util';
import { ChangeDetectorRef_ } from './change_detector_ref';
import { ChangeDetectionError, ExpressionChangedAfterItHasBeenCheckedException, DehydratedException, EventEvaluationErrorContext, EventEvaluationError } from './exceptions';
import { Locals } from './parser/locals';
import { ChangeDetectionStrategy, ChangeDetectorState } from './constants';
import { wtfCreateScope, wtfLeave } from '../profile/profile';
import { isObservable } from './observable_facade';
import { ObservableWrapper } from 'angular2/src/facade/async';
var _scope_check = wtfCreateScope(`ChangeDetector#check(ascii id, bool throwOnChange)`);
class _Context {
    constructor(element, componentElement, context, locals, injector, expression) {
        this.element = element;
        this.componentElement = componentElement;
        this.context = context;
        this.locals = locals;
        this.injector = injector;
        this.expression = expression;
    }
}
export class AbstractChangeDetector {
    constructor(id, numberOfPropertyProtoRecords, bindingTargets, directiveIndices, strategy) {
        this.id = id;
        this.numberOfPropertyProtoRecords = numberOfPropertyProtoRecords;
        this.bindingTargets = bindingTargets;
        this.directiveIndices = directiveIndices;
        this.strategy = strategy;
        this.contentChildren = [];
        this.viewChildren = [];
        // The names of the below fields must be kept in sync with codegen_name_util.ts or
        // change detection will fail.
        this.state = ChangeDetectorState.NeverChecked;
        this.locals = null;
        this.mode = null;
        this.pipes = null;
        this.ref = new ChangeDetectorRef_(this);
    }
    addContentChild(cd) {
        this.contentChildren.push(cd);
        cd.parent = this;
    }
    removeContentChild(cd) { ListWrapper.remove(this.contentChildren, cd); }
    addViewChild(cd) {
        this.viewChildren.push(cd);
        cd.parent = this;
    }
    removeViewChild(cd) { ListWrapper.remove(this.viewChildren, cd); }
    remove() { this.parent.removeContentChild(this); }
    handleEvent(eventName, elIndex, event) {
        if (!this.hydrated()) {
            this.throwDehydratedError(`${this.id} -> ${eventName}`);
        }
        try {
            var locals = new Map();
            locals.set('$event', event);
            var res = !this.handleEventInternal(eventName, elIndex, new Locals(this.locals, locals));
            this.markPathToRootAsCheckOnce();
            return res;
        }
        catch (e) {
            var c = this.dispatcher.getDebugContext(null, elIndex, null);
            var context = isPresent(c) ?
                new EventEvaluationErrorContext(c.element, c.componentElement, c.context, c.locals, c.injector) :
                null;
            throw new EventEvaluationError(eventName, e, e.stack, context);
        }
    }
    handleEventInternal(eventName, elIndex, locals) { return false; }
    detectChanges() { this.runDetectChanges(false); }
    checkNoChanges() {
        if (assertionsEnabled()) {
            this.runDetectChanges(true);
        }
    }
    runDetectChanges(throwOnChange) {
        if (this.mode === ChangeDetectionStrategy.Detached ||
            this.mode === ChangeDetectionStrategy.Checked || this.state === ChangeDetectorState.Errored)
            return;
        var s = _scope_check(this.id, throwOnChange);
        this.detectChangesInRecords(throwOnChange);
        this._detectChangesContentChildren(throwOnChange);
        if (!throwOnChange)
            this.afterContentLifecycleCallbacks();
        this._detectChangesInViewChildren(throwOnChange);
        if (!throwOnChange)
            this.afterViewLifecycleCallbacks();
        if (this.mode === ChangeDetectionStrategy.CheckOnce)
            this.mode = ChangeDetectionStrategy.Checked;
        this.state = ChangeDetectorState.CheckedBefore;
        wtfLeave(s);
    }
    // This method is not intended to be overridden. Subclasses should instead provide an
    // implementation of `detectChangesInRecordsInternal` which does the work of detecting changes
    // and which this method will call.
    // This method expects that `detectChangesInRecordsInternal` will set the property
    // `this.propertyBindingIndex` to the propertyBindingIndex of the first proto record. This is to
    // facilitate error reporting.
    detectChangesInRecords(throwOnChange) {
        if (!this.hydrated()) {
            this.throwDehydratedError(this.id);
        }
        try {
            this.detectChangesInRecordsInternal(throwOnChange);
        }
        catch (e) {
            // throwOnChange errors aren't counted as fatal errors.
            if (!(e instanceof ExpressionChangedAfterItHasBeenCheckedException)) {
                this.state = ChangeDetectorState.Errored;
            }
            this._throwError(e, e.stack);
        }
    }
    // Subclasses should override this method to perform any work necessary to detect and report
    // changes. For example, changes should be reported via `ChangeDetectionUtil.addChange`, lifecycle
    // methods should be called, etc.
    // This implementation should also set `this.propertyBindingIndex` to the propertyBindingIndex of
    // the
    // first proto record to facilitate error reporting. See {@link #detectChangesInRecords}.
    detectChangesInRecordsInternal(throwOnChange) { }
    // This method is not intended to be overridden. Subclasses should instead provide an
    // implementation of `hydrateDirectives`.
    hydrate(context, locals, dispatcher, pipes) {
        this.dispatcher = dispatcher;
        this.mode = ChangeDetectionUtil.changeDetectionMode(this.strategy);
        this.context = context;
        if (this.strategy === ChangeDetectionStrategy.OnPushObserve) {
            this.observeComponent(context);
        }
        this.locals = locals;
        this.pipes = pipes;
        this.hydrateDirectives(dispatcher);
        this.state = ChangeDetectorState.NeverChecked;
    }
    // Subclasses should override this method to hydrate any directives.
    hydrateDirectives(dispatcher) { }
    // This method is not intended to be overridden. Subclasses should instead provide an
    // implementation of `dehydrateDirectives`.
    dehydrate() {
        this.dehydrateDirectives(true);
        // This is an experimental feature. Works only in Dart.
        if (this.strategy === ChangeDetectionStrategy.OnPushObserve) {
            this._unsubsribeFromObservables();
        }
        this._unsubscribeFromOutputs();
        this.dispatcher = null;
        this.context = null;
        this.locals = null;
        this.pipes = null;
    }
    // Subclasses should override this method to dehydrate any directives. This method should reverse
    // any work done in `hydrateDirectives`.
    dehydrateDirectives(destroyPipes) { }
    hydrated() { return isPresent(this.context); }
    destroyRecursive() {
        this.dispatcher.notifyOnDestroy();
        this.dehydrate();
        var children = this.contentChildren;
        for (var i = 0; i < children.length; i++) {
            children[i].destroyRecursive();
        }
        children = this.viewChildren;
        for (var i = 0; i < children.length; i++) {
            children[i].destroyRecursive();
        }
    }
    afterContentLifecycleCallbacks() {
        this.dispatcher.notifyAfterContentChecked();
        this.afterContentLifecycleCallbacksInternal();
    }
    afterContentLifecycleCallbacksInternal() { }
    afterViewLifecycleCallbacks() {
        this.dispatcher.notifyAfterViewChecked();
        this.afterViewLifecycleCallbacksInternal();
    }
    afterViewLifecycleCallbacksInternal() { }
    /** @internal */
    _detectChangesContentChildren(throwOnChange) {
        var c = this.contentChildren;
        for (var i = 0; i < c.length; ++i) {
            c[i].runDetectChanges(throwOnChange);
        }
    }
    /** @internal */
    _detectChangesInViewChildren(throwOnChange) {
        var c = this.viewChildren;
        for (var i = 0; i < c.length; ++i) {
            c[i].runDetectChanges(throwOnChange);
        }
    }
    markAsCheckOnce() { this.mode = ChangeDetectionStrategy.CheckOnce; }
    markPathToRootAsCheckOnce() {
        var c = this;
        while (isPresent(c) && c.mode !== ChangeDetectionStrategy.Detached) {
            if (c.mode === ChangeDetectionStrategy.Checked)
                c.mode = ChangeDetectionStrategy.CheckOnce;
            c = c.parent;
        }
    }
    // This is an experimental feature. Works only in Dart.
    _unsubsribeFromObservables() {
        if (isPresent(this.subscriptions)) {
            for (var i = 0; i < this.subscriptions.length; ++i) {
                var s = this.subscriptions[i];
                if (isPresent(this.subscriptions[i])) {
                    s.cancel();
                    this.subscriptions[i] = null;
                }
            }
        }
    }
    _unsubscribeFromOutputs() {
        if (isPresent(this.outputSubscriptions)) {
            for (var i = 0; i < this.outputSubscriptions.length; ++i) {
                ObservableWrapper.dispose(this.outputSubscriptions[i]);
                this.outputSubscriptions[i] = null;
            }
        }
    }
    // This is an experimental feature. Works only in Dart.
    observeValue(value, index) {
        if (isObservable(value)) {
            this._createArrayToStoreObservables();
            if (isBlank(this.subscriptions[index])) {
                this.streams[index] = value.changes;
                this.subscriptions[index] = value.changes.listen((_) => this.ref.markForCheck());
            }
            else if (this.streams[index] !== value.changes) {
                this.subscriptions[index].cancel();
                this.streams[index] = value.changes;
                this.subscriptions[index] = value.changes.listen((_) => this.ref.markForCheck());
            }
        }
        return value;
    }
    // This is an experimental feature. Works only in Dart.
    observeDirective(value, index) {
        if (isObservable(value)) {
            this._createArrayToStoreObservables();
            var arrayIndex = this.numberOfPropertyProtoRecords + index + 2; // +1 is component
            this.streams[arrayIndex] = value.changes;
            this.subscriptions[arrayIndex] = value.changes.listen((_) => this.ref.markForCheck());
        }
        return value;
    }
    // This is an experimental feature. Works only in Dart.
    observeComponent(value) {
        if (isObservable(value)) {
            this._createArrayToStoreObservables();
            var index = this.numberOfPropertyProtoRecords + 1;
            this.streams[index] = value.changes;
            this.subscriptions[index] = value.changes.listen((_) => this.ref.markForCheck());
        }
        return value;
    }
    _createArrayToStoreObservables() {
        if (isBlank(this.subscriptions)) {
            this.subscriptions = ListWrapper.createFixedSize(this.numberOfPropertyProtoRecords +
                this.directiveIndices.length + 2);
            this.streams = ListWrapper.createFixedSize(this.numberOfPropertyProtoRecords +
                this.directiveIndices.length + 2);
        }
    }
    getDirectiveFor(directives, index) {
        return directives.getDirectiveFor(this.directiveIndices[index]);
    }
    getDetectorFor(directives, index) {
        return directives.getDetectorFor(this.directiveIndices[index]);
    }
    notifyDispatcher(value) {
        this.dispatcher.notifyOnBinding(this._currentBinding(), value);
    }
    logBindingUpdate(value) {
        this.dispatcher.logBindingUpdate(this._currentBinding(), value);
    }
    addChange(changes, oldValue, newValue) {
        if (isBlank(changes)) {
            changes = {};
        }
        changes[this._currentBinding().name] = ChangeDetectionUtil.simpleChange(oldValue, newValue);
        return changes;
    }
    _throwError(exception, stack) {
        var error;
        try {
            var c = this.dispatcher.getDebugContext(null, this._currentBinding().elementIndex, null);
            var context = isPresent(c) ? new _Context(c.element, c.componentElement, c.context, c.locals, c.injector, this._currentBinding().debug) :
                null;
            error = new ChangeDetectionError(this._currentBinding().debug, exception, stack, context);
        }
        catch (e) {
            // if an error happens during getting the debug context, we throw a ChangeDetectionError
            // without the extra information.
            error = new ChangeDetectionError(null, exception, stack, null);
        }
        throw error;
    }
    throwOnChangeError(oldValue, newValue) {
        throw new ExpressionChangedAfterItHasBeenCheckedException(this._currentBinding().debug, oldValue, newValue, null);
    }
    throwDehydratedError(detail) { throw new DehydratedException(detail); }
    _currentBinding() {
        return this.bindingTargets[this.propertyBindingIndex];
    }
}
//# sourceMappingURL=data:application/json;base64,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