import { assertionsEnabled, isPresent, isBlank } from 'angular2/src/facade/lang';
import { ListWrapper } from 'angular2/src/facade/collection';
import { ChangeDetectionUtil } from './change_detection_util';
import { ChangeDetectorRef_ } from './change_detector_ref';
import { ChangeDetectionError, ExpressionChangedAfterItHasBeenCheckedException, DehydratedException, EventEvaluationErrorContext, EventEvaluationError } from './exceptions';
import { Locals } from './parser/locals';
import { ChangeDetectionStrategy, ChangeDetectorState } from './constants';
import { wtfCreateScope, wtfLeave } from '../profile/profile';
import { ObservableWrapper } from 'angular2/src/facade/async';
var _scope_check = wtfCreateScope(`ChangeDetector#check(ascii id, bool throwOnChange)`);
class _Context {
    constructor(element, componentElement, context, locals, injector, expression) {
        this.element = element;
        this.componentElement = componentElement;
        this.context = context;
        this.locals = locals;
        this.injector = injector;
        this.expression = expression;
    }
}
export class AbstractChangeDetector {
    constructor(id, numberOfPropertyProtoRecords, bindingTargets, directiveIndices, strategy) {
        this.id = id;
        this.numberOfPropertyProtoRecords = numberOfPropertyProtoRecords;
        this.bindingTargets = bindingTargets;
        this.directiveIndices = directiveIndices;
        this.strategy = strategy;
        this.contentChildren = [];
        this.viewChildren = [];
        // The names of the below fields must be kept in sync with codegen_name_util.ts or
        // change detection will fail.
        this.state = ChangeDetectorState.NeverChecked;
        this.locals = null;
        this.mode = null;
        this.pipes = null;
        this.ref = new ChangeDetectorRef_(this);
    }
    addContentChild(cd) {
        this.contentChildren.push(cd);
        cd.parent = this;
    }
    removeContentChild(cd) { ListWrapper.remove(this.contentChildren, cd); }
    addViewChild(cd) {
        this.viewChildren.push(cd);
        cd.parent = this;
    }
    removeViewChild(cd) { ListWrapper.remove(this.viewChildren, cd); }
    remove() { this.parent.removeContentChild(this); }
    handleEvent(eventName, elIndex, event) {
        if (!this.hydrated()) {
            this.throwDehydratedError(`${this.id} -> ${eventName}`);
        }
        try {
            var locals = new Map();
            locals.set('$event', event);
            var res = !this.handleEventInternal(eventName, elIndex, new Locals(this.locals, locals));
            this.markPathToRootAsCheckOnce();
            return res;
        }
        catch (e) {
            var c = this.dispatcher.getDebugContext(null, elIndex, null);
            var context = isPresent(c) ?
                new EventEvaluationErrorContext(c.element, c.componentElement, c.context, c.locals, c.injector) :
                null;
            throw new EventEvaluationError(eventName, e, e.stack, context);
        }
    }
    handleEventInternal(eventName, elIndex, locals) { return false; }
    detectChanges() { this.runDetectChanges(false); }
    checkNoChanges() {
        if (assertionsEnabled()) {
            this.runDetectChanges(true);
        }
    }
    runDetectChanges(throwOnChange) {
        if (this.mode === ChangeDetectionStrategy.Detached ||
            this.mode === ChangeDetectionStrategy.Checked || this.state === ChangeDetectorState.Errored)
            return;
        var s = _scope_check(this.id, throwOnChange);
        this.detectChangesInRecords(throwOnChange);
        this._detectChangesContentChildren(throwOnChange);
        if (!throwOnChange)
            this.afterContentLifecycleCallbacks();
        this._detectChangesInViewChildren(throwOnChange);
        if (!throwOnChange)
            this.afterViewLifecycleCallbacks();
        if (this.mode === ChangeDetectionStrategy.CheckOnce)
            this.mode = ChangeDetectionStrategy.Checked;
        this.state = ChangeDetectorState.CheckedBefore;
        wtfLeave(s);
    }
    // This method is not intended to be overridden. Subclasses should instead provide an
    // implementation of `detectChangesInRecordsInternal` which does the work of detecting changes
    // and which this method will call.
    // This method expects that `detectChangesInRecordsInternal` will set the property
    // `this.propertyBindingIndex` to the propertyBindingIndex of the first proto record. This is to
    // facilitate error reporting.
    detectChangesInRecords(throwOnChange) {
        if (!this.hydrated()) {
            this.throwDehydratedError(this.id);
        }
        try {
            this.detectChangesInRecordsInternal(throwOnChange);
        }
        catch (e) {
            // throwOnChange errors aren't counted as fatal errors.
            if (!(e instanceof ExpressionChangedAfterItHasBeenCheckedException)) {
                this.state = ChangeDetectorState.Errored;
            }
            this._throwError(e, e.stack);
        }
    }
    // Subclasses should override this method to perform any work necessary to detect and report
    // changes. For example, changes should be reported via `ChangeDetectionUtil.addChange`, lifecycle
    // methods should be called, etc.
    // This implementation should also set `this.propertyBindingIndex` to the propertyBindingIndex of
    // the
    // first proto record to facilitate error reporting. See {@link #detectChangesInRecords}.
    detectChangesInRecordsInternal(throwOnChange) { }
    // This method is not intended to be overridden. Subclasses should instead provide an
    // implementation of `hydrateDirectives`.
    hydrate(context, locals, dispatcher, pipes) {
        this.dispatcher = dispatcher;
        this.mode = ChangeDetectionUtil.changeDetectionMode(this.strategy);
        this.context = context;
        this.locals = locals;
        this.pipes = pipes;
        this.hydrateDirectives(dispatcher);
        this.state = ChangeDetectorState.NeverChecked;
    }
    // Subclasses should override this method to hydrate any directives.
    hydrateDirectives(dispatcher) { }
    // This method is not intended to be overridden. Subclasses should instead provide an
    // implementation of `dehydrateDirectives`.
    dehydrate() {
        this.dehydrateDirectives(true);
        this._unsubscribeFromOutputs();
        this.dispatcher = null;
        this.context = null;
        this.locals = null;
        this.pipes = null;
    }
    // Subclasses should override this method to dehydrate any directives. This method should reverse
    // any work done in `hydrateDirectives`.
    dehydrateDirectives(destroyPipes) { }
    hydrated() { return isPresent(this.context); }
    destroyRecursive() {
        this.dispatcher.notifyOnDestroy();
        this.dehydrate();
        var children = this.contentChildren;
        for (var i = 0; i < children.length; i++) {
            children[i].destroyRecursive();
        }
        children = this.viewChildren;
        for (var i = 0; i < children.length; i++) {
            children[i].destroyRecursive();
        }
    }
    afterContentLifecycleCallbacks() {
        this.dispatcher.notifyAfterContentChecked();
        this.afterContentLifecycleCallbacksInternal();
    }
    afterContentLifecycleCallbacksInternal() { }
    afterViewLifecycleCallbacks() {
        this.dispatcher.notifyAfterViewChecked();
        this.afterViewLifecycleCallbacksInternal();
    }
    afterViewLifecycleCallbacksInternal() { }
    /** @internal */
    _detectChangesContentChildren(throwOnChange) {
        var c = this.contentChildren;
        for (var i = 0; i < c.length; ++i) {
            c[i].runDetectChanges(throwOnChange);
        }
    }
    /** @internal */
    _detectChangesInViewChildren(throwOnChange) {
        var c = this.viewChildren;
        for (var i = 0; i < c.length; ++i) {
            c[i].runDetectChanges(throwOnChange);
        }
    }
    markAsCheckOnce() { this.mode = ChangeDetectionStrategy.CheckOnce; }
    markPathToRootAsCheckOnce() {
        var c = this;
        while (isPresent(c) && c.mode !== ChangeDetectionStrategy.Detached) {
            if (c.mode === ChangeDetectionStrategy.Checked)
                c.mode = ChangeDetectionStrategy.CheckOnce;
            c = c.parent;
        }
    }
    _unsubscribeFromOutputs() {
        if (isPresent(this.outputSubscriptions)) {
            for (var i = 0; i < this.outputSubscriptions.length; ++i) {
                ObservableWrapper.dispose(this.outputSubscriptions[i]);
                this.outputSubscriptions[i] = null;
            }
        }
    }
    getDirectiveFor(directives, index) {
        return directives.getDirectiveFor(this.directiveIndices[index]);
    }
    getDetectorFor(directives, index) {
        return directives.getDetectorFor(this.directiveIndices[index]);
    }
    notifyDispatcher(value) {
        this.dispatcher.notifyOnBinding(this._currentBinding(), value);
    }
    logBindingUpdate(value) {
        this.dispatcher.logBindingUpdate(this._currentBinding(), value);
    }
    addChange(changes, oldValue, newValue) {
        if (isBlank(changes)) {
            changes = {};
        }
        changes[this._currentBinding().name] = ChangeDetectionUtil.simpleChange(oldValue, newValue);
        return changes;
    }
    _throwError(exception, stack) {
        var error;
        try {
            var c = this.dispatcher.getDebugContext(null, this._currentBinding().elementIndex, null);
            var context = isPresent(c) ? new _Context(c.element, c.componentElement, c.context, c.locals, c.injector, this._currentBinding().debug) :
                null;
            error = new ChangeDetectionError(this._currentBinding().debug, exception, stack, context);
        }
        catch (e) {
            // if an error happens during getting the debug context, we throw a ChangeDetectionError
            // without the extra information.
            error = new ChangeDetectionError(null, exception, stack, null);
        }
        throw error;
    }
    throwOnChangeError(oldValue, newValue) {
        throw new ExpressionChangedAfterItHasBeenCheckedException(this._currentBinding().debug, oldValue, newValue, null);
    }
    throwDehydratedError(detail) { throw new DehydratedException(detail); }
    _currentBinding() {
        return this.bindingTargets[this.propertyBindingIndex];
    }
}
//# sourceMappingURL=data:application/json;base64,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