import { isJsObject, global, isPresent, isBlank, isArray, getSymbolIterator } from 'angular2/src/facade/lang';
export var Map = global.Map;
export var Set = global.Set;
// Safari and Internet Explorer do not support the iterable parameter to the
// Map constructor.  We work around that by manually adding the items.
var createMapFromPairs = (function () {
    try {
        if (new Map([[1, 2]]).size === 1) {
            return function createMapFromPairs(pairs) { return new Map(pairs); };
        }
    }
    catch (e) {
    }
    return function createMapAndPopulateFromPairs(pairs) {
        var map = new Map();
        for (var i = 0; i < pairs.length; i++) {
            var pair = pairs[i];
            map.set(pair[0], pair[1]);
        }
        return map;
    };
})();
var createMapFromMap = (function () {
    try {
        if (new Map(new Map())) {
            return function createMapFromMap(m) { return new Map(m); };
        }
    }
    catch (e) {
    }
    return function createMapAndPopulateFromMap(m) {
        var map = new Map();
        m.forEach((v, k) => { map.set(k, v); });
        return map;
    };
})();
var _clearValues = (function () {
    if ((new Map()).keys().next) {
        return function _clearValues(m) {
            var keyIterator = m.keys();
            var k;
            while (!((k = keyIterator.next()).done)) {
                m.set(k.value, null);
            }
        };
    }
    else {
        return function _clearValuesWithForeEach(m) {
            m.forEach((v, k) => { m.set(k, null); });
        };
    }
})();
// Safari doesn't implement MapIterator.next(), which is used is Traceur's polyfill of Array.from
// TODO(mlaval): remove the work around once we have a working polyfill of Array.from
var _arrayFromMap = (function () {
    try {
        if ((new Map()).values().next) {
            return function createArrayFromMap(m, getValues) {
                return getValues ? Array.from(m.values()) : Array.from(m.keys());
            };
        }
    }
    catch (e) {
    }
    return function createArrayFromMapWithForeach(m, getValues) {
        var res = ListWrapper.createFixedSize(m.size), i = 0;
        m.forEach((v, k) => {
            res[i] = getValues ? v : k;
            i++;
        });
        return res;
    };
})();
export class MapWrapper {
    static clone(m) { return createMapFromMap(m); }
    static createFromStringMap(stringMap) {
        var result = new Map();
        for (var prop in stringMap) {
            result.set(prop, stringMap[prop]);
        }
        return result;
    }
    static toStringMap(m) {
        var r = {};
        m.forEach((v, k) => r[k] = v);
        return r;
    }
    static createFromPairs(pairs) { return createMapFromPairs(pairs); }
    static clearValues(m) { _clearValues(m); }
    static iterable(m) { return m; }
    static keys(m) { return _arrayFromMap(m, false); }
    static values(m) { return _arrayFromMap(m, true); }
}
/**
 * Wraps Javascript Objects
 */
export class StringMapWrapper {
    static create() {
        // Note: We are not using Object.create(null) here due to
        // performance!
        // http://jsperf.com/ng2-object-create-null
        return {};
    }
    static contains(map, key) {
        return map.hasOwnProperty(key);
    }
    static get(map, key) {
        return map.hasOwnProperty(key) ? map[key] : undefined;
    }
    static set(map, key, value) { map[key] = value; }
    static keys(map) { return Object.keys(map); }
    static isEmpty(map) {
        for (var prop in map) {
            return false;
        }
        return true;
    }
    static delete(map, key) { delete map[key]; }
    static forEach(map, callback) {
        for (var prop in map) {
            if (map.hasOwnProperty(prop)) {
                callback(map[prop], prop);
            }
        }
    }
    static merge(m1, m2) {
        var m = {};
        for (var attr in m1) {
            if (m1.hasOwnProperty(attr)) {
                m[attr] = m1[attr];
            }
        }
        for (var attr in m2) {
            if (m2.hasOwnProperty(attr)) {
                m[attr] = m2[attr];
            }
        }
        return m;
    }
    static equals(m1, m2) {
        var k1 = Object.keys(m1);
        var k2 = Object.keys(m2);
        if (k1.length != k2.length) {
            return false;
        }
        var key;
        for (var i = 0; i < k1.length; i++) {
            key = k1[i];
            if (m1[key] !== m2[key]) {
                return false;
            }
        }
        return true;
    }
}
export class ListWrapper {
    // JS has no way to express a statically fixed size list, but dart does so we
    // keep both methods.
    static createFixedSize(size) { return new Array(size); }
    static createGrowableSize(size) { return new Array(size); }
    static clone(array) { return array.slice(0); }
    static createImmutable(array) {
        var result = ListWrapper.clone(array);
        Object.seal(result);
        return result;
    }
    static forEachWithIndex(array, fn) {
        for (var i = 0; i < array.length; i++) {
            fn(array[i], i);
        }
    }
    static first(array) {
        if (!array)
            return null;
        return array[0];
    }
    static last(array) {
        if (!array || array.length == 0)
            return null;
        return array[array.length - 1];
    }
    static indexOf(array, value, startIndex = 0) {
        return array.indexOf(value, startIndex);
    }
    static contains(list, el) { return list.indexOf(el) !== -1; }
    static reversed(array) {
        var a = ListWrapper.clone(array);
        return a.reverse();
    }
    static concat(a, b) { return a.concat(b); }
    static insert(list, index, value) { list.splice(index, 0, value); }
    static removeAt(list, index) {
        var res = list[index];
        list.splice(index, 1);
        return res;
    }
    static removeAll(list, items) {
        for (var i = 0; i < items.length; ++i) {
            var index = list.indexOf(items[i]);
            list.splice(index, 1);
        }
    }
    static remove(list, el) {
        var index = list.indexOf(el);
        if (index > -1) {
            list.splice(index, 1);
            return true;
        }
        return false;
    }
    static clear(list) { list.length = 0; }
    static isEmpty(list) { return list.length == 0; }
    static fill(list, value, start = 0, end = null) {
        list.fill(value, start, end === null ? list.length : end);
    }
    static equals(a, b) {
        if (a.length != b.length)
            return false;
        for (var i = 0; i < a.length; ++i) {
            if (a[i] !== b[i])
                return false;
        }
        return true;
    }
    static slice(l, from = 0, to = null) {
        return l.slice(from, to === null ? undefined : to);
    }
    static splice(l, from, length) { return l.splice(from, length); }
    static sort(l, compareFn) {
        if (isPresent(compareFn)) {
            l.sort(compareFn);
        }
        else {
            l.sort();
        }
    }
    static toString(l) { return l.toString(); }
    static toJSON(l) { return JSON.stringify(l); }
    static maximum(list, predicate) {
        if (list.length == 0) {
            return null;
        }
        var solution = null;
        var maxValue = -Infinity;
        for (var index = 0; index < list.length; index++) {
            var candidate = list[index];
            if (isBlank(candidate)) {
                continue;
            }
            var candidateValue = predicate(candidate);
            if (candidateValue > maxValue) {
                solution = candidate;
                maxValue = candidateValue;
            }
        }
        return solution;
    }
    static isImmutable(list) { return Object.isSealed(list); }
}
export function isListLikeIterable(obj) {
    if (!isJsObject(obj))
        return false;
    return isArray(obj) ||
        (!(obj instanceof Map) &&
            getSymbolIterator() in obj); // JS Iterable have a Symbol.iterator prop
}
export function areIterablesEqual(a, b, comparator) {
    var iterator1 = a[getSymbolIterator()]();
    var iterator2 = b[getSymbolIterator()]();
    while (true) {
        let item1 = iterator1.next();
        let item2 = iterator2.next();
        if (item1.done && item2.done)
            return true;
        if (item1.done || item2.done)
            return false;
        if (!comparator(item1.value, item2.value))
            return false;
    }
}
export function iterateListLike(obj, fn) {
    if (isArray(obj)) {
        for (var i = 0; i < obj.length; i++) {
            fn(obj[i]);
        }
    }
    else {
        var iterator = obj[getSymbolIterator()]();
        var item;
        while (!((item = iterator.next()).done)) {
            fn(item.value);
        }
    }
}
// Safari and Internet Explorer do not support the iterable parameter to the
// Set constructor.  We work around that by manually adding the items.
var createSetFromList = (function () {
    var test = new Set([1, 2, 3]);
    if (test.size === 3) {
        return function createSetFromList(lst) { return new Set(lst); };
    }
    else {
        return function createSetAndPopulateFromList(lst) {
            var res = new Set(lst);
            if (res.size !== lst.length) {
                for (var i = 0; i < lst.length; i++) {
                    res.add(lst[i]);
                }
            }
            return res;
        };
    }
})();
export class SetWrapper {
    static createFromList(lst) { return createSetFromList(lst); }
    static has(s, key) { return s.has(key); }
    static delete(m, k) { m.delete(k); }
}
//# sourceMappingURL=data:application/json;base64,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