'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.GraphQLConfig = exports.GRAPHQL_CONFIG_NAME = undefined;

var _path = require('path');

var _path2 = _interopRequireDefault(_path);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _classCallCheck(instance, Constructor) { if (!(instance instanceof Constructor)) { throw new TypeError("Cannot call a class as a function"); } }

/**
 *  Copyright (c) Facebook, Inc.
 *  All rights reserved.
 *
 *  This source code is licensed under the license found in the
 *  LICENSE file in the root directory of this source tree.
 *
 *  
 */

var PROJECTS_NAME = 'projects';
var CUSTOM_VALIDATION_RULES_NAME = 'customValidationRules';

var GRAPHQL_CONFIG_NAME = exports.GRAPHQL_CONFIG_NAME = '.graphqlconfig';

/**
 * GraphQLConfig class holds various information about the GraphQL app by
 * reading the GraphQL configuration file (.graphqlconfig) located somewhere
 * in the app directory tree. For more information of how to write a
 * configuration for your GraphQL app, please refer to `GraphQLConfigTypes.js`
 * file.
 */

var GraphQLConfig = exports.GraphQLConfig = function () {
  function GraphQLConfig(config, rootDir) {
    _classCallCheck(this, GraphQLConfig);

    this._rootDir = rootDir;
    this._config = config;
  }

  // GraphQL language server utilizes an `extensions` config option to customize
  // for the situation with many apps with a shared code in one repository.
  // This function searches for the additional app configurations and
  // returns the name of the app configuration if found.


  GraphQLConfig.prototype.getAppConfigNameByFilePath = function getAppConfigNameByFilePath(filePath) {
    var _this = this;

    var appConfigs = this._config[PROJECTS_NAME];
    if (!appConfigs) {
      return null;
    }

    var appConfigNames = Object.keys(appConfigs);

    var name = appConfigNames.find(function (appName) {
      var appConfig = _this._config[PROJECTS_NAME] && _this._config[PROJECTS_NAME][appName];
      if (appConfig) {
        return _this.isFileInIncludeDirs(filePath, appName);
      }
      return false;
    });

    return name || null;
  };

  GraphQLConfig.prototype.getRootDir = function getRootDir() {
    return this._rootDir;
  };

  GraphQLConfig.prototype.getName = function getName() {
    return 'GraphQLConfig';
  };

  GraphQLConfig.prototype.getConfig = function getConfig() {
    return this._config;
  };

  /**
   * Below getters inspect the GraphQL configuration in two steps:
   * 1. If `appName` is provided, look for the app configuration and try
   *    returning app-specific properties, overriding the configuration options
   *    at the top level.
   * 2. If step 1 fails and/or `appName` is not passed in, look for the property
   *    at the top level (the "root" configuration).
   */

  GraphQLConfig.prototype.getIncludeDirs = function getIncludeDirs(appName) {
    return this._getPropertyFromConfig('includeDirs', appName, []);
  };

  GraphQLConfig.prototype.getExcludeDirs = function getExcludeDirs(appName) {
    return this._getPropertyFromConfig('excludeDirs', appName, []);
  };

  GraphQLConfig.prototype.getSchemaPath = function getSchemaPath(appName) {
    return this._getPropertyFromConfig('schemaPath', appName, null);
  };

  GraphQLConfig.prototype.isFileInIncludeDirs = function isFileInIncludeDirs(fileName, appName) {
    if (appName) {
      if (this._config[PROJECTS_NAME] && this._config[PROJECTS_NAME][appName] && this._config[PROJECTS_NAME][appName].includeDirs) {
        return this._config[PROJECTS_NAME][appName].includeDirs.some(function (dirPath) {
          return fileName.indexOf(dirPath) !== -1;
        });
      }
    }
    return this._config.includeDirs ? this._config.includeDirs.some(function (dirPath) {
      return fileName.indexOf(dirPath) !== -1;
    }) : false;
  };

  GraphQLConfig.prototype.getCustomValidationRulesModulePath = function getCustomValidationRulesModulePath(appName) {
    var modulePath = void 0;
    if (appName) {
      if (this._config[PROJECTS_NAME] && this._config[PROJECTS_NAME][appName] && this._config[PROJECTS_NAME][appName][CUSTOM_VALIDATION_RULES_NAME]) {
        var appConfig = this._config[PROJECTS_NAME][appName];
        modulePath = appConfig[CUSTOM_VALIDATION_RULES_NAME];
      }
    } else {
      modulePath = this._config[CUSTOM_VALIDATION_RULES_NAME];
    }
    if (!modulePath) {
      return null;
    }
    return this._normalizePath(modulePath);
  };

  GraphQLConfig.prototype._normalizePath = function _normalizePath(modulePath) {
    var resolvedPath = void 0;
    if (modulePath.startsWith('~')) {
      // home directory
      var homeDirPath = process.platform === 'win32' ? process.env.USERPROFILE : process.env.HOME;
      resolvedPath = _path2.default.join(homeDirPath || '', modulePath.slice(1));
    } else if (modulePath.startsWith('./')) {
      // relative local directory
      resolvedPath = _path2.default.join(this._rootDir, modulePath);
    } else {
      // `/` or an actual module name (node_modules)
      resolvedPath = modulePath;
    }

    return resolvedPath;
  };

  GraphQLConfig.prototype._getPropertyFromConfig = function _getPropertyFromConfig(key, appName, defaultValue) {
    if (appName && this._config[PROJECTS_NAME] && this._config[PROJECTS_NAME][appName] && this._config[PROJECTS_NAME][appName][key]) {
      return this._config[PROJECTS_NAME][appName][key];
    }
    return this._config[key] || defaultValue;
  };

  return GraphQLConfig;
}();