/**
 * Copyright 2013-present, Facebook, Inc.
 * All rights reserved.
 *
 * This source code is licensed under the BSD-style license found in the
 * LICENSE file in the root directory of this source tree. An additional grant
 * of patent rights can be found in the PATENTS file in the same directory.
 *
 * @providesModule ReactDOMDebugTool
 */

'use strict';

var ReactDOMUnknownPropertyDevtool = require('./ReactDOMUnknownPropertyDevtool');
var ReactDOMSVGDeprecatedAttributeDevtool = require('./ReactDOMSVGDeprecatedAttributeDevtool');

var warning = require('fbjs/lib/warning');

var eventHandlers = [];
var handlerDoesThrowForEvent = {};

function emitEvent(handlerFunctionName, arg1, arg2, arg3, arg4, arg5) {
  if (process.env.NODE_ENV !== 'production') {
    eventHandlers.forEach(function (handler) {
      try {
        if (handler[handlerFunctionName]) {
          handler[handlerFunctionName](arg1, arg2, arg3, arg4, arg5);
        }
      } catch (e) {
        process.env.NODE_ENV !== 'production' ? warning(!handlerDoesThrowForEvent[handlerFunctionName], 'exception thrown by devtool while handling %s: %s', handlerFunctionName, e.message) : undefined;
        handlerDoesThrowForEvent[handlerFunctionName] = true;
      }
    });
  }
}

var ReactDOMDebugTool = {
  addDevtool: function (devtool) {
    eventHandlers.push(devtool);
  },
  removeDevtool: function (devtool) {
    for (var i = 0; i < eventHandlers.length; i++) {
      if (eventHandlers[i] === devtool) {
        eventHandlers.splice(i, 1);
        i--;
      }
    }
  },
  onCreateMarkupForProperty: function (name, value) {
    emitEvent('onCreateMarkupForProperty', name, value);
  },
  onCreateMarkupForSVGAttribute: function (name, value) {
    emitEvent('onCreateMarkupForSVGAttribute', name, value);
  },
  onSetValueForProperty: function (node, name, value) {
    emitEvent('onSetValueForProperty', node, name, value);
  },
  onSetValueForSVGAttribute: function (node, name, value) {
    emitEvent('onSetValueForSVGAttribute', node, name, value);
  },
  onDeleteValueForProperty: function (node, name) {
    emitEvent('onDeleteValueForProperty', node, name);
  }
};

ReactDOMDebugTool.addDevtool(ReactDOMUnknownPropertyDevtool);
ReactDOMDebugTool.addDevtool(ReactDOMSVGDeprecatedAttributeDevtool);

module.exports = ReactDOMDebugTool;