"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const childProcess = require("child_process");
const fs = require("fs");
const loglevel = require("loglevel");
const path = require("path");
const rimraf = require("rimraf");
const semver = require("semver");
const http_utils_1 = require("./utils/http_utils");
const log = loglevel.getLogger('webdriver-manager');
class Appium {
    constructor(providerConfig) {
        this.requestUrl = 'http://registry.npmjs.org/appium';
        if (providerConfig) {
            this.ignoreSSL = providerConfig.ignoreSSL;
            if (providerConfig.outDir) {
                this.outDir = providerConfig.outDir;
            }
            if (providerConfig.proxy) {
                this.proxy = providerConfig.proxy;
            }
            if (providerConfig.requestUrl) {
                this.requestUrl = providerConfig.requestUrl;
            }
        }
    }
    /**
     * If no valid version is provided get version from appium
     */
    async getVersion() {
        const body = await http_utils_1.requestBody(this.requestUrl, { proxy: this.proxy, ignoreSSL: this.ignoreSSL });
        return JSON.parse(body)['dist-tags']['latest'];
    }
    /**
     * Creates appium directory and package.json file.
     * @param version Optional to provide the version number or latest.
     */
    async setup(version) {
        if (!semver.valid(version)) {
            version = await this.getVersion();
        }
        this.outDirAppium = path.resolve(this.outDir, 'appium');
        try {
            rimraf.sync(this.outDirAppium);
        }
        catch (err) {
        }
        fs.mkdirSync(this.outDirAppium);
        const packageJson = {
            scripts: { appium: 'appium' },
            dependencies: { appium: '^' + version }
        };
        fs.writeFileSync(path.resolve(this.outDirAppium, 'package.json'), JSON.stringify(packageJson));
    }
    /**
     * Creates an appium/package.json file and installs the appium dependency.
     * @param version Optional to provide the version number or latest.
     */
    async updateBinary(version) {
        log.info('appium: installing appium');
        await this.setup(version);
        childProcess.execSync('npm install', { cwd: this.outDirAppium });
    }
}
exports.Appium = Appium;
//# sourceMappingURL=appium.js.map