import { ProviderConfig, ProviderInterface } from './provider';
export declare class ChromeDriver implements ProviderInterface {
    cacheFileName: string;
    configFileName: string;
    ignoreSSL: boolean;
    osType: string;
    osArch: string;
    outDir: string;
    proxy: string;
    requestUrl: string;
    seleniumFlag: string;
    constructor(providerConfig?: ProviderConfig);
    /**
     * Should update the cache and download, find the version to download,
     * then download that binary.
     * @param version Optional to provide the version number or latest.
     */
    updateBinary(version?: string): Promise<void>;
    /**
     * Gets the binary file path.
     * @param version Optional to provide the version number or latest.
     */
    getBinaryPath(version?: string): string | null;
    /**
     * Gets a comma delimited list of versions downloaded. Also has the "latest"
     * downloaded noted.
     */
    getStatus(): string | null;
    /**
     * Get a line delimited list of files removed.
     */
    cleanFiles(): string;
}
/**
 * Helps translate the os type and arch to the download name associated
 * with composing the download link.
 * @param ostype The operating stystem type.
 * @param osarch The chip architecture.
 * @returns The download name associated with composing the download link.
 */
export declare function osHelper(ostype: string, osarch: string): string;
/**
 * Captures the version name which includes the semantic version and extra
 * metadata. So an example for 12.34/chromedriver_linux64.zip,
 * the version is 12.34.
 *
 * The new version is 70.0.3538.16/chromedriver_linux64.zip. This will return
 * 70.0.3538.16.
 * @param xmlKey The xml key including the partial url.
 */
export declare function versionParser(xmlKey: string): string;
/**
 * Captures the version name which includes the semantic version and extra
 * metadata. So an example for 12.34/chromedriver_linux64.zip,
 * the version is 12.34.0.
 *
 * The new version is 70.0.3538.16/chromedriver_linux64.zip. This will return
 * 70.0.3538.
 * @param xmlKey The xml key including the partial url.
 */
export declare function semanticVersionParser(xmlKey: string): string | null;
/**
 * Matches the installed binaries depending on the operating system.
 * @param ostype The operating stystem type.
 */
export declare function matchBinaries(ostype: string): RegExp | null;
/**
 * Specifically to chromedriver, when downloading a version, the version
 * you give webdriver-manager is not in the same as the output from the
 * semanticVersionParser. So getting the version to the versionList will not
 * be just a dictionary look up versionList[version]. The version will have
 * to be formatted.
 *
 * Example:
 *   2.44 will return with the formatted semantic version 2.44.0
 *   70.0.1.1 will return with the formatted semantic version 70.0.1
 *
 * @param version The actual version.
 */
export declare function formatVersion(version: string): string | null;
