"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const fs = require("fs");
const os = require("os");
const path = require("path");
const provider_1 = require("./provider");
const cloud_storage_xml_1 = require("./utils/cloud_storage_xml");
const file_utils_1 = require("./utils/file_utils");
const http_utils_1 = require("./utils/http_utils");
const version_list_1 = require("./utils/version_list");
class IEDriver {
    constructor(providerConfig) {
        this.cacheFileName = 'iedriver.xml';
        this.configFileName = 'iedriver.config.json';
        this.ignoreSSL = false;
        this.osType = os.type();
        this.osArch = os.arch();
        this.outDir = provider_1.OUT_DIR;
        this.proxy = null;
        this.requestUrl = 'https://selenium-release.storage.googleapis.com/';
        this.seleniumFlag = '-Dwebdriver.ie.driver';
        if (providerConfig) {
            if (providerConfig.cacheFileName) {
                this.cacheFileName = providerConfig.cacheFileName;
            }
            if (providerConfig.configFileName) {
                this.configFileName = providerConfig.configFileName;
            }
            this.ignoreSSL = providerConfig.ignoreSSL;
            if (providerConfig.osArch) {
                this.osArch = providerConfig.osArch;
            }
            if (providerConfig.osType) {
                this.osType = providerConfig.osType;
            }
            if (providerConfig.outDir) {
                this.outDir = providerConfig.outDir;
            }
            if (providerConfig.proxy) {
                this.proxy = providerConfig.proxy;
            }
            if (providerConfig.requestUrl) {
                this.requestUrl = providerConfig.requestUrl;
            }
        }
    }
    /**
     * Should update the cache and download, find the version to download,
     * then download that binary.
     * @param version Optional to provide the version number or latest.
     */
    async updateBinary(version) {
        await cloud_storage_xml_1.updateXml(this.requestUrl, {
            fileName: path.resolve(this.outDir, this.cacheFileName),
            ignoreSSL: this.ignoreSSL,
            proxy: this.proxy
        });
        const versionList = cloud_storage_xml_1.convertXmlToVersionList(path.resolve(this.outDir, this.cacheFileName), '.zip', versionParser, semanticVersionParser);
        const versionObj = version_list_1.getVersion(versionList, osHelper(this.osType, this.osArch), version);
        const chromeDriverUrl = this.requestUrl + versionObj.url;
        const chromeDriverZip = path.resolve(this.outDir, versionObj.name);
        // We should check the zip file size if it exists. The size will
        // be used to either make the request, or quit the request if the file
        // size matches.
        let fileSize = 0;
        try {
            fileSize = fs.statSync(chromeDriverZip).size;
        }
        catch (err) {
        }
        await http_utils_1.requestBinary(chromeDriverUrl, {
            fileName: chromeDriverZip,
            fileSize,
            ignoreSSL: this.ignoreSSL,
            proxy: this.proxy
        });
        // Unzip and rename all the files (a grand total of 1) and set the
        // permissions.
        const fileList = file_utils_1.zipFileList(chromeDriverZip);
        const fileItem = path.resolve(this.outDir, fileList[0]);
        file_utils_1.unzipFile(chromeDriverZip, this.outDir);
        const renamedFileName = file_utils_1.renameFileWithVersion(fileItem, '_' + versionObj.version);
        file_utils_1.generateConfigFile(this.outDir, path.resolve(this.outDir, this.configFileName), matchBinaries(this.osType), renamedFileName);
        return Promise.resolve();
    }
    /**
     * Gets the binary file path.
     * @param version Optional to provide the version number or latest.
     */
    getBinaryPath(version) {
        try {
            const configFilePath = path.resolve(this.outDir, this.configFileName);
            return file_utils_1.getBinaryPathFromConfig(configFilePath, version);
        }
        catch (_) {
            return null;
        }
    }
    /**
     * Gets a comma delimited list of versions downloaded. Also has the "latest"
     * downloaded noted.
     */
    getStatus() {
        try {
            const configFilePath = path.resolve(this.outDir, this.configFileName);
            const configJson = JSON.parse(fs.readFileSync(configFilePath).toString());
            const versions = [];
            for (const binaryPath of configJson['all']) {
                let version = '';
                const regex = /.*IEDriverServer_(\d+.\d+.\d+.*).exe/g;
                try {
                    const exec = regex.exec(binaryPath);
                    if (exec && exec[1]) {
                        version = exec[1];
                    }
                }
                catch (_) {
                }
                if (configJson['last'] === binaryPath) {
                    version += ' (latest)';
                }
                versions.push(version);
            }
            return versions.join(', ');
        }
        catch (_) {
            return null;
        }
    }
    /**
     * Get a line delimited list of files removed.
     */
    cleanFiles() {
        return file_utils_1.removeFiles(this.outDir, [/IEDriverServer.*/g, /iedriver.*/g]);
    }
}
exports.IEDriver = IEDriver;
/**
 * Helps translate the os type and arch to the download name associated
 * with composing the download link.
 * @param ostype The operating stystem type.
 * @param osarch The chip architecture.
 * @returns The download name associated with composing the download link.
 */
function osHelper(ostype, osarch) {
    if (ostype === 'Windows_NT') {
        if (osarch === 'x64') {
            return 'Win32';
        }
        else if (osarch === 'x32') {
            return 'Win32';
        }
    }
    return null;
}
exports.osHelper = osHelper;
/**
 * Captures the version name which includes the semantic version and extra
 * metadata. So an example for 12.34/IEDriverServer_win32_12.34.56.zip,
 * the version is 12.34.56.
 * @param xmlKey The xml key including the partial url.
 */
function versionParser(xmlKey) {
    const regex = /.*\/IEDriverServer_[a-zA-Z0-9]*_([0-9]*.[0-9]*.[0-9]*).zip/g;
    try {
        return regex.exec(xmlKey)[1];
    }
    catch (_) {
        return null;
    }
}
exports.versionParser = versionParser;
/**
 * Captures the semantic version name which includes the semantic version and
 * extra metadata. So an example for 12.34/IEDriverServer_win32_12.34.56.zip,
 * the version is 12.34.56.
 * @param xmlKey The xml key including the partial url.
 */
function semanticVersionParser(xmlKey) {
    const regex = /.*\/IEDriverServer_[a-zA-Z0-9]*_([0-9]*.[0-9]*.[0-9]*).zip/g;
    try {
        return regex.exec(xmlKey)[1];
    }
    catch (_) {
        return null;
    }
}
exports.semanticVersionParser = semanticVersionParser;
/**
 * Matches the installed binaries depending on the operating system.
 * @param ostype The operating stystem type.
 */
function matchBinaries(ostype) {
    if (ostype === 'Windows_NT') {
        return /IEDriverServer_\d+.\d+.\d+.exe/g;
    }
    return null;
}
exports.matchBinaries = matchBinaries;
//# sourceMappingURL=iedriver.js.map