"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const fs = require("fs");
const loglevel = require("loglevel");
const path = require("path");
const file_utils_1 = require("./file_utils");
const http_utils_1 = require("./http_utils");
const log = loglevel.getLogger('webdriver-manager');
/**
 * Read the json file from cache. If the cache time has been exceeded or the
 * file does not exist, make an http request and write it to the file.
 * @param jsonUrl The json url.
 * @param httpOptions The http options for the request.
 * @param oauthToken An optional oauth token.
 */
async function updateJson(jsonUrl, httpOptions, oauthToken) {
    if (file_utils_1.isExpired(httpOptions.fileName)) {
        let contents;
        // Create the folder to store the cache.
        const dir = path.dirname(httpOptions.fileName);
        try {
            fs.mkdirSync(dir);
        }
        catch (err) {
        }
        // Check the rate limit and if there is quota for this request.
        if (await hasQuota(oauthToken)) {
            contents = await requestGitHubJson(jsonUrl, httpOptions, oauthToken);
            fs.writeFileSync(httpOptions.fileName, contents);
            return JSON.parse(contents);
        }
        else {
            return null;
        }
    }
    else {
        return file_utils_1.readJson(httpOptions.fileName);
    }
}
exports.updateJson = updateJson;
/**
 * Get the GitHub rate limit with the oauth token.
 * @param oauthToken An optional oauth token.
 * @param requestMethod An overriding requesting method.
 * @returns A promised string of the response body.
 */
function requestRateLimit(oauthToken, requestMethod) {
    const rateLimitUrl = 'https://api.github.com/rate_limit';
    if (requestMethod) {
        return requestMethod(rateLimitUrl, {}, oauthToken);
    }
    else {
        return requestGitHubJson(rateLimitUrl, {}, oauthToken);
    }
}
exports.requestRateLimit = requestRateLimit;
/**
 * Request the GitHub json url and log the curl.
 * @param jsonUrl The json url.
 * @param httpOptions The http options for the request.
 * @param oauthToken An optional oauth token.
 * @returns A promised string of the response body.
 */
function requestGitHubJson(jsonUrl, httpOptions, oauthToken) {
    if (!httpOptions.headers) {
        httpOptions.headers = {};
    }
    httpOptions.headers['User-Agent'] = 'angular/webdriver-manager';
    if (oauthToken) {
        httpOptions.headers['Authorization'] = 'token ' + oauthToken;
    }
    else if (process.env['GITHUB_TOKEN'] || process.env['github_token']) {
        const token = process.env['GITHUB_TOKEN'] || process.env['github_token'];
        httpOptions.headers['Authorization'] = 'token ' + token;
    }
    return http_utils_1.requestBody(jsonUrl, httpOptions);
}
exports.requestGitHubJson = requestGitHubJson;
/**
 * Check quota for remaining GitHub requests.
 * @param oauthToken An optional oauth token.
 * @param requestMethod An overriding requesting method.
 */
async function hasQuota(oauthToken, requestMethod) {
    try {
        const requesteRateLimit = await requestRateLimit(oauthToken, requestMethod);
        if (!requesteRateLimit) {
            throw new Error('Request encountered an error. Received null, expecting json.');
        }
        const rateLimit = JSON.parse(requesteRateLimit);
        if (rateLimit['resources']['core']['remaining'] === 0) {
            if (oauthToken) {
                log.warn('[WARN] No remaining quota for requests to GitHub.');
            }
            else {
                log.warn('[WARN] Provide an oauth token. ' +
                    'See https://github.com/settings/tokens');
            }
            log.warn('[WARN] Stopping updates for gecko driver.');
            return false;
        }
        return true;
    }
    catch (err) {
        log.error('[ERROR]: ', err);
        return false;
    }
}
exports.hasQuota = hasQuota;
/**
 * Returns a list of versions and the partial url paths.
 * @param fileName the location of the xml file to read.
 * @returns the version list from the xml file.
 */
function convertJsonToVersionList(fileName) {
    const githubJson = file_utils_1.readJson(fileName);
    if (!githubJson) {
        return null;
    }
    const versionList = {};
    for (const githubObj of githubJson) {
        const assets = githubObj['assets'];
        const version = githubObj['tag_name'].replace('v', '');
        versionList[version] = {};
        for (const asset of assets) {
            const name = asset['name'];
            const downloadUrl = asset['browser_download_url'];
            const size = asset['size'];
            versionList[version][name] = { name, size, url: downloadUrl, version };
        }
    }
    return versionList;
}
exports.convertJsonToVersionList = convertJsonToVersionList;
//# sourceMappingURL=github_json.js.map