/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2015 Red Hat, Inc., and individual contributors
 * as indicated by the @author tags.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.wildfly.security.mechanism.scram;

import java.util.Arrays;

/**
 * Initial client message for the SCRAM authentication.
 *
 * @author <a href="mailto:david.lloyd@redhat.com">David M. Lloyd</a>
 */
public final class ScramInitialClientMessage {
    private final ScramMechanism mechanism;
    private final String authorizationId;
    private final String authenticationName;
    private final boolean binding;
    private final String bindingType;
    private final byte[] bindingData;
    private final byte[] nonce;
    private final int initialPartIndex;
    private final byte[] messageBytes;

    /**
     * Constructs a new {@code ScramInitialClientMessage} instance using data from the {@code scramClient}.
     *
     * @param scramClient the SCRAM client providing binding type and data, SCRAM mechanism and authorization ID.
     * @param authenticationName the name of the user that is authenticated.
     * @param binding whether the client supports channel binding.
     * @param nonce a unique value generated by the client to the server.
     * @param initialPartIndex index of the initial part of the message.
     * @param messageBytes the byte array of the message.
     */
    ScramInitialClientMessage(final ScramClient scramClient, final String authenticationName, final boolean binding, final byte[] nonce, final int initialPartIndex, final byte[] messageBytes) {
        this.binding = binding;
        this.initialPartIndex = initialPartIndex;
        bindingType = scramClient.getBindingType();
        bindingData = scramClient.getRawBindingData();
        mechanism = scramClient.getMechanism();
        authorizationId = scramClient.getAuthorizationId();
        this.authenticationName = authenticationName;
        this.nonce = nonce;
        this.messageBytes = messageBytes;
    }

    /**
     * Constructs a new {@code ScramInitialClientMessage} instance.
     *
     * @param mechanism the SCRAM mechanism used for the authentication.
     * @param authorizationId the ID of the user to be authorized.
     * @param authenticationName the name of the user that is authenticated.
     * @param binding whether the client supports channel binding.
     * @param bindingType the binding type for the "PLUS" channel binding option.
     * @param bindingData the binding data for the "PLUS" channel binding option.
     * @param nonce a unique value generated by the client to the server.
     * @param initialPartIndex index of the initial part of the message.
     * @param messageBytes the byte array of the message.
     */
    ScramInitialClientMessage(final ScramMechanism mechanism, final String authorizationId, final String authenticationName, final boolean binding, final String bindingType, final byte[] bindingData, final byte[] nonce, final int initialPartIndex, final byte[] messageBytes) {
        this.mechanism = mechanism;
        this.authorizationId = authorizationId;
        this.authenticationName = authenticationName;
        this.binding = binding;
        this.bindingType = bindingType;
        this.bindingData = bindingData;
        this.nonce = nonce;
        this.initialPartIndex = initialPartIndex;
        this.messageBytes = messageBytes;
    }

    /**
     * Returns the SCRAM mechanism used for the authentication.
     *
     * @return the SCRAM mechanism used for the authentication.
     */
    public ScramMechanism getMechanism() {
        return mechanism;
    }

    /**
     * Returns the name of the user that is authenticated.
     *
     * @return the name of the user that is authenticated.
     */
    public String getAuthenticationName() {
        return authenticationName;
    }

    /**
     * Returns a copy of a unique value generated by the client to the server.
     *
     * @return a copy of a unique value generated by the client to the server.
     */
    public byte[] getNonce() {
        return nonce.clone();
    }

    /**
     * Returns a unique value generated by the client to the server.
     *
     * @return a unique value generated by the client to the server.
     */
    byte[] getRawNonce() {
        return nonce;
    }

    /**
     * Returns the initial part of the message.
     *
     * @return the initial part of the message up to the length of {@code initialPartIndex}.
     */
    public byte[] getInitialPart() {
        return Arrays.copyOfRange(messageBytes, 0, initialPartIndex);
    }

    /**
     * Returns a copy of the byte array of the message.
     *
     * @return a copy of the byte array of the message.
     */
    public byte[] getMessageBytes() {
        return messageBytes.clone();
    }

    /**
     * Returns the ID of the user to be authorized.
     *
     * @return the ID of the user to be authorized.
     */
    public String getAuthorizationId() {
        return authorizationId;
    }

    /**
     * Returns whether the client supports channel binding.
     *
     * @return {@code true} if the client supports channel binding, {@code false} otherwise.
     */
    public boolean isBinding() {
        return binding;
    }

    /**
     * Returns the binding type for the "PLUS" channel binding option.
     *
     * @return the binding type for the "PLUS" channel binding option.
     */
    public String getBindingType() {
        return bindingType;
    }

    /**
     * Returns a copy of the binding data for the "PLUS" channel binding option.
     *
     * @return a copy of the binding data for the "PLUS" channel binding option.
     */
    public byte[] getBindingData() {
        return bindingData == null ? null : bindingData.clone();
    }

    /**
     * Returns the binding data for the "PLUS" channel binding option.
     *
     * @return the binding data for the "PLUS" channel binding option.
     */
    byte[] getRawBindingData() {
        return bindingData;
    }

    /**
     * Returns index of the initial part of the message.
     *
     * @return index of the initial part of the message.
     */
    int getInitialPartIndex() {
        return initialPartIndex;
    }

    /**
     * Returns the byte array of the message.
     *
     * @return the byte array of the message.
     */
    byte[] getRawMessageBytes() {
        return messageBytes;
    }
}
