/*
 * Decompiled with CFR 0.152.
 */
package org.infinispan.util.concurrent.locks.impl;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReferenceFieldUpdater;
import org.infinispan.commons.util.Util;
import org.infinispan.configuration.cache.Configuration;
import org.infinispan.context.InvocationContext;
import org.infinispan.factories.annotations.ComponentName;
import org.infinispan.factories.annotations.Inject;
import org.infinispan.jmx.annotations.DataType;
import org.infinispan.jmx.annotations.MBean;
import org.infinispan.jmx.annotations.ManagedAttribute;
import org.infinispan.util.concurrent.TimeoutException;
import org.infinispan.util.concurrent.locks.DeadlockDetectedException;
import org.infinispan.util.concurrent.locks.ExtendedLockPromise;
import org.infinispan.util.concurrent.locks.KeyAwareLockListener;
import org.infinispan.util.concurrent.locks.KeyAwareLockPromise;
import org.infinispan.util.concurrent.locks.LockListener;
import org.infinispan.util.concurrent.locks.LockManager;
import org.infinispan.util.concurrent.locks.LockPromise;
import org.infinispan.util.concurrent.locks.LockState;
import org.infinispan.util.concurrent.locks.impl.InfinispanLock;
import org.infinispan.util.concurrent.locks.impl.LockContainer;
import org.infinispan.util.logging.Log;
import org.infinispan.util.logging.LogFactory;

@MBean(objectName="LockManager", description="Manager that handles MVCC locks for entries")
public class DefaultLockManager
implements LockManager {
    private static final Log log = LogFactory.getLog(DefaultLockManager.class);
    private static final boolean trace = log.isTraceEnabled();
    private static final AtomicReferenceFieldUpdater<CompositeLockPromise, LockState> UPDATER = AtomicReferenceFieldUpdater.newUpdater(CompositeLockPromise.class, LockState.class, "lockState");
    private LockContainer lockContainer;
    private Configuration configuration;
    private ScheduledExecutorService scheduler;

    @Inject
    public void inject(LockContainer container, Configuration configuration, @ComponentName(value="org.infinispan.executors.timeout") ScheduledExecutorService executorService) {
        this.lockContainer = container;
        this.configuration = configuration;
        this.scheduler = executorService;
    }

    @Override
    public KeyAwareLockPromise lock(Object key, Object lockOwner, long time, TimeUnit unit) {
        InfinispanLock lock;
        Objects.requireNonNull(key, "Key must be non null");
        Objects.requireNonNull(lockOwner, "Lock owner must be non null");
        Objects.requireNonNull(unit, "Time unit must be non null");
        if (trace) {
            log.tracef("Lock key=%s for owner=%s. timeout=%s (%s)", new Object[]{Util.toStr((Object)key), lockOwner, time, unit});
        }
        if (key == lockOwner && (lock = this.lockContainer.getLock(key)) != null && lock.getLockOwner() == key) {
            log.tracef("Not locking key=%s as it is already held by the same lock owner", key);
            return KeyAwareLockPromise.NO_OP;
        }
        ExtendedLockPromise promise = this.lockContainer.acquire(key, lockOwner, time, unit);
        return new KeyAwareExtendedLockPromise(promise, key, unit.toMillis(time)).scheduleLockTimeoutTask(this.scheduler);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public KeyAwareLockPromise lockAll(Collection<?> keys, Object lockOwner, long time, TimeUnit unit) {
        Objects.requireNonNull(keys, "Keys must be non null");
        Objects.requireNonNull(lockOwner, "Lock owner must be non null");
        Objects.requireNonNull(unit, "Time unit must be non null");
        if (keys.isEmpty()) {
            if (trace) {
                log.tracef("Lock all: no keys found for owner=%s", lockOwner);
            }
            return KeyAwareLockPromise.NO_OP;
        }
        if (keys.size() == 1) {
            return this.lock(keys.iterator().next(), lockOwner, time, unit);
        }
        Set<Object> uniqueKeys = this.filterDistinctKeys(keys);
        if (uniqueKeys.size() == 1) {
            return this.lock(uniqueKeys.iterator().next(), lockOwner, time, unit);
        }
        if (trace) {
            log.tracef("Lock all keys=%s for owner=%s. timeout=%s (%s)", new Object[]{Util.toStr(uniqueKeys), lockOwner, time, unit});
        }
        CompositeLockPromise compositeLockPromise = new CompositeLockPromise(uniqueKeys.size());
        DefaultLockManager defaultLockManager = this;
        synchronized (defaultLockManager) {
            for (Object key : uniqueKeys) {
                compositeLockPromise.addLock(new KeyAwareExtendedLockPromise(this.lockContainer.acquire(key, lockOwner, time, unit), key, unit.toMillis(time)));
            }
        }
        compositeLockPromise.markListAsFinal();
        return compositeLockPromise.scheduleLockTimeoutTask(this.scheduler, time, unit);
    }

    private Set<Object> filterDistinctKeys(Collection<?> collection) {
        if (collection instanceof Set) {
            return (Set)collection;
        }
        return new HashSet<Object>(collection);
    }

    @Override
    public void unlock(Object key, Object lockOwner) {
        if (trace) {
            log.tracef("Release lock for key=%s. owner=%s", key, lockOwner);
        }
        this.lockContainer.release(key, lockOwner);
    }

    @Override
    public void unlockAll(Collection<?> keys, Object lockOwner) {
        if (trace) {
            log.tracef("Release locks for keys=%s. owner=%s", Util.toStr(keys), lockOwner);
        }
        if (keys.isEmpty()) {
            return;
        }
        for (Object key : keys) {
            if (key == lockOwner) {
                log.tracef("Ignoring key %s as it matches lock owner", key);
                continue;
            }
            this.lockContainer.release(key, lockOwner);
        }
    }

    @Override
    public void unlockAll(InvocationContext context) {
        this.unlockAll(context.getLockedKeys(), context.getLockOwner());
        context.clearLockedKeys();
    }

    @Override
    public boolean ownsLock(Object key, Object lockOwner) {
        Object currentOwner = this.getOwner(key);
        return currentOwner != null && currentOwner.equals(lockOwner);
    }

    @Override
    public boolean isLocked(Object key) {
        return this.getOwner(key) != null;
    }

    @Override
    public Object getOwner(Object key) {
        InfinispanLock lock = this.lockContainer.getLock(key);
        return lock == null ? null : lock.getLockOwner();
    }

    @Override
    public String printLockInfo() {
        return this.lockContainer.toString();
    }

    @Override
    @ManagedAttribute(description="The number of exclusive locks that are held.", displayName="Number of locks held")
    public int getNumberOfLocksHeld() {
        return this.lockContainer.getNumLocksHeld();
    }

    @ManagedAttribute(description="The concurrency level that the MVCC Lock Manager has been configured with.", displayName="Concurrency level", dataType=DataType.TRAIT)
    public int getConcurrencyLevel() {
        return this.configuration.locking().concurrencyLevel();
    }

    @ManagedAttribute(description="The number of exclusive locks that are available.", displayName="Number of locks available")
    public int getNumberOfLocksAvailable() {
        return this.lockContainer.size() - this.lockContainer.getNumLocksHeld();
    }

    @Override
    public InfinispanLock getLock(Object key) {
        return this.lockContainer.getLock(key);
    }

    private static class CompositeLockPromise
    implements KeyAwareLockPromise,
    LockListener,
    Callable<Void> {
        private final List<KeyAwareExtendedLockPromise> lockPromiseList;
        private final CompletableFuture<LockState> notifier;
        volatile LockState lockState = LockState.ACQUIRED;
        private final AtomicInteger countersLeft = new AtomicInteger();

        private CompositeLockPromise(int size) {
            this.lockPromiseList = new ArrayList<KeyAwareExtendedLockPromise>(size);
            this.notifier = new CompletableFuture();
        }

        void addLock(KeyAwareExtendedLockPromise lockPromise) {
            this.lockPromiseList.add(lockPromise);
        }

        void markListAsFinal() {
            this.countersLeft.set(this.lockPromiseList.size());
            for (LockPromise lockPromise : this.lockPromiseList) {
                lockPromise.addListener(this);
            }
        }

        @Override
        public boolean isAvailable() {
            return this.notifier.isDone();
        }

        @Override
        public void lock() throws InterruptedException, TimeoutException {
            InterruptedException interruptedException = null;
            TimeoutException timeoutException = null;
            DeadlockDetectedException deadlockException = null;
            RuntimeException runtimeException = null;
            for (ExtendedLockPromise extendedLockPromise : this.lockPromiseList) {
                try {
                    extendedLockPromise.lock();
                }
                catch (InterruptedException e) {
                    interruptedException = e;
                }
                catch (TimeoutException e) {
                    timeoutException = e;
                }
                catch (DeadlockDetectedException e) {
                    deadlockException = e;
                }
                catch (RuntimeException e) {
                    runtimeException = e;
                }
            }
            if (interruptedException != null) {
                throw interruptedException;
            }
            if (timeoutException != null) {
                throw timeoutException;
            }
            if (deadlockException != null) {
                throw deadlockException;
            }
            if (runtimeException != null) {
                throw runtimeException;
            }
        }

        @Override
        public void addListener(LockListener listener) {
            this.notifier.thenAccept(listener::onEvent);
        }

        @Override
        public void onEvent(LockState state) {
            if (this.notifier.isDone()) {
                return;
            }
            if (state != LockState.ACQUIRED) {
                this.cancelAll(state);
                return;
            }
            if (this.countersLeft.decrementAndGet() == 0) {
                this.notifier.complete(this.lockState);
            }
        }

        private void cancelAll(LockState state) {
            if (UPDATER.compareAndSet(this, LockState.ACQUIRED, state)) {
                this.notifier.complete(state);
                this.lockPromiseList.forEach(promise -> promise.cancel(state));
            }
        }

        @Override
        public void addListener(KeyAwareLockListener listener) {
            for (KeyAwareExtendedLockPromise lockPromise : this.lockPromiseList) {
                lockPromise.addListener(listener);
            }
        }

        @Override
        public Void call() throws Exception {
            this.lockPromiseList.forEach(promise -> promise.cancel(LockState.TIMED_OUT));
            return null;
        }

        CompositeLockPromise scheduleLockTimeoutTask(ScheduledExecutorService executorService, long time, TimeUnit unit) {
            if (executorService != null && time > 0L && !this.isAvailable()) {
                ScheduledFuture<Void> future = executorService.schedule(this, time, unit);
                this.addListener((LockState state) -> future.cancel(false));
            }
            return this;
        }
    }

    private static class KeyAwareExtendedLockPromise
    implements KeyAwareLockPromise,
    ExtendedLockPromise,
    Callable<Void> {
        private final ExtendedLockPromise lockPromise;
        private final Object key;
        private final long timeoutMillis;

        private KeyAwareExtendedLockPromise(ExtendedLockPromise lockPromise, Object key, long timeoutMillis) {
            this.lockPromise = lockPromise;
            this.key = key;
            this.timeoutMillis = timeoutMillis;
        }

        @Override
        public void cancel(LockState cause) {
            this.lockPromise.cancel(cause);
        }

        @Override
        public Object getRequestor() {
            return this.lockPromise.getRequestor();
        }

        @Override
        public Object getOwner() {
            return this.lockPromise.getOwner();
        }

        @Override
        public boolean isAvailable() {
            return this.lockPromise.isAvailable();
        }

        @Override
        public void lock() throws InterruptedException, TimeoutException {
            try {
                this.lockPromise.lock();
            }
            catch (TimeoutException e) {
                throw log.unableToAcquireLock(Util.prettyPrintTime((long)this.timeoutMillis), Util.toStr((Object)this.key), this.lockPromise.getRequestor(), this.lockPromise.getOwner());
            }
        }

        @Override
        public void addListener(LockListener listener) {
            this.lockPromise.addListener(listener);
        }

        @Override
        public void addListener(KeyAwareLockListener listener) {
            this.lockPromise.addListener((LockState state) -> listener.onEvent(this.key, state));
        }

        @Override
        public Void call() throws Exception {
            this.lockPromise.cancel(LockState.TIMED_OUT);
            return null;
        }

        KeyAwareExtendedLockPromise scheduleLockTimeoutTask(ScheduledExecutorService executorService) {
            if (executorService != null && this.timeoutMillis > 0L && !this.isAvailable()) {
                ScheduledFuture<Void> future = executorService.schedule(this, this.timeoutMillis, TimeUnit.MILLISECONDS);
                this.lockPromise.addListener((LockState state) -> future.cancel(false));
            }
            return this;
        }
    }
}

