/*
 * XML 2 Java Binding (X2JB) - the excellent Java tool.
 * Copyright 2009, by Richard Opalka.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not see the FSF site:
 * http://www.fsf.org/ and search for the LGPL License document there.
 */
package com.x2jb.bind.handler;

import org.w3c.dom.Attr;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.Text;

import org.x2jb.bind.BindingException;
import org.x2jb.bind.spi.handler.AttributeHandler;
import org.x2jb.bind.spi.handler.ElementHandler;

/**
 * Template handler extended by other numeric handlers in this package.
 *
 * @author <a href="mailto:richard_opalka@yahoo.com">Richard Opalka</a>
 */
abstract class AbstractHandler implements ElementHandler, AttributeHandler
{

    /**
     * Incorrect number value message. 
     */
    private static final String INCORRECT_VALUE_MESSAGE = "Incorrect value: ";

    /**
     * Binds attribute value to target class.
     *
     * @param a attribute which value will be bound
     * @param clazz target class
     * @return target class instance
     */
    public Object bind( final Attr a, final Class< ? > clazz )
    {
        final String value = a.getValue();
        try
        {
            return this.create( value );
        }
        catch ( NumberFormatException nfe )
        {
            throw new BindingException( AbstractHandler.INCORRECT_VALUE_MESSAGE + value, nfe );
        }
    }

    /**
     * Binds element text to target class.
     *
     * @param e element which text value will be bound
     * @param clazz target class
     * @return target class instance
     */
    public Object bind( final Element e, final Class< ? > clazz )
    {
        final Node childNode = e.getFirstChild();
        if ( ( childNode == null ) || ( !( childNode instanceof Text ) ) )
        {
            throw new BindingException( "Text node expected" );
        }
        else
        {
            final String value = ( ( Text ) childNode ).getData();
            try
            {
                return this.create( value );
            }
            catch ( NumberFormatException nfe )
            {
                throw new BindingException( AbstractHandler.INCORRECT_VALUE_MESSAGE + value, nfe );
            }
        }
    }

    /**
     * Returns default value associated with target class.
     * This template method returns null (can be overwritten in subclasses).
     *
     * @param defaultValue default value
     * @param clazz target class
     * @return default value associated with target class
     */
    public Object getDefault( final String defaultValue, final Class< ? > clazz )
    {
        if ( defaultValue == null )
        {
            return null;
        }
        else
        {
            return this.create( defaultValue );
        }
    }

    /**
     * Constructs return value from passed string.
     * Each subclass have to implement this method.
     *
     * @param value string value representation
     * @return instance of passed value
     */
    protected abstract Object create( String value );

}
