/*
 * ============================================================================
 * (C) Copyright Schalk W. Cronje 2016 - 2023
 *
 * This software is licensed under the Apache License 2.0
 * See http://www.apache.org/licenses/LICENSE-2.0 for license details
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 * ============================================================================
 */
package org.ysb33r.grolifant.api.v4.downloader

import groovy.transform.CompileDynamic
import groovy.transform.CompileStatic
import groovy.util.logging.Slf4j
import org.gradle.wrapper.PathAssembler
import org.gradle.wrapper.WrapperConfiguration
import org.ysb33r.grolifant.api.core.CheckSumVerification
import org.ysb33r.grolifant.api.core.ExclusiveFileAccess
import org.ysb33r.grolifant.api.errors.DistributionFailedException
import org.ysb33r.grolifant.internal.v4.downloader.Downloader

import java.util.concurrent.Callable

import static org.ysb33r.grolifant.api.core.LegacyLevel.PRE_7_1

/**
 * Performs low-level downloading work.
 *
 * @author Schalk W. Cronjé
 *
 * @deprecated Use {@link org.ysb33r.grolifant.api.core.downloader.ArtifactDownloader}
 *
 * @since 1.1
 */
@CompileStatic
@Slf4j
@Deprecated
class ArtifactDownloader {

    /** Creates an instance which takes care of the actual downloading and caching.
     *
     * @param downloadURI URI to download package from.
     * @param downloadRoot Base directory where to download to.
     * @param projectDir The directory of the project that requested the download.
     * @param basePath Relative path to the downloadRoot.
     * @param verifyArtifactRoot Callback to verify the unpacked artifact. Never {@code null}.
     * @param verifyDownloadChecksum Callback to verify the checksum of the downloaded target.
     *   Can be {@code null}.
     */
    @SuppressWarnings('ParameterCount')
    ArtifactDownloader(
        final URI downloadURI,
        final File downloadRoot,
        final File projectDir,
        final String basePath,
        final org.ysb33r.grolifant.api.core.downloader.ArtifactRootVerification verifyArtifactRoot,
        final org.ysb33r.grolifant.api.core.downloader.ArtifactUnpacker unpacker,
        final CheckSumVerification verifyDownloadChecksum
    ) {
        this.downloadURI = downloadURI
        this.downloadRoot = downloadRoot
        this.basePath = basePath
        this.verifyDownloadChecksum = verifyDownloadChecksum
        this.verifyArtifactRoot = verifyArtifactRoot
        this.unpacker = unpacker
        this.requiresDownload = DOWNLOAD_IF_NOT_EXISTS
        this.projectDir = projectDir
    }

    /** Download an artifact without unpacking it.
     *
     * @param downloadURI URI to download package from.
     * @param downloadRoot Base directory where to download to.
     * @param projectDir The directory of the project that requested the download.
     * @param basePath Relative path to the downloadRoot.
     * @param requiresDownload Indicates whether download is required.
     * @param verifyDownloadChecksum Callback to verify the checksum of the downloaded target.
     *   Can be {@code null}.
     */
    @SuppressWarnings('ParameterCount')
    ArtifactDownloader(
        final URI downloadURI,
        final File downloadRoot,
        final File projectDir,
        final String basePath,
        final org.ysb33r.grolifant.api.core.downloader.ArtifactRequiresDownload requiresDownload,
        final CheckSumVerification verifyDownloadChecksum
    ) {
        this.downloadURI = downloadURI
        this.downloadRoot = downloadRoot
        this.basePath = basePath
        this.verifyDownloadChecksum = verifyDownloadChecksum
        this.requiresDownload = requiresDownload
        this.projectDir = projectDir
    }

    /** Creates a distribution/file it it does not exist already.
     *
     * @param description Name of the downloaded entity.
     * @param offlineMode Whether to operate in download mode.
     * @param downloadInstance Download & logger instances to use
     *
     * @return Location of distribution
     */
    File getFromCache(final String description, boolean offlineMode, final Downloader downloadInstance) {
        final WrapperConfiguration configuration = newWrapperConfiguration
        final PathAssembler pathAssembler = getPathAssembler(downloadRoot)
        final PathAssembler.LocalDistribution localDistribution = pathAssembler.getDistribution(configuration)
        final File distDir = localDistribution.distributionDir

        // This is not always compressed or a Zipfile. We still have to use the method as the original
        // Wrapper Configuration assumes that only ZIP files are downloaded.
        final File localDownloadedFile = localDistribution.zipFile

        Callable<File> downloadAction = new DownloadAction(
            offlineMode: offlineMode,
            description: description,
            distDir: distDir,
            localDownloadedFile: localDownloadedFile,
            markerFile: new File(localDownloadedFile.parentFile, localDownloadedFile.name + '.ok'),
            downloadURI: this.downloadURI,
            distributionUrl: configuration.distribution,
            verifyArtifactRoot: this.verifyArtifactRoot,
            requiresDownload: this.requiresDownload,
            downloadInstance: downloadInstance,
            verifyDownloadChecksum: this.verifyDownloadChecksum,
            unpacker: this.unpacker
        )
        exclusiveFileAccessManager.access(localDownloadedFile, downloadAction)
    }

    private WrapperConfiguration getNewWrapperConfiguration() {
        final WrapperConfiguration configuration = new WrapperConfiguration()
        configuration.distribution = this.downloadURI
        configuration.distributionPath = configuration.zipPath = basePath

        configChecksum = configuration
    }

//    @CompileDynamic
    private WrapperConfiguration setConfigChecksum(WrapperConfiguration configuration) {
        if (verifyDownloadChecksum) {
            configuration.distributionSha256Sum = verifyDownloadChecksum.checksum
        }

        configuration
    }

    @CompileDynamic
    private PathAssembler getPathAssembler(File resolvedDownloadRoot) {
        if (PRE_7_1) {
            new PathAssembler(resolvedDownloadRoot)
        } else {
            new PathAssembler(resolvedDownloadRoot, projectDir)
        }
    }

    static private final org.ysb33r.grolifant.api.core.downloader.ArtifactRequiresDownload DOWNLOAD_IF_NOT_EXISTS =
        { URI d, File f -> !f.file } as org.ysb33r.grolifant.api.core.downloader.ArtifactRequiresDownload

    private final ExclusiveFileAccess exclusiveFileAccessManager = new ExclusiveFileAccess(120000, 200)
    private final URI downloadURI
    private final File downloadRoot
    private final File projectDir
    private final String basePath
    private final CheckSumVerification verifyDownloadChecksum
    private final org.ysb33r.grolifant.api.core.downloader.ArtifactUnpacker unpacker
    private final org.ysb33r.grolifant.api.core.downloader.ArtifactRootVerification verifyArtifactRoot
    private final org.ysb33r.grolifant.api.core.downloader.ArtifactRequiresDownload requiresDownload

    private static class DownloadAction implements Callable<File> {

        boolean offlineMode
        String description
        File distDir
        File markerFile
        File localDownloadedFile
        URI downloadURI
        URI distributionUrl
        org.ysb33r.grolifant.api.core.downloader.ArtifactRootVerification verifyArtifactRoot
        org.ysb33r.grolifant.api.core.downloader.ArtifactRequiresDownload requiresDownload
        Downloader downloadInstance
        CheckSumVerification verifyDownloadChecksum
        org.ysb33r.grolifant.api.core.downloader.ArtifactUnpacker unpacker

        @Override
        File call() throws Exception {
            if (distDir.directory && markerFile.file) {
                return verifyArtifactRoot.apply(distDir)
            }

            if (requiresDownload.download(downloadURI, localDownloadedFile)) {
                if (offlineMode && distributionUrl.scheme != 'file') {
                    throw new DistributionFailedException("Cannot download ${description} as currently offline")
                }

                File tmpDownloadedFile = new File(
                    localDownloadedFile.parentFile, "${localDownloadedFile.name}.part"
                )
                tmpDownloadedFile.delete()
                downloadInstance.downloader.download(distributionUrl, tmpDownloadedFile)
                tmpDownloadedFile.renameTo(localDownloadedFile)
            }

            List<File> topLevelDirs = distDir.listFiles(new FileFilter() {
                @Override
                boolean accept(File pathname) {
                    pathname.directory
                }
            }) as List<File>

            for (File dir : topLevelDirs) {
                downloadInstance.progressLogger.log("Deleting directory ${dir.absolutePath}")
                dir.deleteDir()
            }

            if (verifyDownloadChecksum) {
                verifyDownloadChecksum.verify(localDownloadedFile)
            }

            if (unpacker) {
                downloadInstance.progressLogger.log(
                    "Unpacking ${localDownloadedFile.absolutePath} to ${distDir.absolutePath}"
                )
                unpacker.accept(localDownloadedFile, distDir)
            }

            File root = verifyArtifactRoot ? verifyArtifactRoot.apply(distDir) : distDir
            markerFile.createNewFile()

            root
        }
    }
}
