/*
 * ============================================================================
 * (C) Copyright Schalk W. Cronje 2016 - 2023
 *
 * This software is licensed under the Apache License 2.0
 * See http://www.apache.org/licenses/LICENSE-2.0 for license details
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 * ============================================================================
 */
package org.ysb33r.grolifant.internal.v5.executable

import groovy.transform.CompileStatic
import org.gradle.api.provider.Provider
import org.gradle.process.ProcessForkOptions
import org.ysb33r.grolifant.api.core.ProjectOperations
import org.ysb33r.grolifant.api.core.ProviderToString
import org.ysb33r.grolifant.api.core.executable.ExecutableEntryPoint
import org.ysb33r.grolifant.internal.core.runnable.EnvironmentVariableProviders

import static org.ysb33r.grolifant.api.core.LegacyLevel.PRE_6_6

/**
 * Implementation of an {@link ExecutableEntryPoint} for Gradle 5.0 - 6.9.
 *
 * @author Schalk W. Cronjé
 *
 * @since 2.0
 */
@CompileStatic
class InternalEntryPointProxy implements ExecutableEntryPoint {

    InternalEntryPointProxy(
        ProjectOperations po,
        ProcessForkOptions instance,
        EnvironmentVariableProviders evp
    ) {
        this.delegate = instance
        this.evp = evp
        this.projectOperations = po
    }

    /**
     * Sets the name of the executable to use.
     *
     * @param exec The executable. Must not be null.
     */
    @Override
    void setExecutable(Object exec) {
        if (PRE_6_6) {
            delegate.executable(exec)
        } else {
            delegate.executable(ProviderToString.proxy(
                projectOperations.provider { -> projectOperations.stringTools.stringize(exec) }
            ))
        }
    }

    /**
     * Sets the name of the executable to use.
     *
     * @param exec The executable. Must not be null.
     * @return this
     */
    @Override
    ProcessForkOptions executable(Object exec) {
        if (PRE_6_6) {
            delegate.executable(exec)
        } else {
            delegate.executable(ProviderToString.proxy(
                projectOperations.provider { -> projectOperations.stringTools.stringize(exec) }
            ))
        }
        delegate
    }

    /**
     * Sets the working directory for the process. The supplied argument is evaluated as per {@link
     * org.gradle.api.Project#file(Object)}.
     *
     * @param dir The working directory. Must not be null.
     */
    @Override
    void setWorkingDir(Object dir) {
        if (PRE_6_6) {
            delegate.workingDir(dir)
        } else {
            delegate.workingDir(projectOperations.provider { -> projectOperations.fsOperations.file(dir) })
        }
    }

    /**
     * Sets the working directory for the process. The supplied argument is evaluated as per {@link
     * org.gradle.api.Project#file(Object)}.
     *
     * @param dir The working directory. Must not be null.
     * @return this
     */
    @Override
    ProcessForkOptions workingDir(Object dir) {
        if (PRE_6_6) {
            delegate.workingDir(dir)
        } else {
            delegate.workingDir(projectOperations.provider { -> projectOperations.fsOperations.file(dir) })
        }
        delegate
    }

    /**
     * Adds a provider to environment variables.
     *
     * The values of the provider are processed after any value set via an {@code environment} call.
     *
     * @param envProvider Provider to a resolved map.
     */
    @Override
    void addEnvironmentProvider(Provider<Map<String, String>> envProvider) {
        evp.addEnvironmentProvider(envProvider)
    }

    @Delegate(interfaces = true, deprecated = true, methodAnnotations = true, parameterAnnotations = true)
    private final ProcessForkOptions delegate

    private final EnvironmentVariableProviders evp
    private final ProjectOperations projectOperations
}
