/*
 * ============================================================================
 * (C) Copyright Schalk W. Cronje 2016 - 2024
 *
 * This software is licensed under the Apache License 2.0
 * See http://www.apache.org/licenses/LICENSE-2.0 for license details
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 * ============================================================================
 */
package org.ysb33r.grolifant.internal.v5.jvm

import groovy.transform.CompileStatic
import org.gradle.api.model.ObjectFactory
import org.gradle.api.tasks.Internal
import org.gradle.process.CommandLineArgumentProvider
import org.gradle.process.JavaExecSpec
import org.ysb33r.grolifant.api.core.ProjectOperations
import org.ysb33r.grolifant.api.core.jvm.JavaForkOptionsWithEnvProvider
import org.ysb33r.grolifant.api.core.jvm.ModularitySpec
import org.ysb33r.grolifant.internal.core.Transform
import org.ysb33r.grolifant.internal.core.runnable.EnvironmentVariableProviders
import org.ysb33r.grolifant.internal.v4.jvm.FakeModularitySpec
import org.ysb33r.grolifant.internal.v4.jvm.InternalAbstractJvmAppRunnerSpec

import java.util.function.Function

import static org.ysb33r.grolifant.api.core.LegacyLevel.PRE_5_6

/**
 * Provides a class that can be populated with various fork options for Java
 * and which can then be used to copy to other methods in the Gradle API that provides a
 * {@link org.gradle.process.JavaForkOptions} in the parameters.
 *
 * Intended for Gradle 5.x
 *
 * @author Schalk W. Cronjé
 *
 * @since 2.0
 */
@CompileStatic
class InternalJvmAppRunnerSpec extends InternalAbstractJvmAppRunnerSpec {

    InternalJvmAppRunnerSpec(ProjectOperations po, ObjectFactory objects) {
        super(
            po,
            { JavaExecSpec jes, EnvironmentVariableProviders evp -> createJfoProxy(jes, evp, objects) },
            { JavaExecSpec jes -> new FakeModularitySpec(po) }
        )
    }

    InternalJvmAppRunnerSpec(
        ProjectOperations po,
        ObjectFactory objects,
        Function<JavaExecSpec, ModularitySpec> modularitySpecFactory
    ) {
        super(
            po,
            { JavaExecSpec jes, EnvironmentVariableProviders evp -> createJfoProxy(jes, evp, objects) },
            modularitySpecFactory
        )
    }

    /**
     * Copies command arguments (non-JVM) target {@link JavaExecSpec}.
     *
     * @param target Target {@link JavaExecSpec}.
     */
    @Override
    void copyArguments(JavaExecSpec target) {
        target.args = arguments.args
        def cmdlineProviders = Transform.toList(arguments.commandLineArgumentProviders) { p ->
            new CommandLineArgumentProvider() {
                @Override
                Iterable<String> asArguments() {
                    p.get()
                }
            }
        }
        target.argumentProviders.addAll(cmdlineProviders)
        commandLineProcessors.each {
            final providedArgs = it.argumentSpec.commandLineArgumentProviders

            if (it.order) {
                final allArgs = it.argumentSpec.allArgs
                target.argumentProviders.add({ -> allArgs.get() } as CommandLineArgumentProvider)
            } else {
                target.args(it.argumentSpec.args)

                providedArgs.each { p ->
                    target.argumentProviders.add({ -> p.get() } as CommandLineArgumentProvider)
                }
            }
        }
    }

    /**
     * Copies debug options to target {@link JavaExecSpec}.
     *
     * @param targetTarget {@link JavaExecSpec}.
     */
    @Override
    void copyDebugOptions(JavaExecSpec target) {
        if (!PRE_5_6) {
            def dopt = javaExecSpec.debugOptions
            target.debugOptions.identity {
                enabled.set(dopt.enabled)
                server.set(dopt.server)
                suspend.set(dopt.suspend)
                port.set(dopt.port)
            }
        }
    }

    /**
     * A unique string which determines whether there were any changes.
     *
     * @return Signature string
     */
    @Override
    String getExecutionSignature() {
        executionParameters.toString().sha256()
    }

    /**
     * Builds a list of arguments by taking all the set arguments as well as the argument providers.
     *
     * THe main purpose of this method is to provide a list of arguments to be passed to a Worker instance.
     *
     * @return All application arguments
     */
    @Override
    @SuppressWarnings('UnnecessaryCast')
    protected List<String> buildArguments() {
        final List<String> result = []
        final tmpSpec = projectOperations.jvmTools.javaExecSpec()
        copyArguments(tmpSpec)
        result.addAll(tmpSpec.args)
        result.addAll(tmpSpec.argumentProviders*.asArguments().flatten().toList() as List<String>)
        result
    }

    /**
     * Loads executions parameters from the current execution specification.
     * <p>
     *     The primary purpose of this method is to build a map for use by {@link #getExecutionSignature}.
     *     The default implementation will use the executable path and the arguments.
     *     </p>
     * @return Execution parameters.
     */
    @Internal
    @SuppressWarnings('UnnecessaryCast')
    protected Map<String, ?> getExecutionParameters() {
        final st = projectOperations.stringTools
        final ep = super.executionParameters
        ep.putAll([
            args2: st.stringizeDropNull(javaExecSpec.jvmArgumentProviders*.asArguments().flatten()),
            args4: st.stringizeDropNull(javaExecSpec.argumentProviders*.asArguments().flatten()),
        ] as Map<String, ?>)
        ep
    }

    private static JavaForkOptionsWithEnvProvider createJfoProxy(
        JavaExecSpec es,
        EnvironmentVariableProviders evp,
        ObjectFactory objects
    ) {
        if (PRE_5_6) {
            new org.ysb33r.grolifant.internal.v4.jvm.JavaForkOptionsWithEnvProviderProxy(es, evp)
        } else {
            new JavaForkOptionsWithEnvProviderProxy(es, evp, objects)
        }
    }
}
