/*
 * ============================================================================
 * (C) Copyright Schalk W. Cronje 2016 - 2024
 *
 * This software is licensed under the Apache License 2.0
 * See http://www.apache.org/licenses/LICENSE-2.0 for license details
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 * ============================================================================
 */
package org.ysb33r.grolifant.loadable.v7

import groovy.transform.CompileStatic
import org.gradle.api.Action
import org.gradle.api.Project
import org.gradle.api.model.ObjectFactory
import org.gradle.process.ExecSpec
import org.gradle.process.internal.DefaultExecSpec
import org.ysb33r.grolifant.api.core.ExecTools
import org.ysb33r.grolifant.api.core.ProjectOperations
import org.ysb33r.grolifant.api.core.downloader.Downloader
import org.ysb33r.grolifant.api.core.executable.AppRunnerSpec
import org.ysb33r.grolifant.api.core.executable.CommandEntryPoint
import org.ysb33r.grolifant.internal.core.executable.CommandArgumentSpec
import org.ysb33r.grolifant.internal.core.executable.ExecUtils
import org.ysb33r.grolifant.internal.v7.downloader.InternalDownloader
import org.ysb33r.grolifant.internal.v7.executable.InternalAppRunnerSpec

import java.util.function.Function

/**
 * Non-JVM process execution tools for Gradle 7.x.
 *
 * @author Schalk W. Cronjé
 *
 * @since 2.0.
 */
@CompileStatic
class DefaultExecTools implements ExecTools {

    DefaultExecTools(ProjectOperations po, Project project) {
        objectFactory = project.objects
        projectOperations = po
    }

    /**
     * Returns something that looks like an {@link ExecSpec}.
     *
     * @return Instance of an executable specification.
     */
    @Override
    ExecSpec execSpec() {
        objectFactory.newInstance(DefaultExecSpec)
    }

    /**
     * Creates a {@link AppRunnerSpec}.
     *
     * THis is primarily used internally by classes that implements execution specifications for non-JVM processes.
     *
     * @return Implementation of {@link AppRunnerSpec}
     */
    @Override
    AppRunnerSpec appRunnerSpec() {
        new InternalAppRunnerSpec(projectOperations)
    }

    /**
     * Returns an implementation that is optimised for the running version of Gradle.
     *
     * @return Instance of {@link org.ysb33r.grolifant.api.core.executable.CommandEntryPoint}. Never {@code null}
     */
    @Override
    CommandEntryPoint commandEntryPoint() {
        new CommandArgumentSpec(projectOperations, objectFactory)
    }

    /**
     * Creates a new downloader for downlaoding packages / distributions.
     *
     * @param distributionName Name of distribution to be downloaded.
     *
     * @return Instance of {@link Downloader} that is optimised for the running Gradle version.
     */
    @Override
    Downloader downloader(final String distributionName) {
        new InternalDownloader(distributionName, projectOperations)
    }

    /**
     * Simplifies running an executable to obtain a version.
     * This is primarily used to implement a {@code runExecutableAndReturnVersion} method.
     *
     * @param argsForVersion Arguments required to obtain a version
     * @param executablePath Location of the executable
     * @param versionParser A parser for the output of running the executable which could extract the version.
     * @param configurator Additional configurator to customise the execution specification.
     * @return The version string.
     */
    @Override
    String parseVersionFromOutput(
        Iterable<String> argsForVersion,
        File executablePath,
        Function<String, String> versionParser,
        Action<ExecSpec> configurator
    ) {
        ExecUtils.parseVersionFromOutput(
            projectOperations,
            argsForVersion,
            executablePath,
            versionParser,
            configurator
        )
    }

    private final ObjectFactory objectFactory
    private final ProjectOperations projectOperations
}
