/*
 * Decompiled with CFR 0.152.
 */
package de.zalando.spring.cloud.config.aws.kms;

import com.amazonaws.services.kms.AWSKMS;
import com.amazonaws.services.kms.model.DecryptRequest;
import com.amazonaws.services.kms.model.EncryptRequest;
import de.zalando.spring.cloud.config.aws.kms.EncryptedToken;
import de.zalando.spring.cloud.config.aws.kms.KmsTextEncryptorOptions;
import de.zalando.spring.cloud.config.aws.kms.OutputMode;
import java.nio.ByteBuffer;
import java.util.Base64;
import java.util.Optional;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.security.crypto.encrypt.TextEncryptor;
import org.springframework.util.Assert;

public class KmsTextEncryptor
implements TextEncryptor {
    private static final Base64.Encoder BASE64_ENCODER;
    private static final String EMPTY_STRING = "";
    private static final boolean IS_ALGORITHM_AVAILABLE;
    private final Logger log = LoggerFactory.getLogger(this.getClass());
    private final AWSKMS kms;
    private final String kmsKeyId;
    private final String encryptionAlgorithm;

    public KmsTextEncryptor(AWSKMS kms, String kmsKeyId, String encryptionAlgorithm) {
        Assert.notNull((Object)kms, (String)"KMS client must not be null");
        Assert.notNull((Object)encryptionAlgorithm, (String)"encryptionAlgorithm must not be null");
        this.kms = kms;
        this.kmsKeyId = kmsKeyId;
        this.encryptionAlgorithm = encryptionAlgorithm;
        this.checkAlgorithm(encryptionAlgorithm);
    }

    public String encrypt(String text) {
        Assert.hasText((String)this.kmsKeyId, (String)"kmsKeyId must not be blank");
        if (text == null || text.isEmpty()) {
            return EMPTY_STRING;
        }
        EncryptRequest encryptRequest = new EncryptRequest().withKeyId(this.kmsKeyId).withPlaintext(ByteBuffer.wrap(text.getBytes()));
        this.checkAlgorithm(this.encryptionAlgorithm);
        if (IS_ALGORITHM_AVAILABLE) {
            encryptRequest.setEncryptionAlgorithm(this.encryptionAlgorithm);
        }
        ByteBuffer encryptedBytes = this.kms.encrypt(encryptRequest).getCiphertextBlob();
        return KmsTextEncryptor.extractString(encryptedBytes, OutputMode.BASE64);
    }

    public String decrypt(String encryptedText) {
        if (encryptedText == null || encryptedText.isEmpty()) {
            return EMPTY_STRING;
        }
        EncryptedToken token = EncryptedToken.parse(encryptedText);
        DecryptRequest decryptRequest = new DecryptRequest().withCiphertextBlob(token.getCipherBytes()).withEncryptionContext(token.getEncryptionContext());
        KmsTextEncryptorOptions options = token.getOptions();
        String keyId = Optional.ofNullable(options.getKeyId()).orElse(this.kmsKeyId);
        String algorithm = Optional.ofNullable(options.getEncryptionAlgorithm()).orElse(this.encryptionAlgorithm);
        this.checkAlgorithm(algorithm);
        if (IS_ALGORITHM_AVAILABLE) {
            decryptRequest.setEncryptionAlgorithm(algorithm);
            if (KmsTextEncryptor.isAsymmetricEncryption(algorithm)) {
                Assert.hasText((String)keyId, (String)"kmsKeyId must not be blank. Asymmetric decryption requires the key to be known");
                decryptRequest.setKeyId(keyId);
            }
        }
        return KmsTextEncryptor.extractString(this.kms.decrypt(decryptRequest).getPlaintext(), options.getOutputMode());
    }

    private static String extractString(ByteBuffer bb, OutputMode outputMode) {
        if (bb.hasRemaining()) {
            byte[] bytes = new byte[bb.remaining()];
            bb.get(bytes, bb.arrayOffset(), bb.remaining());
            if (outputMode == OutputMode.BASE64) {
                return BASE64_ENCODER.encodeToString(bytes);
            }
            return new String(bytes);
        }
        return EMPTY_STRING;
    }

    private void checkAlgorithm(String algorithm) {
        if (KmsTextEncryptor.isAsymmetricEncryption(algorithm) && !IS_ALGORITHM_AVAILABLE) {
            this.log.warn("Asymmetric encryption '{}' has been configured,but the version of aws-java-sdk you are using is outdated and does not support it. Please upgrade to a more recent version.", (Object)algorithm);
        }
    }

    private static boolean isAsymmetricEncryption(String algorithm) {
        return !algorithm.equals("SYMMETRIC_DEFAULT");
    }

    static {
        boolean available;
        BASE64_ENCODER = Base64.getEncoder();
        try {
            Class.forName("com.amazonaws.services.kms.model.EncryptionAlgorithmSpec");
            available = true;
        }
        catch (Exception e) {
            available = false;
        }
        IS_ALGORITHM_AVAILABLE = available;
    }
}

