/*
 * Copyright 2007 Gerd Ziegler (www.gerdziegler.de)
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific
 * language governing permissions and limitations under the License.
 * @author www.gerdziegler.de
 */

package org.zclasspath;

import java.lang.annotation.Annotation;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.log4j.Logger;

public class ZClassRepository implements ZIClassRepository
{
  private static Logger log = Logger.getLogger(ZClassRepository.class);

  private final List<Class> classes = new ArrayList<Class>();

  private final Map<String, List<Class>> annotatedBy = new HashMap<String, List<Class>>();


  public ZClassRepository(ZClassPathScanner scanner)
  {
    try
    {

      final List<Exception> errors = new ArrayList<Exception>();
      scanner.setVisitor(new ZIClassPathVisitor()
      {
        public void visitClass(ZClassPathClass classPathClass)
        {
          try
          {
            Class clazz = Class.forName(classPathClass.getClassName());
            visitClass(clazz);
          }
          catch (Exception e)
          {
            errors.add(e);
          }
        }


        private void visitClass(Class clazz) throws Exception
        {
          classes.add(clazz);
          for (Annotation ann : clazz.getAnnotations())
          {
            String key = ann.annotationType().getName();
            List<Class> list = annotatedBy.get(key);
            if (list == null)
            {
              list = new ArrayList<Class>();
              annotatedBy.put(key, list);
            }
            list.add(clazz);
          }
        }


        public void visitResource(ZIClassPathElement element)
        {
        }
      });

      scanner.scan();

      Collections.sort(errors, new Comparator<Exception>()
      {
        public int compare(Exception e1, Exception e2)
        {
          return e1.getMessage().compareTo(e2.getMessage());
        }
      });

      for (Exception e : errors)
      {
        log.error(e.getClass().getName() + " --- " + e.getMessage());
      }
    }
    catch (Exception e)
    {
      log.error("", e);
    }
    finally
    {
      log.debug("scan completed");
    }
  }


  public static ZClassRepository create(Class clazz)
  {
    return create(ZJavaClassPath.getItems(), clazz.getPackage().getName());
  }


  public static ZClassRepository create(String pojoPackage)
  {
    return create(ZJavaClassPath.getItems(), pojoPackage);
  }


  public static ZClassRepository create(List<ZIClassPathItem> items)
  {
    return create(items, null);
  }


  public static ZClassRepository create(List<ZIClassPathItem> items, final String pojoPackage)
  {
    ZClassPathScanner scanner = new ZClassPathScanner();
    scanner.setClassPathItems(items);
    if (pojoPackage != null)
    {
      scanner.setFilter(new ZIClassPathFilter()
      {
        public boolean acceptClass(String name) throws Exception
        {
          return (pojoPackage == null || name.startsWith(pojoPackage)) && !name.startsWith("org.apache.") && !name.startsWith("flex.")
              && !name.startsWith("java.") && !name.startsWith("javax.") && !name.startsWith("org.junit");
        }


        public boolean acceptClasspathPart(String name) throws Exception
        {
          return true;
        }


        public boolean acceptResource(String resource) throws Exception
        {
          return false;
        }
      });
    }
    return new ZClassRepository(scanner);
  }


  public List<Class> getClassesAnnotatedWithAndAssignableFrom(Class<? extends Annotation> annotationClass, Class clazz)
  {
    List<Class> ret = new ArrayList<Class>();

    List<Class> annotated = getClassesAnnotatedWith(annotationClass);
    for (Class c : annotated)
    {
      if (clazz.isAssignableFrom(c))
      {
        ret.add(c);
      }
    }
    return ret;
  }


  public List<Class> getClassesAnnotatedWithAndAssignableTo(Class<? extends Annotation> annotationClass, Class clazz)
  {
    List<Class> ret = new ArrayList<Class>();

    List<Class> annotated = getClassesAnnotatedWith(annotationClass);
    for (Class c : annotated)
    {
      if (c.isAssignableFrom(clazz))
      {
        ret.add(c);
      }
    }
    return ret;
  }


  public List<Class> getClassesAnnotatedWith(Class<? extends Annotation> annotationClass)
  {
    String key = annotationClass.getName();
    List<Class> ret = annotatedBy.get(key);
    return ret != null ? ret : new ArrayList<Class>();
  }


  public List<Class> getClassesAssignableTo(Class clazz)
  {
    List<Class> ret = new ArrayList<Class>();

    for (Class c : classes)
    {
      if (clazz.isAssignableFrom(c))
      {
        ret.add(c);
      }
    }
    return ret;
  }


  public List<Class> getClassesAssignableFrom(Class clazz)
  {
    List<Class> ret = new ArrayList<Class>();

    for (Class c : classes)
    {
      if (c.isAssignableFrom(clazz))
      {
        ret.add(c);
      }
    }
    return ret;
  }


  public Class getDeclaringClass(Class pojoClass, Class<? extends Annotation> findAnnot)
  {
    if (pojoClass == null)
    {
      return null;
    }
    Annotation[] annot = pojoClass.getDeclaredAnnotations();
    for (Annotation crtAnnot : annot)
    {
      if (crtAnnot.annotationType().equals(findAnnot))
      {
        return pojoClass;
      }
    }
    return getDeclaringClass(pojoClass.getSuperclass(), findAnnot);
  }


  public List<Class> getClasses()
  {
    return classes;
  }
}
