/*
 * Copyright 2015 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package pl.touk.tscreload;

import io.vavr.*;
import lombok.extern.slf4j.Slf4j;
import pl.touk.tscreload.impl.*;

import java.util.Optional;

import static pl.touk.tscreload.TransformationResult.withPropagateChangeWhenValueChanged;

@Slf4j
public abstract class Reloadable<T> extends Observable<T> {

    private volatile T current;

    protected Reloadable(T current) {
        this.current = current;
    }

    protected synchronized void updateCurrentValue(Function1<Optional<T>, TransformationResult<T>> transform) {
        TransformationResult<T> transformationResult = transform.apply(Optional.of(current));
        if (log.isTraceEnabled()) {
            log.trace("{} Updating current value. Change {} be propagated.", this, (transformationResult.isPropagateChange() ? "will" : "won't"));
        }
        current = transformationResult.getValue();
        if (transformationResult.isPropagateChange())
            notifyObservers(transformationResult.getValue());
    }

    public <U> Reloadable<U> map(Function1<T, U> f) {
        return map((t, prev) -> withPropagateChangeWhenValueChanged(prev, f.apply(t)));
    }

    public <U> Reloadable<U> map(Function2<T, Optional<U>, TransformationResult<U>> f) {
        Reloadable1<T, U> child = new Reloadable1<>(currentValue(), f);
        addWeakObserver(child);
        return child;
    }

    public T currentValue() {
        return current;
    }

    public static <R1, R2, U> Reloadable<U> compose(Reloadable<R1> r1,
                                                    Reloadable<R2> r2,
                                                    Function2<R1, R2, U> f) {
        return compose(r1, r2, (p1, p2, prev) -> withPropagateChangeWhenValueChanged(prev, f.apply(p1, p2)));
    }

    public static <R1, R2, U> Reloadable<U> compose(Reloadable<R1> r1,
                                                    Reloadable<R2> r2,
                                                    Function3<R1, R2, Optional<U>, TransformationResult<U>> f) {
        Reloadable2<R1, R2, U> reloadable = new Reloadable2<>(
                r1.currentValue(),
                r2.currentValue(),
                f);
        r1.addWeakObserver(reloadable.observer1);
        r2.addWeakObserver(reloadable.observer2);
        return reloadable;
    }

    public static <R1, R2, R3, U> Reloadable<U> compose(Reloadable<R1> r1,
                                                        Reloadable<R2> r2,
                                                        Reloadable<R3> r3,
                                                        Function3<R1, R2, R3, U> f) {
        return compose(r1, r2, r3,
                (p1, p2, p3, prev) -> withPropagateChangeWhenValueChanged(prev, f.apply(p1, p2, p3)));
    }

    public static <R1, R2, R3, U> Reloadable<U> compose(Reloadable<R1> r1,
                                                        Reloadable<R2> r2,
                                                        Reloadable<R3> r3,
                                                        Function4<R1, R2, R3, Optional<U>, TransformationResult<U>> f) {
        Reloadable3<R1, R2, R3, U> reloadable = new Reloadable3<>(
                r1.currentValue(),
                r2.currentValue(),
                r3.currentValue(),
                f);
        r1.addWeakObserver(reloadable.observer1);
        r2.addWeakObserver(reloadable.observer2);
        r3.addWeakObserver(reloadable.observer3);
        return reloadable;
    }


    public static <R1, R2, R3, R4, U> Reloadable<U> compose(Reloadable<R1> r1,
                                                            Reloadable<R2> r2,
                                                            Reloadable<R3> r3,
                                                            Reloadable<R4> r4,
                                                            Function4<R1, R2, R3, R4, U> f) {
        return compose(r1, r2, r3, r4,
                (p1, p2, p3, p4, prev) -> withPropagateChangeWhenValueChanged(prev, f.apply(p1, p2, p3, p4)));
    }

    public static <R1, R2, R3, R4, U> Reloadable<U> compose(Reloadable<R1> r1,
                                                            Reloadable<R2> r2,
                                                            Reloadable<R3> r3,
                                                            Reloadable<R4> r4,
                                                            Function5<R1, R2, R3, R4, Optional<U>, TransformationResult<U>> f) {
        Reloadable4<R1, R2, R3, R4, U> reloadable = new Reloadable4<>(
                r1.currentValue(),
                r2.currentValue(),
                r3.currentValue(),
                r4.currentValue(),
                f);
        r1.addWeakObserver(reloadable.observer1);
        r2.addWeakObserver(reloadable.observer2);
        r3.addWeakObserver(reloadable.observer3);
        r4.addWeakObserver(reloadable.observer4);
        return reloadable;
    }

    public static <R1, R2, R3, R4, R5, U> Reloadable<U> compose(Reloadable<R1> r1,
                                                                Reloadable<R2> r2,
                                                                Reloadable<R3> r3,
                                                                Reloadable<R4> r4,
                                                                Reloadable<R5> r5,
                                                                Function5<R1, R2, R3, R4, R5, U> f) {
        return compose(r1, r2, r3, r4, r5,
                (p1, p2, p3, p4, p5, prev) -> withPropagateChangeWhenValueChanged(prev, f.apply(p1, p2, p3, p4, p5)));
    }

    public static <R1, R2, R3, R4, R5, U> Reloadable<U> compose(Reloadable<R1> r1,
                                                                Reloadable<R2> r2,
                                                                Reloadable<R3> r3,
                                                                Reloadable<R4> r4,
                                                                Reloadable<R5> r5,
                                                                Function6<R1, R2, R3, R4, R5, Optional<U>, TransformationResult<U>> f) {
        Reloadable5<R1, R2, R3, R4, R5, U> reloadable = new Reloadable5<>(
                r1.currentValue(),
                r2.currentValue(),
                r3.currentValue(),
                r4.currentValue(),
                r5.currentValue(),
                f);
        r1.addWeakObserver(reloadable.observer1);
        r2.addWeakObserver(reloadable.observer2);
        r3.addWeakObserver(reloadable.observer3);
        r4.addWeakObserver(reloadable.observer4);
        r5.addWeakObserver(reloadable.observer5);
        return reloadable;
    }

}