/*******************************************************************************
 *     Copyright 2016-2017 the original author or authors.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *  
 *******************************************************************************/
package pro.parseq.vcf.fields;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import pro.parseq.vcf.fields.types.InfoFieldType;
import pro.parseq.vcf.types.Metadata;
import pro.parseq.vcf.utils.VcfGrammar;

/**
 * Represents VCF INFO meta-information field
 * 
 * @author Alexander Afanasyev <a href="mailto:aafanasyev@parseq.pro">aafanasyev@parseq.pro</a>
 */
public class Information extends Metadata {

	private String id;
	private String number;
	private InfoFieldType type;
	private String description;
	private String source;
	private String version;

	public Information(String line) {

		super(line);

		// Mandatory attributes
		Matcher idMatcher = VcfGrammar.infoIdPattern.matcher(line);
		Matcher numberMatcher = VcfGrammar.infoNumberPattern.matcher(line);
		Matcher typeMatcher = VcfGrammar.infoTypePattern.matcher(line);
		Matcher descriptionMatcher = VcfGrammar.infoDescriptionPattern.matcher(line);
		if (idMatcher.find() && numberMatcher.find() && typeMatcher.find() && descriptionMatcher.find()) {
			id = line.substring(idMatcher.start(), idMatcher.end());
			number = line.substring(numberMatcher.start(), numberMatcher.end());
			type = InfoFieldType.getEnum(line.substring(typeMatcher.start(), typeMatcher.end()));
			description = line.substring(descriptionMatcher.start(), descriptionMatcher.end());
		}

		// Recommended attributes
		Matcher sourceMatcher = VcfGrammar.infoSourcePattern.matcher(line);
		Matcher versionMatcher = VcfGrammar.infoVersionPattern.matcher(line);
		if (sourceMatcher.find() && versionMatcher.find()) {
			source = line.substring(sourceMatcher.start(), sourceMatcher.end());
			version = line.substring(versionMatcher.start(), versionMatcher.end());
		}
	}

	public String getId() {
		return id;
	}

	public void setId(String id) {
		this.id = id;
	}

	public String getNumber() {
		return number;
	}

	public void setNumber(String number) {
		this.number = number;
	}

	public InfoFieldType getType() {
		return type;
	}

	public void setType(InfoFieldType type) {
		this.type = type;
	}

	public String getDescription() {
		return description;
	}

	public void setDescription(String description) {
		this.description = description;
	}

	public String getSource() {
		return source;
	}

	public void setSource(String source) {
		this.source = source;
	}

	public String getVersion() {
		return version;
	}

	public void setVersion(String version) {
		this.version = version;
	}

	/**
	 * Allows to extract field values from data lines
	 * 
	 * @param value Data line INFO value specified for the field
	 * @return {@link List} of found values (see <a href="https://samtools.github.io/hts-specs/VCFv4.2.pdf">VCFv4.2 specification</a> for more details)
	 */
	public List<? extends Serializable> getValue(String value) {

		if (value == null) {
			return null;
		}

		String[] values = value.split(",");
		if (VcfGrammar.exactCountNumberValuePattern.matcher(number).matches()) {
			int numberValue = Integer.parseInt(number);
			if (numberValue != values.length) {
				// TODO: change this to throw proper exception
				return null;
			}
		}

		switch (type) {
		case CHARACTER:
		case STRING:
			return Arrays.asList(values);
		case FLOAT:
			List<Double> doubleValues = new ArrayList<>();
			for (int i = 0; i < values.length; ++i) {
				try {
					doubleValues.add(Double.parseDouble(values[i]));
				} catch (NumberFormatException e) {
					// TODO: change this to throw proper exception
					return null;
				}
			}
			return doubleValues;
		case INTEGER:
			List<Integer> intValues = new ArrayList<>();
			for (int i = 0; i < values.length; ++i) {
				try {
					intValues.add(Integer.parseInt(values[i]));
				} catch (NumberFormatException e) {
					// TODO: change this to throw proper exception
					return null;
				}
			}
			return intValues;
		default:
			return null;
		}
	}

	@Override
	protected Pattern getPattern() {
		return VcfGrammar.infoPattern;
	}

	public static final boolean isInformation(String line) {
		return VcfGrammar.infoPattern.matcher(line).matches();
	}
}
