/*******************************************************************************
 *     Copyright 2016-2017 the original author or authors.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *  
 *******************************************************************************/
package pro.parseq.vcf.types;

import java.util.ArrayList;
import java.util.List;
import java.util.regex.Pattern;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import pro.parseq.vcf.utils.VcfGrammar;

/**
 * Represents VCF header line
 * 
 * @author Alexander Afanasyev <a href="mailto:aafanasyev@parseq.pro">aafanasyev@parseq.pro</a>
 */
public class Header extends VcfLine {

	private static final Logger logger = LoggerFactory.getLogger(Header.class);

	// Found sample names if present
	private List<String> sampleNames = new ArrayList<>();

	public Header(String line) {

		super(line);
		if (!isValid()) {
			logger.error("Malformed header line: {}", line);
			return;
		}

		String[] columns = line.split(VcfGrammar.COLUMN_DELIMITER);
		if (columns.length > VcfGrammar.MANDATORY_COLUMNS_NUMBER) {
			/**
			 * Sample names go after FORMAT column if present (i.e. 9th, 10th etc. column 0-based indexes)
			 */
			for (int columnIdx = VcfGrammar.MANDATORY_COLUMNS_NUMBER + 1; columnIdx < columns.length; ++columnIdx) {
				sampleNames.add(columns[columnIdx]);
			}
		}
	}

	public List<String> getSampleNames() {
		return sampleNames;
	}

	@Override
	protected Pattern getPattern() {
		return VcfGrammar.headerPattern;
	}

	public static final boolean isHeader(String line) {
		return VcfGrammar.headerPattern.matcher(line).matches();
	}
}
