/*******************************************************************************
 *     Copyright 2016-2017 the original author or authors.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *  
 *******************************************************************************/
package pro.parseq.vcf.types;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import pro.parseq.vcf.fields.Filter;
import pro.parseq.vcf.fields.types.Genotype;

/**
 * Class to hold an information about single genetic variant from VCF data line
 * 
 * @author Alexander Afanasyev <a href="mailto:aafanasyev@parseq.pro">aafanasyev@parseq.pro</a>
 */
public class Variant {

	private String chrom;
	private long pos;
	private List<String> ids;
	private String ref;
	private String alt;
	private Double qual;
	private boolean filtered = false;
	private List<Filter> filters = new ArrayList<>();
	private Map<String, List<? extends Serializable>> info = new HashMap<>();
	// Keeps original allele number from VCF file
	private int allele;

	/**
	 * Maps sample name to format fields values described by FORMAT column
	 */
	private Map<String, Map<String, List<? extends Serializable>>> formats = new HashMap<>();

	/**
	 * Maps sample name to zygosity information
	 */
	private Map<String, Genotype> genotypes = new HashMap<>();

	public Variant() {}

	public String getChrom() {
		return chrom;
	}

	public void setChrom(String chrom) {
		this.chrom = chrom;
	}

	public long getPos() {
		return pos;
	}

	public void setPos(long pos) {
		this.pos = pos;
	}

	public List<String> getIds() {
		return ids;
	}

	public boolean addId(String id) {
		return ids.add(id);
	}

	public void setIds(List<String> ids) {
		this.ids = ids;
	}

	public String getRef() {
		return ref;
	}

	public void setRef(String ref) {
		this.ref = ref;
	}

	public String getAlt() {
		return alt;
	}

	public void setAlt(String alt) {
		this.alt = alt;
	}

	public Double getQual() {
		return qual;
	}

	public void setQual(Double qual) {
		this.qual = qual;
	}

	public boolean isFiltered() {
		return filtered;
	}

	public void setFiltered(boolean filtered) {
		this.filtered = filtered;
	}

	public List<Filter> getFilters() {
		return filters;
	}

	public boolean addFilter(Filter filter) {
		return filters.add(filter);
	}

	public void setFilters(List<Filter> filters) {
		this.filters = filters;
	}

	public Map<String, List<? extends Serializable>> getInfo() {
		return info;
	}

	public List<? extends Serializable> putInfo(String key,
			List<? extends Serializable> value) {
		return info.put(key, value);
	}

	public void setInfo(Map<String, List<? extends Serializable>> info) {
		this.info = info;
	}

	public Map<String, Map<String, List<? extends Serializable>>> getFormats() {
		return formats;
	}

	public Map<String, List<? extends Serializable>> putFormat(String sampleName,
			Map<String, List<? extends Serializable>> format) {
		return formats.put(sampleName, format);
	}

	public void setFormats(Map<String, Map<String,
			List<? extends Serializable>>> formats) {
		this.formats = formats;
	}

	public int getAllele() {
		return allele;
	}

	public void setAllele(int allele) {
		this.allele = allele;
	}

	public Map<String, Genotype> getGenotypes() {
		return genotypes;
	}

	public Genotype putGenotype(String sampleName, Genotype zygosity) {
		return genotypes.put(sampleName, zygosity);
	}

	public void setGenotypes(Map<String, Genotype> genotypes) {
		this.genotypes = genotypes;
	}

	public Genotype getZygosityFor(String sampleName) {

		Genotype genotype = genotypes.get(sampleName);
		if (genotype == null) {
			return Genotype.UNDEFINED;
		}

		return genotype;
	}

	public boolean isFoundVariantFor(String sampleName) {

		switch (getZygosityFor(sampleName)) {
		case HEMIZYGOUS_VARIANT:
		case HETEROZYGOUS_WITH_REFERENCE:
		case HETEROZYGOUS_WITH_VARIANT:
		case HOMOZYGOUS_VARIANT:
			return true;
		case HOMOZYGOUS_REFERENCE:
		case UNDEFINED:
		default:
			return false;
		}
	}
}
