/*******************************************************************************
 *     Copyright 2016-2017 the original author or authors.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *  
 *******************************************************************************/
package pro.parseq.vcf.types;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import pro.parseq.vcf.fields.Filter;
import pro.parseq.vcf.fields.Format;
import pro.parseq.vcf.fields.Information;

/**
 * Class to hold VCF file data
 * 
 * @author Alexander Afanasyev <a href="mailto:aafanasyev@parseq.pro">aafanasyev@parseq.pro</a>
 */
public class VcfFile {

	// List of every single line from input VCF file
	private List<VcfLine> lines = new ArrayList<>();

	// Only meta-information lines from input VCF file
	private List<Metadata> metaLines = new ArrayList<>();
	// Only data lines from input VCF file
	private List<VcfLine> dataLines = new ArrayList<>();

	// FILTERS fields described in meta-information section
	private Map<String, Filter> filters = new HashMap<>();
	// FORMATS fields described in meta-information section
	private Map<String, Format> formats = new HashMap<>();
	// INFO fields described in meta-information section
	private Map<String, Information> infos = new HashMap<>();
	// All other meta-information fields
	private Map<String, List<Metadata>> otherMetadata = new HashMap<>();

	// Sample names found in the header line
	private List<String> sampleNames = new ArrayList<>();

	// List of single genetic variants found in VCF fie
	private List<Variant> variants = new ArrayList<>();

	public VcfFile() {}

	public List<VcfLine> getLines() {
		return lines;
	}

	public boolean addLine(VcfLine line) {
		return lines.add(line);
	}

	public void setLines(List<VcfLine> lines) {
		this.lines = lines;
	}

	public List<Metadata> getMetaLines() {
		return metaLines;
	}

	public boolean addMetaLine(Metadata metaLine) {
		return addLine(metaLine) && metaLines.add(metaLine);
	}

	public void setMetaLines(List<Metadata> metaLines) {
		this.metaLines = metaLines;
	}

	public List<VcfLine> getDataLines() {
		return dataLines;
	}

	public boolean addDataLine(VcfLine dataLine) {
		return addLine(dataLine) && dataLines.add(dataLine);
	}

	public void setDataLines(List<VcfLine> dataLines) {
		this.dataLines = dataLines;
	}

	public Map<String, Filter> getFilters() {
		return filters;
	}

	public Filter putFilter(Filter filter) {
		addMetaLine(filter);
		return filters.put(filter.getId(), filter);
	}

	public void setFilters(Map<String, Filter> filters) {
		this.filters = filters;
	}

	public Map<String, Format> getFormats() {
		return formats;
	}

	public Format putFormat(Format format) {
		addMetaLine(format);
		return formats.put(format.getId(), format);
	}

	public void setFormats(Map<String, Format> formats) {
		this.formats = formats;
	}

	public Map<String, Information> getInfos() {
		return infos;
	}

	public Information putInfo(Information info) {
		addMetaLine(info);
		return infos.put(info.getId(), info);
	}

	public void setInfos(Map<String, Information> infos) {
		this.infos = infos;
	}

	public Map<String, List<Metadata>> getOtherMetadata() {
		return otherMetadata;
	}

	public List<Metadata> putOtherMetadata(Metadata other) {

		addMetaLine(other);

		List<Metadata> otherValue = otherMetadata.get(other.getKey());
		if (otherValue == null) {
			otherValue = new ArrayList<>();
		}
		otherValue.add(other);

		return otherMetadata.put(other.getKey(), otherValue);
	}

	public void setOtherMetadata(Map<String, List<Metadata>> otherMetadata) {
		this.otherMetadata = otherMetadata;
	}

	public List<String> getSampleNames() {
		return sampleNames;
	}

	public boolean addSampleName(String sampleName) {
		return sampleNames.add(sampleName);
	}

	public void setSampleNames(List<String> sampleNames) {
		this.sampleNames = sampleNames;
	}

	public List<Variant> getVariants() {
		return variants;
	}

	public boolean addDataLineVariants(DataLine dataLine) {
		addDataLine(dataLine);
		return variants.addAll(dataLine.getVariants());
	}

	public void setVariants(List<Variant> variants) {
		this.variants = variants;
	}

	public int size() {
		return lines.size();
	}
}
