/*******************************************************************************
 *     Copyright 2016-2017 the original author or authors.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *  
 *******************************************************************************/
package pro.parseq.vcf.utils;

import java.util.ArrayList;
import java.util.List;

import pro.parseq.vcf.exceptions.InvalidVcfFileException;
import pro.parseq.vcf.validators.LineValidator;

/**
 * Abstract class representing an entity that can be validated
 * 
 * @author Alexander Afanasyev <a href="mailto:aafanasyev@parseq.pro">aafanasyev@parseq.pro</a>
 */
public abstract class Validatable {

	abstract public List<LineValidator> getValidators();

	public List<LineValidator> validate(String line)
			throws InvalidVcfFileException {

		List<LineValidator> failedValidators = new ArrayList<>();
		for (LineValidator validator: getValidators()) {
			if (!validator.validate(line)) {
				failedValidators.add(validator);
			}
		}

		return failedValidators;
	}

	protected void raiseValidationException(List<LineValidator> failedValidators)
			throws InvalidVcfFileException {

		StringBuilder sb = new StringBuilder();
		for (LineValidator validator: failedValidators) {
			sb.append(validator.getFailureMessage()).append(";");
		}

		throw new InvalidVcfFileException(sb.toString());
	}

	protected void raiseValidationException(int line, List<LineValidator> failedValidators)
			throws InvalidVcfFileException {

		StringBuilder sb = new StringBuilder()
				.append("Line ")
				.append(line)
				.append(": ");
		for (LineValidator validator: failedValidators) {
			sb.append(validator.getFailureMessage()).append(";");
		}

		throw new InvalidVcfFileException(line, sb.toString());
	}
}
