/*
 * Decompiled with CFR 0.152.
 */
package redis.clients.jedis;

import java.io.Closeable;
import java.io.IOException;
import java.net.Socket;
import java.net.SocketException;
import java.nio.ByteBuffer;
import java.nio.CharBuffer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.function.Supplier;
import redis.clients.jedis.BuilderFactory;
import redis.clients.jedis.ClientSetInfoConfig;
import redis.clients.jedis.CommandArguments;
import redis.clients.jedis.CommandObject;
import redis.clients.jedis.ConnectionPool;
import redis.clients.jedis.DefaultJedisSocketFactory;
import redis.clients.jedis.HostAndPort;
import redis.clients.jedis.JedisClientConfig;
import redis.clients.jedis.JedisSocketFactory;
import redis.clients.jedis.Protocol;
import redis.clients.jedis.RedisCredentials;
import redis.clients.jedis.RedisCredentialsProvider;
import redis.clients.jedis.args.ClientAttributeOption;
import redis.clients.jedis.args.Rawable;
import redis.clients.jedis.commands.ProtocolCommand;
import redis.clients.jedis.exceptions.JedisConnectionException;
import redis.clients.jedis.exceptions.JedisDataException;
import redis.clients.jedis.exceptions.JedisException;
import redis.clients.jedis.util.IOUtils;
import redis.clients.jedis.util.JedisMetaInfo;
import redis.clients.jedis.util.RedisInputStream;
import redis.clients.jedis.util.RedisOutputStream;
import redis.clients.jedis.util.SafeEncoder;

public class Connection
implements Closeable {
    private ConnectionPool memberOf;
    private final JedisSocketFactory socketFactory;
    private Socket socket;
    private RedisOutputStream outputStream;
    private RedisInputStream inputStream;
    private int soTimeout = 0;
    private int infiniteSoTimeout = 0;
    private boolean broken = false;

    public Connection() {
        this("127.0.0.1", 6379);
    }

    public Connection(String host, int port) {
        this(new HostAndPort(host, port));
    }

    public Connection(HostAndPort hostAndPort) {
        this(new DefaultJedisSocketFactory(hostAndPort));
    }

    public Connection(HostAndPort hostAndPort, JedisClientConfig clientConfig) {
        this(new DefaultJedisSocketFactory(hostAndPort, clientConfig));
        this.infiniteSoTimeout = clientConfig.getBlockingSocketTimeoutMillis();
        this.initializeFromClientConfig(clientConfig);
    }

    public Connection(JedisSocketFactory socketFactory, JedisClientConfig clientConfig) {
        this.socketFactory = socketFactory;
        this.soTimeout = clientConfig.getSocketTimeoutMillis();
        this.infiniteSoTimeout = clientConfig.getBlockingSocketTimeoutMillis();
        this.initializeFromClientConfig(clientConfig);
    }

    public Connection(JedisSocketFactory socketFactory) {
        this.socketFactory = socketFactory;
    }

    public String toString() {
        return "Connection{" + this.socketFactory + "}";
    }

    public final void setHandlingPool(ConnectionPool pool) {
        this.memberOf = pool;
    }

    final HostAndPort getHostAndPort() {
        return ((DefaultJedisSocketFactory)this.socketFactory).getHostAndPort();
    }

    public int getSoTimeout() {
        return this.soTimeout;
    }

    public void setSoTimeout(int soTimeout) {
        this.soTimeout = soTimeout;
        if (this.socket != null) {
            try {
                this.socket.setSoTimeout(soTimeout);
            }
            catch (SocketException ex) {
                this.broken = true;
                throw new JedisConnectionException(ex);
            }
        }
    }

    public void setTimeoutInfinite() {
        try {
            if (!this.isConnected()) {
                this.connect();
            }
            this.socket.setSoTimeout(this.infiniteSoTimeout);
        }
        catch (SocketException ex) {
            this.broken = true;
            throw new JedisConnectionException(ex);
        }
    }

    public void rollbackTimeout() {
        try {
            this.socket.setSoTimeout(this.soTimeout);
        }
        catch (SocketException ex) {
            this.broken = true;
            throw new JedisConnectionException(ex);
        }
    }

    public Object executeCommand(ProtocolCommand cmd) {
        return this.executeCommand(new CommandArguments(cmd));
    }

    public Object executeCommand(CommandArguments args) {
        this.sendCommand(args);
        return this.getOne();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public <T> T executeCommand(CommandObject<T> commandObject) {
        CommandArguments args = commandObject.getArguments();
        this.sendCommand(args);
        if (!args.isBlocking()) {
            return commandObject.getBuilder().build(this.getOne());
        }
        try {
            this.setTimeoutInfinite();
            T t = commandObject.getBuilder().build(this.getOne());
            return t;
        }
        finally {
            this.rollbackTimeout();
        }
    }

    public void sendCommand(ProtocolCommand cmd) {
        this.sendCommand(new CommandArguments(cmd));
    }

    public void sendCommand(ProtocolCommand cmd, Rawable keyword) {
        this.sendCommand(new CommandArguments(cmd).add(keyword));
    }

    public void sendCommand(ProtocolCommand cmd, String ... args) {
        this.sendCommand(new CommandArguments(cmd).addObjects(args));
    }

    public void sendCommand(ProtocolCommand cmd, byte[] ... args) {
        this.sendCommand(new CommandArguments(cmd).addObjects((Object[])args));
    }

    public void sendCommand(CommandArguments args) {
        try {
            this.connect();
            Protocol.sendCommand(this.outputStream, args);
        }
        catch (JedisConnectionException ex) {
            try {
                String errorMessage = Protocol.readErrorLineIfPossible(this.inputStream);
                if (errorMessage != null && errorMessage.length() > 0) {
                    ex = new JedisConnectionException(errorMessage, ex.getCause());
                }
            }
            catch (Exception exception) {
                // empty catch block
            }
            this.broken = true;
            throw ex;
        }
    }

    public void connect() throws JedisConnectionException {
        if (!this.isConnected()) {
            try {
                this.socket = this.socketFactory.createSocket();
                this.soTimeout = this.socket.getSoTimeout();
                this.outputStream = new RedisOutputStream(this.socket.getOutputStream());
                this.inputStream = new RedisInputStream(this.socket.getInputStream());
                this.broken = false;
            }
            catch (JedisConnectionException jce) {
                this.setBroken();
                throw jce;
            }
            catch (IOException ioe) {
                this.setBroken();
                throw new JedisConnectionException("Failed to create input/output stream", ioe);
            }
            finally {
                if (this.broken) {
                    IOUtils.closeQuietly(this.socket);
                }
            }
        }
    }

    @Override
    public void close() {
        if (this.memberOf != null) {
            ConnectionPool pool = this.memberOf;
            this.memberOf = null;
            if (this.isBroken()) {
                pool.returnBrokenResource(this);
            } else {
                pool.returnResource(this);
            }
        } else {
            this.disconnect();
        }
    }

    public void disconnect() {
        if (this.isConnected()) {
            try {
                this.outputStream.flush();
                this.socket.close();
            }
            catch (IOException ex) {
                throw new JedisConnectionException(ex);
            }
            finally {
                IOUtils.closeQuietly(this.socket);
                this.setBroken();
            }
        }
    }

    public boolean isConnected() {
        return this.socket != null && this.socket.isBound() && !this.socket.isClosed() && this.socket.isConnected() && !this.socket.isInputShutdown() && !this.socket.isOutputShutdown();
    }

    public boolean isBroken() {
        return this.broken;
    }

    public void setBroken() {
        this.broken = true;
    }

    public String getStatusCodeReply() {
        this.flush();
        byte[] resp = (byte[])this.readProtocolWithCheckingBroken();
        if (null == resp) {
            return null;
        }
        return SafeEncoder.encode(resp);
    }

    public String getBulkReply() {
        byte[] result = this.getBinaryBulkReply();
        if (null != result) {
            return SafeEncoder.encode(result);
        }
        return null;
    }

    public byte[] getBinaryBulkReply() {
        this.flush();
        return (byte[])this.readProtocolWithCheckingBroken();
    }

    public Long getIntegerReply() {
        this.flush();
        return (Long)this.readProtocolWithCheckingBroken();
    }

    public List<String> getMultiBulkReply() {
        return BuilderFactory.STRING_LIST.build(this.getBinaryMultiBulkReply());
    }

    public List<byte[]> getBinaryMultiBulkReply() {
        this.flush();
        return (List)this.readProtocolWithCheckingBroken();
    }

    public List<Object> getUnflushedObjectMultiBulkReply() {
        return (List)this.readProtocolWithCheckingBroken();
    }

    public List<Object> getObjectMultiBulkReply() {
        this.flush();
        return this.getUnflushedObjectMultiBulkReply();
    }

    public List<Long> getIntegerMultiBulkReply() {
        this.flush();
        return (List)this.readProtocolWithCheckingBroken();
    }

    public Object getOne() {
        this.flush();
        return this.readProtocolWithCheckingBroken();
    }

    protected void flush() {
        try {
            this.outputStream.flush();
        }
        catch (IOException ex) {
            this.broken = true;
            throw new JedisConnectionException(ex);
        }
    }

    protected Object readProtocolWithCheckingBroken() {
        if (this.broken) {
            throw new JedisConnectionException("Attempting to read from a broken connection");
        }
        try {
            return Protocol.read(this.inputStream);
        }
        catch (JedisConnectionException exc) {
            this.broken = true;
            throw exc;
        }
    }

    public List<Object> getMany(int count) {
        this.flush();
        ArrayList<Object> responses = new ArrayList<Object>(count);
        for (int i = 0; i < count; ++i) {
            try {
                responses.add(this.readProtocolWithCheckingBroken());
                continue;
            }
            catch (JedisDataException e) {
                responses.add(e);
            }
        }
        return responses;
    }

    private static boolean validateClientInfo(String info) {
        for (int i = 0; i < info.length(); ++i) {
            char c = info.charAt(i);
            if (c >= '!' && c <= '~') continue;
            throw new JedisException("client info cannot contain spaces, newlines or special characters.");
        }
        return true;
    }

    private void initializeFromClientConfig(JedisClientConfig config) {
        try {
            ClientSetInfoConfig setInfoConfig;
            this.connect();
            Supplier<RedisCredentials> credentialsProvider = config.getCredentialsProvider();
            if (credentialsProvider instanceof RedisCredentialsProvider) {
                try {
                    ((RedisCredentialsProvider)credentialsProvider).prepare();
                    this.auth(credentialsProvider);
                }
                finally {
                    ((RedisCredentialsProvider)credentialsProvider).cleanUp();
                }
            } else {
                this.auth(credentialsProvider);
            }
            int dbIndex = config.getDatabase();
            if (dbIndex > 0) {
                this.select(dbIndex);
            }
            ArrayList<CommandArguments> fireAndForgetMsg = new ArrayList<CommandArguments>();
            String clientName = config.getClientName();
            if (clientName != null && Connection.validateClientInfo(clientName)) {
                fireAndForgetMsg.add(new CommandArguments(Protocol.Command.CLIENT).add(Protocol.Keyword.SETNAME).add(clientName));
            }
            if ((setInfoConfig = config.getClientSetInfoConfig()) == null) {
                setInfoConfig = ClientSetInfoConfig.DEFAULT;
            }
            if (!setInfoConfig.isDisabled()) {
                String libVersion;
                String libName = JedisMetaInfo.getArtifactId();
                if (libName != null && Connection.validateClientInfo(libName)) {
                    String libNameSuffix = setInfoConfig.getLibNameSuffix();
                    if (libNameSuffix != null) {
                        libName = libName + '(' + libNameSuffix + ')';
                    }
                    fireAndForgetMsg.add(new CommandArguments(Protocol.Command.CLIENT).add(Protocol.Keyword.SETINFO).add(ClientAttributeOption.LIB_NAME.getRaw()).add(libName));
                }
                if ((libVersion = JedisMetaInfo.getVersion()) != null && Connection.validateClientInfo(libVersion)) {
                    fireAndForgetMsg.add(new CommandArguments(Protocol.Command.CLIENT).add(Protocol.Keyword.SETINFO).add(ClientAttributeOption.LIB_VER.getRaw()).add(libVersion));
                }
            }
            for (CommandArguments arg : fireAndForgetMsg) {
                this.sendCommand(arg);
            }
            this.getMany(fireAndForgetMsg.size());
        }
        catch (JedisException je) {
            try {
                this.disconnect();
            }
            catch (Exception exception) {
                // empty catch block
            }
            throw je;
        }
    }

    private void auth(Supplier<RedisCredentials> credentialsProvider) {
        RedisCredentials credentials = credentialsProvider.get();
        if (credentials == null || credentials.getPassword() == null) {
            return;
        }
        ByteBuffer passBuf = Protocol.CHARSET.encode(CharBuffer.wrap(credentials.getPassword()));
        byte[] rawPass = Arrays.copyOfRange(passBuf.array(), passBuf.position(), passBuf.limit());
        Arrays.fill(passBuf.array(), (byte)0);
        if (credentials.getUser() != null) {
            this.sendCommand((ProtocolCommand)Protocol.Command.AUTH, SafeEncoder.encode(credentials.getUser()), rawPass);
        } else {
            this.sendCommand((ProtocolCommand)Protocol.Command.AUTH, new byte[][]{rawPass});
        }
        Arrays.fill(rawPass, (byte)0);
        this.getStatusCodeReply();
    }

    public String select(int index) {
        this.sendCommand((ProtocolCommand)Protocol.Command.SELECT, new byte[][]{Protocol.toByteArray(index)});
        return this.getStatusCodeReply();
    }

    @Deprecated
    public String quit() {
        this.sendCommand(Protocol.Command.QUIT);
        String quitReturn = this.getStatusCodeReply();
        this.disconnect();
        this.setBroken();
        return quitReturn;
    }

    public boolean ping() {
        this.sendCommand(Protocol.Command.PING);
        String status = this.getStatusCodeReply();
        if (!"PONG".equals(status)) {
            throw new JedisException(status);
        }
        return true;
    }
}

