/*
 * Decompiled with CFR 0.152.
 */
package tdl.record.screen.video;

import io.humble.video.Codec;
import io.humble.video.Coder;
import io.humble.video.ContainerFormat;
import io.humble.video.Encoder;
import io.humble.video.MediaPacket;
import io.humble.video.MediaPicture;
import io.humble.video.MediaSampled;
import io.humble.video.Muxer;
import io.humble.video.MuxerFormat;
import io.humble.video.PixelFormat;
import io.humble.video.Rational;
import io.humble.video.awt.MediaPictureConverter;
import io.humble.video.awt.MediaPictureConverterFactory;
import java.awt.image.BufferedImage;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;
import java.time.Duration;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import tdl.record.screen.image.input.EnsureEvenHeightAndWidth;
import tdl.record.screen.image.input.ImageInput;
import tdl.record.screen.image.input.InputImageGenerationException;
import tdl.record.screen.metrics.VideoRecordingListener;
import tdl.record.screen.metrics.VideoRecordingMetricsCollector;
import tdl.record.screen.time.SystemTimeSource;
import tdl.record.screen.time.TimeSource;
import tdl.record.screen.video.VideoRecorderException;

public class VideoRecorder {
    private static final Logger log = LoggerFactory.getLogger(VideoRecorder.class);
    private final ImageInput imageInput;
    private final TimeSource timeSource;
    private final VideoRecordingListener videoRecordingListener;
    private long fragmentationMicros;
    private Muxer muxer;
    private Encoder encoder;
    private Rational videoFrameRate;
    private Rational inputFrameRate;
    private MediaPacket packet;
    private String destinationFilename;
    private final AtomicBoolean shouldStopJob = new AtomicBoolean(false);

    public static void runSanityCheck() {
        Rational.make();
    }

    private VideoRecorder(ImageInput imageInput, TimeSource timeSource, VideoRecordingListener videoRecordingListener, long bFragmentationMicros) {
        this.imageInput = new EnsureEvenHeightAndWidth(imageInput);
        this.timeSource = timeSource;
        this.videoRecordingListener = videoRecordingListener;
        this.fragmentationMicros = bFragmentationMicros;
    }

    public void open(String filename, int snapsPerSecond, int timeSpeedUpFactor) throws VideoRecorderException {
        this.destinationFilename = filename;
        this.inputFrameRate = Rational.make((int)1, (int)snapsPerSecond);
        this.videoFrameRate = Rational.make((int)1, (int)(timeSpeedUpFactor * snapsPerSecond));
        log.info("Open the input stream");
        try {
            this.imageInput.open();
        }
        catch (InputImageGenerationException e) {
            throw new VideoRecorderException("Could not open input source", e);
        }
        this.muxer = VideoRecorder.createMP4MuxerWithFragmentation(filename, this.fragmentationMicros);
        this.encoder = Encoder.make((Codec)VideoRecorder.getMP4Codec());
        this.encoder.setWidth(this.imageInput.getWidth());
        this.encoder.setHeight(this.imageInput.getHeight());
        this.encoder.setPixelFormat(PixelFormat.Type.PIX_FMT_YUV420P);
        this.encoder.setTimeBase(this.videoFrameRate);
        if (this.muxer.getFormat().getFlag(ContainerFormat.Flag.GLOBAL_HEADER)) {
            this.encoder.setFlag(Coder.Flag.FLAG_GLOBAL_HEADER, true);
        }
        this.encoder.open(null, null);
        this.muxer.addNewStream((Coder)this.encoder);
        Path lockFilePath = Paths.get(filename + ".lock", new String[0]);
        try {
            Files.write(lockFilePath, new byte[0], StandardOpenOption.CREATE);
            this.muxer.open(null, null);
        }
        catch (IOException | InterruptedException e) {
            throw new VideoRecorderException("Failed to open destination", e);
        }
        this.packet = MediaPacket.make();
    }

    private static Muxer createMP4MuxerWithFragmentation(String destinationFilename, long fragmentationMicros) {
        Muxer muxer = Muxer.make((String)destinationFilename, null, (String)"mp4");
        muxer.setProperty("movflags", "frag_keyframe");
        muxer.setProperty("frag_duration", fragmentationMicros);
        return muxer;
    }

    private static Codec getMP4Codec() {
        MuxerFormat format = MuxerFormat.guessFormat((String)"mp4", null, null);
        return Codec.findEncodingCodec((Codec.ID)format.getDefaultVideoCodecId());
    }

    public void start(Duration duration) throws VideoRecorderException {
        try {
            this.videoRecordingListener.notifyRecordingStart(this.destinationFilename, this.inputFrameRate, this.videoFrameRate);
            this.doRecord(duration);
        }
        catch (RuntimeException e) {
            throw new VideoRecorderException("Fatal exception while recording", e);
        }
        finally {
            this.flush();
            this.videoRecordingListener.notifyRecordingEnd();
        }
    }

    private void doRecord(Duration duration) throws VideoRecorderException {
        BufferedImage sampleImage;
        MediaPicture picture = MediaPicture.make((int)this.encoder.getWidth(), (int)this.encoder.getHeight(), (PixelFormat.Type)this.encoder.getPixelFormat());
        picture.setTimeBase(this.videoFrameRate);
        try {
            sampleImage = this.imageInput.getSampleImage();
        }
        catch (InputImageGenerationException e) {
            throw new VideoRecorderException("Could not get sample image from input source", e);
        }
        MediaPictureConverter converter = MediaPictureConverterFactory.createConverter((BufferedImage)sampleImage, (MediaPicture)picture);
        double totalNumberOfFrames = this.inputFrameRate.rescale(duration.getSeconds(), Rational.make((double)1.0));
        double timeBetweenFramesMillis = this.inputFrameRate.getValue() * 1000.0;
        long frameIndex = 0L;
        while ((double)frameIndex < totalNumberOfFrames) {
            BufferedImage screen;
            long timestampBeforeProcessing = this.timeSource.currentTimeNano();
            this.videoRecordingListener.notifyFrameRenderingStart(timestampBeforeProcessing, TimeUnit.NANOSECONDS, frameIndex);
            try {
                screen = this.imageInput.readImage();
            }
            catch (InputImageGenerationException e) {
                log.error("Failed to acquire image", (Throwable)e);
                break;
            }
            converter.toPicture(picture, screen, frameIndex);
            do {
                this.encoder.encode(this.packet, (MediaSampled)picture);
                if (!this.packet.isComplete()) continue;
                this.muxer.write(this.packet, false);
            } while (this.packet.isComplete());
            this.videoRecordingListener.notifyFrameRenderingEnd(this.timeSource.currentTimeNano(), TimeUnit.NANOSECONDS, frameIndex);
            long nextTimestamp = timestampBeforeProcessing + TimeUnit.MILLISECONDS.toNanos((long)timeBetweenFramesMillis);
            try {
                this.timeSource.wakeUpAt(nextTimestamp, TimeUnit.NANOSECONDS);
            }
            catch (InterruptedException e) {
                log.debug("Interrupted while sleeping", (Throwable)e);
            }
            if (this.shouldStopJob.get()) break;
            ++frameIndex;
        }
    }

    public void stop() {
        if (!this.shouldStopJob.get()) {
            log.info("Stopping recording");
            this.shouldStopJob.set(true);
        } else {
            log.info("Recording already stopping");
        }
    }

    public void close() {
        try {
            log.info("Closing the video stream");
            this.imageInput.close();
            this.muxer.close();
            Path lockFilePath = Paths.get(this.muxer.getURL() + ".lock", new String[0]);
            Files.delete(lockFilePath);
        }
        catch (IOException e) {
            throw new RuntimeException("Can't delete *.lock file.", e);
        }
    }

    private void flush() {
        log.info("Flushing remaining frames");
        do {
            this.encoder.encode(this.packet, null);
            if (!this.packet.isComplete()) continue;
            this.muxer.write(this.packet, false);
        } while (this.packet.isComplete());
    }

    public static class Builder {
        private final ImageInput bImageInput;
        private TimeSource bTimeSource;
        private VideoRecordingListener bVideoRecordingListener;
        private long bFragmentationMicros;

        public Builder(ImageInput imageInput) {
            this.bImageInput = imageInput;
            this.bTimeSource = new SystemTimeSource();
            this.bVideoRecordingListener = new VideoRecordingMetricsCollector();
            this.bFragmentationMicros = TimeUnit.MINUTES.toMicros(5L);
        }

        public Builder withTimeSource(TimeSource timeSource) {
            this.bTimeSource = timeSource;
            return this;
        }

        public Builder withRecordingListener(VideoRecordingListener videoRecordingListener) {
            this.bVideoRecordingListener = videoRecordingListener;
            return this;
        }

        public Builder withFragmentation(int fragmentation, TimeUnit timeUnit) {
            this.bFragmentationMicros = timeUnit.toMicros(fragmentation);
            return this;
        }

        public VideoRecorder build() {
            return new VideoRecorder(this.bImageInput, this.bTimeSource, this.bVideoRecordingListener, this.bFragmentationMicros);
        }
    }
}

